package de.julielab.gene.candidateretrieval;

import com.google.common.cache.LoadingCache;
import de.julielab.gene.candidateretrieval.scoring.LuceneScorer;
import de.julielab.gene.candidateretrieval.scoring.MaxEntScorer;
import de.julielab.geneexpbase.GeneExpRuntimeException;
import de.julielab.geneexpbase.TermNormalizer;
import de.julielab.geneexpbase.candidateretrieval.CandidateCacheKey;
import de.julielab.geneexpbase.candidateretrieval.GeneCandidateRetrievalException;
import de.julielab.geneexpbase.candidateretrieval.QueryGenerator;
import de.julielab.geneexpbase.candidateretrieval.SynHit;
import de.julielab.geneexpbase.configuration.Parameters;
import de.julielab.geneexpbase.genemodel.GeneMention;
import de.julielab.geneexpbase.genemodel.GeneName;
import de.julielab.geneexpbase.scoring.Scorer;
import de.julielab.geneexpbase.scoring.*;
import de.julielab.geneexpbase.services.CacheService;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.lucene.analysis.custom.CustomAnalyzer;
import org.apache.lucene.analysis.ngram.NGramFilterFactory;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.*;
import org.apache.lucene.search.*;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.similarities.ClassicSimilarity;
import org.apache.lucene.search.spell.SpellChecker;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.util.QueryBuilder;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static de.julielab.gene.candidateretrieval.SynonymIndexFieldNames.*;
import static de.julielab.geneexpbase.scoring.Scorer.PERFECT_SCORE;
import static org.apache.lucene.search.BooleanClause.Occur.*;

public class NameCentricRetrieval implements CandidateRetrieval {

    public static final QueryGenerator CONJUNCTION = new BooleanQueryGenerator(MUST, 0);
    public static final QueryGenerator DISJUNCTION = new BooleanQueryGenerator(SHOULD, -1);
    public static final QueryGenerator DISJUNCTION_MINUS_1 = new BooleanQueryGenerator(SHOULD, 1);
    public static final QueryGenerator DISJUNCTION_MINUS_2 = new BooleanQueryGenerator(SHOULD, 2);
    public static final QueryGenerator NGRAM_2_3 = new NGramQueryGenerator(2, 3);
    public static final String NAME_PRIO_DELIMITER = "__";

    public static final String LOGGER_NAME_CANDIDATES = "de.julielab.jules.ae.genemapper.candidates";
    public static final int SIMPLE_SCORER = 0;
    public static final int TOKEN_JAROWINKLER_SCORER = 1;
    public static final int MAXENT_SCORER = 2;
    public static final int JAROWINKLER_SCORER = 3;
    public static final int LEVENSHTEIN_SCORER = 4;
    public static final int TFIDF = 5;
    public static final int LUCENE_SCORER = 10;
    /**
     * default model for MaxEntScorer
     */
    public static final String MAXENT_SCORER_MODEL = "/genemapper_jules_mallet.mod";
    public static final Logger candidateLog = LoggerFactory.getLogger(LOGGER_NAME_CANDIDATES);
    /**
     * the maximal number of hits lucene returns for a query
     */
    public static final int LUCENE_MAX_HITS = CandidateCacheKey.DEFAULT_MAX_CANDIDATES;
    private static final Logger log = LoggerFactory.getLogger(LuceneCandidateRetrieval.class);
    /**
     * This static map is supposed to make candidate caches available for all
     * instances of this class across the JVM. This is important since we often
     * use multiple gene-mapper instances in the same pipeline. It can save a
     * lot of time and also space.
     */
    private static final ConcurrentHashMap<String, LoadingCache<CandidateCacheKey, List<SynHit>>> caches = new ConcurrentHashMap<>();
    private static final AtomicLong totalCacheGettime = new AtomicLong();
    private static final AtomicLong totalCachePuttime = new AtomicLong();
    private static final AtomicLong totalLuceneQueryTime = new AtomicLong();
    private static final Map<Thread, IndexSearcher> mentionIndexSearcher = new ConcurrentHashMap<>();
    private static final Map<Thread, IndexSearcher> geneRecordIndexSearcher = new ConcurrentHashMap<>();
    private static ExecutorService executorService;
    private final Map<String, Float> globalFieldWeights;
    private final Scorer exactScorer;
    private final Scorer approxScorer;
    private final javax.cache.Cache<CandidateCacheKey, List> candidateCache;
    private final Configuration configuration;
    private final IndexReader geneRecordIndexReader;
    private final IndexReader nameCentricIndexReader;
    private final Boolean useLuceneCandidateCache;
    private CustomAnalyzer ngramAnalyzer;
    // the model to be loaded for MaxEnt scorer
    // (can be specified in properties file)
    private String maxEntModel = MAXENT_SCORER_MODEL;
    private TermNormalizer normalizer;
    private SpellChecker spellingChecker;

    @Inject
    public NameCentricRetrieval(Configuration config, CacheService cacheService) throws GeneCandidateRetrievalException {
        configuration = config;
        this.useLuceneCandidateCache = config.getBoolean(Configuration.USE_LUCENE_CANDIDATES_CACHE).orElse(false);
        // lucene mention index
        String mentionIndex = config.getProperty(Configuration.NAME_CENTRIC_INDEX);
        if (mentionIndex == null) {
            throw new GeneCandidateRetrievalException("mention index not specified in configuration file (critical).");
        }
        // lucene gene record index
        String geneRecordIndex = config.getProperty(Configuration.GENE_RECORD_INDEX);
        if (geneRecordIndex == null) {
            throw new GeneCandidateRetrievalException("geneRecordIndex index not specified in configuration file (critical).");
        }

        try {

            nameCentricIndexReader = DirectoryReader.open(FSDirectory.open(Paths.get(mentionIndex)));
            int luceneConcurrencyLevel = Integer.parseInt((String) config.getOrDefault(de.julielab.geneexpbase.configuration.Configuration.CONCURRENCY_LEVEL, "1"));
            synchronized (LuceneCandidateRetrieval.class) {
                if (executorService == null) {
                    executorService = Executors.newFixedThreadPool(luceneConcurrencyLevel, new ThreadFactory() {
                        private final ThreadFactory tf = Executors.defaultThreadFactory();

                        @Override
                        public Thread newThread(@NotNull Runnable r) {
                            Thread t = tf.newThread(r);
                            t.setName("pool-lucene-candidate-retrieval");
                            return t;
                        }
                    });
                }
            }

            log.debug("mention index loaded.");

            geneRecordIndexReader = DirectoryReader.open(FSDirectory.open(Paths.get(geneRecordIndex)));
            log.info("Gene record index has {} segments", ((StandardDirectoryReader) geneRecordIndexReader).getSegmentInfos().size());


            String spellingIndexPath = config.getProperty(Configuration.SPELLING_INDEX);
            if (spellingIndexPath != null) {
                File spellingIndex = new File(spellingIndexPath);
                if (spellingIndex.exists())
                    spellingChecker = new SpellChecker(FSDirectory.open(spellingIndex.toPath()));
            }
            if (spellingChecker == null)
                log.warn(
                        "Spelling index was not given or file does not exist. No spelling correction can be done. Specified spelling index: {}",
                        spellingIndexPath);

            // scorer types
            String scorerType = config.getProperty(Configuration.EXACT_SCORER_TYPE);
            if (scorerType == null) {
                log.debug("No configuration value given for " + Configuration.EXACT_SCORER_TYPE);
                exactScorer = setScorerType(LUCENE_SCORER);
            }
            else {
                exactScorer = setScorerType(Integer.valueOf(scorerType));
            }

            scorerType = config.getProperty(Configuration.APPROX_SCORER_TYPE);
            if (scorerType == null) {
                log.debug("No configuration value given for " + Configuration.APPROX_SCORER_TYPE);
                approxScorer = setScorerType(LUCENE_SCORER);
            }
            else {
                approxScorer = setScorerType(Integer.valueOf(scorerType));
            }

            // maxent model
            String maxEntModel = config.getProperty("maxent_model");
            if (maxEntModel != null) {
                this.maxEntModel = maxEntModel;
            }

            this.normalizer = new TermNormalizer();
        } catch (IOException e) {
            throw new GeneCandidateRetrievalException(e);
        }

        log.info("Mention index: " + mentionIndex);
        log.info("Exact scorer: " + exactScorer);
        log.info("Approx scorer: " + approxScorer);

        candidateCache = cacheService.getCacheManager().getCache("candidates-cache-name-centric");

        try {
            Map<String, String> ngramFilterSettings = new HashMap<>();
            ngramFilterSettings.put("minGramSize", "2");
            ngramFilterSettings.put("maxGramSize", "3");
            ngramAnalyzer = CustomAnalyzer.builder()
                    .withTokenizer("whitespace")
                    .addTokenFilter(NGramFilterFactory.class, ngramFilterSettings)
                    .build();
        } catch (IOException e) {
            e.printStackTrace();
        }

        globalFieldWeights = new HashMap<>();
        config.getDouble(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, Configuration.PARAM_DISMAX_TIE_BREAKER)).ifPresent(d -> globalFieldWeights.put(Configuration.PARAM_DISMAX_TIE_BREAKER, (float) d));
        for (String field : GeneRecordQueryGenerator.ALL_FIELDS) {
            OptionalDouble fieldWeight = config.getDouble(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, field));
            fieldWeight.ifPresent(d -> globalFieldWeights.put(field, (float) d));
        }
        for (String field : GeneRecordQueryGenerator.SYNONYM_FIELDS) {
            OptionalDouble fieldWeight = config.getDouble(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, field + "_exact"));
            fieldWeight.ifPresent(d -> globalFieldWeights.put(field, (float) d));
        }
    }

    public static AtomicLong getTotalCacheGettime() {
        return totalCacheGettime;
    }

    public static AtomicLong getTotalCachePuttime() {
        return totalCachePuttime;
    }

    public static AtomicLong getTotalLuceneQueryTime() {
        return totalLuceneQueryTime;
    }

    public static void shutdownExecutor() {
        executorService.shutdown();
    }

    public TermNormalizer getNormalizer() {
        return normalizer;
    }

    public void setNormalizer(TermNormalizer normalizer) {
        this.normalizer = normalizer;
    }

    // public ArrayList<SynHit> getCandidates(CandidateCacheKey key) throws
    // GeneCandidateRetrievalException {
    //
    // try {
    // TopDocs foundDocs = getCandidatesFromIndex(key);
    // // 2. assign score
    // ArrayList<SynHit> scoredHits = new ArrayList<SynHit>();
    // try {
    // scoredHits = scoreHits(foundDocs, key.geneName);
    // } catch (IOException e) {
    // e.printStackTrace();
    // log.error("getCandidates() - error scoring hits: " + e.getMessage());
    // }
    // // 3. combine single hits to candidate clusters
    // ArrayList<SynHit> hits = combineHits(scoredHits);
    // // 4. sort by SynHit's score (lucene score)
    // Collections.sort(hits);
    //
    // // -------- TODO testing
    // // if (!ApproximateMatchUtils.hasExactHits(hits)) {
    // // GeneMention mention = new GeneMention();
    // // mention.text = geneMention.getMention();
    // //
    // // hits = (ArrayList<SynHit>)
    // // ApproximateMatchUtils.seekExactHitCandidates(mention, hits,
    // // this);
    // // }
    // // -------- TODO testing
    // log.debug("Returning {} candidates for gene mention {}", hits.size(),
    // key.geneName.getText());
    // // scoredHits = null;
    // return hits;
    // // return scoredHits;
    // } catch (ExecutionException e) {
    // throw new GeneCandidateRetrievalException(e);
    // }
    // }

    public Scorer getScorer() {
        return exactScorer;
    }

    public SpellChecker getSpellingChecker() {
        return spellingChecker;
    }

    public Scorer setScorerType(int type) throws GeneCandidateRetrievalException {
        Scorer scorer;
        if (type == SIMPLE_SCORER) {
            scorer = new SimpleScorer();
        } else if (type == TOKEN_JAROWINKLER_SCORER) {
            scorer = new TokenJaroSimilarityScorer();
        } else if (type == MAXENT_SCORER) {
            if (!maxEntModel.equals(MAXENT_SCORER_MODEL)) {
                // InputStream in =
                // this.getClass().getResourceAsStream(MAXENT_SCORER_MODEL);
                scorer = new MaxEntScorer(new File(maxEntModel));
            } else {
                InputStream in = this.getClass().getResourceAsStream(MAXENT_SCORER_MODEL);
                scorer = new MaxEntScorer(in);
            }
        } else if (type == JAROWINKLER_SCORER) {
            scorer = new JaroWinklerScorer();
        } else if (type == LUCENE_SCORER) {
            scorer = new LuceneScorer();
        } else if (type == LEVENSHTEIN_SCORER) {
            scorer = new LevenshteinScorer();
        } else if (type == TFIDF) {
            Thread currentThread = Thread.currentThread();
            TFIDFUtils tfidfNormalizedName = new TFIDFUtils();
//            TFIDFUtils tfidfOriginalName = new TFIDFUtils();
//            TFIDFUtils tfidfNormalizedVariant = new TFIDFUtils();
//            tfidfOriginalName.learnFromLuceneIndex(nameCentricIndexReader,
//                    ORIGINAL_NAME);
            tfidfNormalizedName.learnFromLuceneIndex(nameCentricIndexReader,
                    LOOKUP_SYN_FIELD);
//            tfidfNormalizedVariant.learnFromLuceneIndex(nameCentricIndexReader,
//                    VARIANT_NAME);
            scorer = new TFIDFScorer(tfidfNormalizedName);
        } else {
            throw new GeneCandidateRetrievalException("Unknown mention scorer type: " + type);
        }
        return scorer;
    }

    public String getScorerInfo() {
        if (exactScorer == null) {
            return "Lucene Score (unnormalized)";
        } else {
            return exactScorer.info();
        }
    }

    public int getScorerType() {
        return exactScorer.getScorerType();
    }

    @Override
    public List<SynHit> getCandidates(String originalSearchTerm, QueryGenerator queryGenerator) {
        GeneMention geneMention = new GeneMention(originalSearchTerm, normalizer);
        return getCandidates(geneMention, queryGenerator);
    }

    @Override
    public List<SynHit> getCandidates(GeneMention geneMention, QueryGenerator queryGenerator) {
        return getCandidates(geneMention, geneMention.getTaxonomyIds(), queryGenerator);
    }

    public List<SynHit> getCandidates(GeneMention geneMention, Collection<String> organisms, QueryGenerator queryGenerator) {
        return getCandidates(geneMention, null, organisms != null ? organisms : Collections.emptyList(), queryGenerator);
    }

    @Override
    public List<SynHit> getCandidates(GeneMention geneMention, Collection<String> geneIdsFilter, Collection<String> organisms, QueryGenerator queryGenerator) {
        return getCandidates(geneMention, geneIdsFilter, organisms, true, null, queryGenerator);
    }

    public List<SynHit> getCandidates(GeneMention geneMention, Collection<String> geneIdsFilter, Collection<String> organisms, boolean loadFields, Parameters parameters, QueryGenerator queryGenerator) {
        return getCandidates(geneMention, geneIdsFilter, organisms, loadFields, parameters, -1, queryGenerator);

    }

    public List<SynHit> getCandidates(GeneMention geneMention, Collection<String> geneIdsFilter, Collection<String> organisms, boolean loadFields, Parameters parameters, int numReturnedHits, QueryGenerator queryGenerator) {
        List<SynHit> hits = new ArrayList<>();
        CandidateCacheKey key = new CandidateCacheKey(geneMention.getGeneName());
        key.setLoadSynHitFields(loadFields);
        key.setQueryGenerator(queryGenerator);
        if (parameters != null && parameters.getBoolean(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, Configuration.PARAM_USE_QUERY_FIELD_WEIGHTS)))
            key.setFieldWeights(getFieldWeightsFromParameters(parameters));
        if (queryGenerator instanceof GeneRecordQueryGenerator && ((GeneRecordQueryGenerator) queryGenerator).isUseContextGenesAsRelevanceSignal())
            key.setContextNames(geneMention.getContextGeneNames().collect(Collectors.toSet()));
        if (geneIdsFilter != null)
            key.setGeneIdsFilter(geneIdsFilter);
        if (numReturnedHits > 0)
            key.setMaxHits(numReturnedHits);
        if (organisms == null || organisms.isEmpty()) {
            hits = getCandidatesFromIndex(key);
            if (log.isDebugEnabled()) {
                int geneBegin = geneMention.getOffsets() != null ? geneMention.getBegin() : -1;
                int geneEnd = geneMention.getOffsets() != null ? geneMention.getEnd() : -1;
                log.debug("Returning {} candidates for gene mention {}[{}-{}]", hits.size(), key.getGeneName().getText(), geneBegin, geneEnd);
            }
        }
        if (organisms != null) {
            for (String taxonomyId : organisms) {
                key.setTaxId(taxonomyId);
                hits.addAll(getCandidatesFromIndex(key));
                // TopDocs foundDocs = getCandidatesFromIndex(key);
                // 2. assign score
                // List<SynHit> scoredHits = new ArrayList<SynHit>();
                // scoredHits = scoreHits(foundDocs, key.geneName);
                // 3. combine single hits to candidate clusters
                // ArrayList<SynHit> hits = combineHits(scoredHits);
                // 4. sort by SynHit's score (lucene score)

                if (log.isDebugEnabled()) {
                    int begin = -1;
                    int end = -1;
                    if (geneMention.getOffsets() != null) {
                        begin = geneMention.getBegin();
                        end = geneMention.getEnd();
                    }
                    log.debug("Returning {} candidates for gene mention {}[{}-{}] for taxonomy ID {}",
                            hits.size(), key.getGeneName().getText(), begin, end, organisms);
                }
            }
        }
        //hits = combineHits(hits);
        hits.stream().forEach(h -> h.setCompareType(SynHit.CompareType.SCORE));
        List<SynHit> sortedHits = hits.stream().sorted().collect(Collectors.toList());
        return sortedHits;

    }

    @Override
    public List<SynHit> getCandidates(String geneMentionText, Collection<String> geneIdsFilter, Collection<String> organism, QueryGenerator queryGenerator) {
        GeneMention geneMention = new GeneMention(geneMentionText, normalizer);
        return getCandidates(geneMention, geneIdsFilter, organism, queryGenerator);
    }

    public List<SynHit> getCandidates(String geneMentionText, Collection<String> geneIdsFilter, Collection<String> organism, boolean loadFields, QueryGenerator queryGenerator) {
        GeneMention geneMention = new GeneMention(geneMentionText, normalizer);
        return getCandidates(geneMention, geneIdsFilter, organism, queryGenerator);
    }

    /**
     * This is the method that access the cache. This is important because before the SynHits are returned,
     * they must be cloned or changed on them will write back into the cache.
     *
     * @param key The cache key.
     * @return A new list that contains copies of the cached SynHits.
     * @throws ExecutionException If there is an issue with the cache.
     */
    private List<SynHit> getCandidatesFromIndex(CandidateCacheKey key) {
        List<SynHit> synHits = useLuceneCandidateCache ? candidateCache.get(key) : null;
        if (synHits == null) {
            try {
                synHits = getCandidatesFromIndexWithoutCache(key);
            } catch (IOException e) {
                throw new GeneExpRuntimeException(e);
            }
            candidateCache.put(key, synHits);
        }
        try {
            return synHits.stream().map(SynHit::clone).collect(Collectors.toList());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    private List<SynHit> getCandidatesFromIndexWithoutCache(CandidateCacheKey key)
            throws IOException, BooleanQuery.TooManyClauses {
        long luceneQueryTime = System.currentTimeMillis();
        Query searchQuery = key.generateQuery();
        TopDocs foundDocs = getNameCentricIndexSearcher().search(searchQuery, key.getMaxHits());
        log.debug("searching with query: " + searchQuery + "; found hits: " + foundDocs.totalHits);
        ArrayList<SynHit> synHits = scoreHits(foundDocs, key.getGeneName(), key.isLoadSynHitFields());
//        List<SynHit> collapsedHits = combineHits(synHits);
        luceneQueryTime = System.currentTimeMillis() - luceneQueryTime;
        totalLuceneQueryTime.addAndGet(luceneQueryTime);
        return synHits;
    }

    /**
     * calculate score for each hit
     *
     * @param foundDocs
     * @param geneName
     * @throws IOException
     * @throws CorruptIndexException
     * @throws Exception
     */
    private ArrayList<SynHit> scoreHits(TopDocs foundDocs, GeneName geneName, boolean loadFields)
            throws CorruptIndexException, IOException {
        ArrayList<SynHit> allHits = new ArrayList<>();

        String originalMention = geneName.getText().toLowerCase();
        String normalizedMention = geneName.getNormalizedText();

        ScoreDoc[] scoredDocs = foundDocs.scoreDocs;
        log.debug("ordering candidates for best match to this reference term: " + originalMention + " for top "
                + scoredDocs.length + " candidates");
        candidateLog.trace("Search term: " + normalizedMention);
        IndexSearcher indexSearcher = getNameCentricIndexSearcher();
        for (int i = 0; i < scoredDocs.length; i++) {
            int docID = scoredDocs[i].doc;
            Document d = indexSearcher.doc(docID);
            String indexNormalizedName = d.getField(LOOKUP_SYN_FIELD).stringValue();
            List<String> ids = new ArrayList<>();
            List<Number> priorities = new ArrayList<>();
            String source = null;
            String entityType = null;
            List<String> taxIds = Collections.emptyList();
            Arrays.stream(d.getFields(ID_FIELD)).map(IndexableField::stringValue).map(idAndSyn -> idAndSyn.split(NAME_PRIO_DELIMITER)).forEach(split -> {
                ids.add(split[0]);
                priorities.add(Integer.valueOf(split[1]));
            });
            if (loadFields) {
                source = d.getField(SOURCE).stringValue();
                entityType = d.getField(ENTITY_TYPE).stringValue();
                taxIds = Arrays.stream(d.getFields(TAX_ID_FIELD)).map(IndexableField::stringValue).collect(Collectors.toList());
            }


            double score = 0;
            Scorer scorer = indexNormalizedName.equals(normalizedMention) ? exactScorer : approxScorer;
            if (scorer.getScorerType() == Scorer.LUCENE_SCORER) {
                // use Lucene scoring
                if (indexNormalizedName.equals(normalizedMention)) {
                    // exact matches get perfect score
                    score = PERFECT_SCORE;
                    // score = scoredDocs[i].score;
                } else {
                    // approximate matches get lucene score
                    score = scoredDocs[i].score;
                }
                // Actually, using the DisMax query, another index field might
                // have given the best hit; but we can't say which. The
                // normalized mention is a reasonable choice because the very
                // most good hits stem from normalized variants.
            } else {
                // use external scoring
                score = scorer.getScore(normalizedMention, indexNormalizedName);
            }
            // now make a new synhit object
            SynHit m = new SynHit(indexNormalizedName, score, ids, source, entityType, taxIds);
            m.setMappedMention(originalMention);
            m.setMappedGeneName(geneName);
            m.setSynonymPriorities(priorities);
            m.setLuceneScore(scoredDocs[i].score);
            if (score == PERFECT_SCORE)
                m.setExactMatch(true);
            allHits.add(m);
        }

        return allHits;
    }

    /**
     * Combines all hits with same ID: only best hit is maintained, all others
     * are removed.
     */
    private List<SynHit> combineHits(List<SynHit> allHits) {
        Map<String, SynHit> hitList = new HashMap<>();
        for (SynHit currHit : allHits) {
            for (String id : currHit.getIds()) {
                if (hitList.containsKey(id)) {
                    // if ID already contained check whether current score is
                    // higher
                    SynHit oldHit = hitList.get(id);
                    if (currHit.getLexicalScore() >= oldHit.getLexicalScore()) {
                        hitList.put(id, currHit);
                    }
                } else {
                    // just add new hit if ID is not yet contained
                    hitList.put(id, currHit);
                }
            }
        }
        // now convert into an ArrayList
        Set<SynHit> combinedHits = new HashSet<>();
        for (Iterator<String> iter = hitList.keySet().iterator(); iter.hasNext(); ) {
            String id = iter.next();
            SynHit currHit = hitList.get(id);
            currHit.setCompareType(SynHit.CompareType.SCORE);
            combinedHits.add(currHit);
        }
        return combinedHits.stream().sorted().collect(Collectors.toList());
    }

    @Override
    public List<SynHit> getCandidates(GeneMention geneMention, String organism, QueryGenerator queryGenerator) {
        return getCandidates(geneMention, organism != null ? Arrays.asList(organism) : Collections.emptyList(), queryGenerator);
    }

    @Override
    public List<SynHit> getCandidates(String geneMentionText, String organism, QueryGenerator queryGenerator) {
        return getCandidates(new GeneMention(geneMentionText, normalizer), organism != null ? Arrays.asList(organism) : Collections.emptyList(), queryGenerator);
    }

    @Override
    public List<SynHit> getCandidates(String geneMentionText, Collection<String> organism, QueryGenerator queryGenerator) {
        return getCandidates(new GeneMention(geneMentionText, normalizer), organism, queryGenerator);
    }

    @Override
    /**
     * This will look up the Gene ID of a gene mention in the Lucene index and
     * return the matching Taxonomy IDs for it.
     *
     * @param geneId
     *            A Gene ID to look up in the index
     * @return A Taxonomy ID
     */
    public String mapGeneIdToTaxId(String geneId) {
        try {
            final String fieldValue = geneId + LuceneCandidateRetrieval.NAME_PRIO_DELIMITER + -1;
            TermQuery idQuery = new TermQuery(new Term(ID_FIELD, fieldValue));
            TermQuery geneTypeQuery = new TermQuery(new Term(ENTITY_TYPE, SynHit.TYPE_GEPRO));
            BooleanQuery geneIdQuery = new BooleanQuery.Builder().add(idQuery, MUST).add(geneTypeQuery, FILTER).build();
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs topDocs = indexSearcher.search(geneIdQuery, 1);
            ScoreDoc[] scoredDocs = topDocs.scoreDocs;
            // As mappings should be unique, the set should have a size of one.
            if (topDocs.totalHits.value > 0) {
                int docID = scoredDocs[0].doc;
                Document d = indexSearcher.doc(docID);
                final List<String> ids = Arrays.stream(d.getFields(ID_FIELD)).map(IndexableField::stringValue).map(idandprio -> idandprio.split(LuceneCandidateRetrieval.NAME_PRIO_DELIMITER)).map(split -> split[0]).collect(Collectors.toList());
                final List<String> taxIds = Arrays.stream(d.getFields(TAX_ID_FIELD)).map(IndexableField::stringValue).collect(Collectors.toList());
                String taxId = "";
                for (int i = 0; i < ids.size(); i++) {
                    if (ids.get(i).equals(geneId))
                        taxId = taxIds.get(i);
                }
                if (taxId.isBlank()) {
                    log.warn("GeneID: " + geneId + " has no TaxId assigned.");
                }
                return taxId;
            }
            log.warn("GeneID: " + geneId + " was not found in the index.");
            return "";
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
    }

    public List<SynHit> getIndexEntries(List<String> ids) throws IOException {
        log.warn("LuceneCandidateRetrieval.getIndexEntries(): This method currently does not work as intended since the synonym index is now synonym-centric instead of id-centric. The ID field values have the form id_priority, thus at this place a wildcard query for all priorities would be needed");
        List<SynHit> entries = new ArrayList<>(ids.size());
        for (String id : ids) {
            BooleanClause clause = new BooleanClause(new TermQuery(new Term(ID_FIELD, id + LuceneCandidateRetrieval.NAME_PRIO_DELIMITER + "-1")),
                    FILTER);
            BooleanQuery query = new BooleanQuery.Builder().add(clause).build();
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs result = indexSearcher.search(query, 1);
            if (result.totalHits.value > 0) {
                int docID = result.scoreDocs[0].doc;
                Document d = indexSearcher.doc(docID);
                String source = d.getField(SOURCE).stringValue();
                String entityType = d.getField(ENTITY_TYPE).stringValue();
                List<String> taxIdField = Arrays.stream(d.getFields(TAX_ID_FIELD)).map(IndexableField::stringValue).filter(tax -> !StringUtils.isBlank(tax)).collect(Collectors.toList());
                if (taxIdField.isEmpty()) {
                    log.warn("GeneID: " + id + " has no TaxId assigned.");
                }
                SynHit m = new SynHit("<none>", 0d, Arrays.asList(id), source, entityType, taxIdField);
                entries.add(m);
            }
            entries.add(null);
        }
        return entries;
    }

    /**
     * The record-index equivalent to {@link #getIndexEntries(List)}. Returns {@link GeneRecordHit} instances with all fields loaded.
     *
     * @param ids
     * @return
     * @throws IOException
     */
    public List<SynHit> getIndexRecords(Collection<String> ids) throws IOException {
        List<SynHit> entries = new ArrayList<>(ids.size());
        for (String id : ids) {
            BooleanClause clause = new BooleanClause(new TermQuery(new Term(ID_FIELD, id)),
                    FILTER);
            BooleanQuery query = new BooleanQuery.Builder().add(clause).build();
            IndexSearcher indexSearcher = getGeneRecordIndexSearcher();
            TopDocs result = indexSearcher.search(query, 1);
            if (result.totalHits.value > 0) {
                int docID = result.scoreDocs[0].doc;
                Document d = indexSearcher.doc(docID);
                GeneRecordHit m = getRecordHit(true, new GeneName("<retrieved by id " + id + ">", normalizer), result.scoreDocs[0], d);
                entries.add(m);
            }

        }
        return entries;
    }

    public List<String> getSynonyms(String id) {
        try {
            List<String> ret = Collections.emptyList();
            BooleanClause clause = new BooleanClause(new WildcardQuery(new Term(ID_FIELD, id + NAME_PRIO_DELIMITER + "*")),
                    Occur.FILTER);
            BooleanQuery query = new BooleanQuery.Builder().add(clause).build();
            int maxRet = 200;
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs result = indexSearcher.search(query, maxRet);
            if (result.totalHits.value > 0) {
                ret = new ArrayList<>(maxRet);
                for (int i = 0; i < result.scoreDocs.length; ++i) {
                    Document doc = indexSearcher.doc(result.scoreDocs[i].doc);
                    String geneName = doc.getField(LOOKUP_SYN_FIELD).stringValue();
                    ret.add(geneName);
                }
            }
            return ret;
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
    }

    public List<SynHit> scoreIdsByNGramSynonyms(String synonymsString, Set<String> geneIds) {
        new CandidateCacheKey(new GeneName(synonymsString, normalizer));
        QueryBuilder qb = new QueryBuilder(ngramAnalyzer);
        Query ngramQuery = qb.createBooleanQuery(LOOKUP_SYN_FIELD_NGRAMS, synonymsString);
        Stream<TermQuery> idTermQueries = geneIds.stream().map(id -> new TermQuery(new Term(ID_FIELD, id)));

        BooleanQuery.Builder filterBuilder = new BooleanQuery.Builder();
        idTermQueries.forEach(q -> filterBuilder.add(q, Occur.SHOULD));

        // qb can be null if the tokens of the synonyms have all only one character. Then, there are no 2 or 3 grams
        if (qb != null) {
            BooleanQuery.Builder mainQb = new BooleanQuery.Builder().add(ngramQuery, Occur.MUST).add(filterBuilder.build(), Occur.FILTER);
            List<SynHit> allHits = new ArrayList<>();
            try {
                IndexSearcher indexSearcher = getNameCentricIndexSearcher();
                TopDocs topdocs = indexSearcher.search(mainQb.build(), 1000);
                for (ScoreDoc doc : topdocs.scoreDocs) {
                    Document d = indexSearcher.doc(doc.doc);
                    String indexNormalizedName = d.getField(LOOKUP_SYN_FIELD).stringValue();
                    List<String> ids = Collections.emptyList();
                    List<String> taxIds = Collections.emptyList();
                    String source = null;
                    String entityType = null;
//                List<String> ids = new ArrayList<>();
//                List<Number> priorities = new ArrayList<>();
//                String source = d.getField(SynonymIndexFieldNames.SOURCE).stringValue();
//                String entityType = d.getField(SynonymIndexFieldNames.ENTITY_TYPE).stringValue();
//                Arrays.stream(d.getFields(SynonymIndexFieldNames.ID_FIELD)).map(IndexableField::stringValue).map(idAndSyn -> idAndSyn.split(NAME_PRIO_DELIMITER)).forEach(split -> {
//                    ids.add(split[0]);
//                    priorities.add(Integer.valueOf(split[1]));
//                });
//                List<String> taxIds = Arrays.stream(d.getFields(SynonymIndexFieldNames.TAX_ID_FIELD)).map(IndexableField::stringValue).collect(Collectors.toList());
                    SynHit m = new SynHit(indexNormalizedName, doc.score, ids, source, entityType, taxIds);
//                    m.setSynonymPriorities(priorities);
                    allHits.add(m);
                }

            } catch (IOException e) {
                throw new GeneExpRuntimeException(e);
            }
            return allHits;
        }
        return Collections.emptyList();
    }

    @Override
    public List<SynHit> scoreIdsByBoWSynonyms(Collection<String> allSynonyms, Set<String> ids, QueryGenerator qg) {
        CandidateCacheKey cacheKey = new CandidateCacheKey(new GeneName(String.join(" ", allSynonyms), normalizer), null);
        cacheKey.setGeneIdsFilter(ids);
        cacheKey.setQueryGenerator(qg);
        cacheKey.setMaxHits(1000);
        cacheKey.setLoadSynHitFields(false);
        return getCandidatesFromIndex(cacheKey);
    }

    @Override
    public List<SynHit> getCandidates(GeneMention gm, Collection<String> taxId, Parameters parameters, QueryGenerator queryGenerator) {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    /**
     * <p>Scores each synonym in <tt>allSynonym</tt> against the IDs in <tt>ids</tt>.</p>
     * <p>Each resulting SynHit adds its mention score to the ID represented by this SynHit.</p>
     *
     * @param allSynonyms
     * @param ids
     * @param qg
     * @return
     */
    public Pair<Map<String, Double>, Map<String, List<String>>> scoreSynonymsRecordIndex(Collection<GeneName> allSynonyms, Set<String> ids, Function<GeneRecordHit, String[]> synhit2namesFunc, QueryGenerator qg) {
        Map<String, Double> scores = new HashMap<>();
        Map<String, List<String>> ids2synonyms = new HashMap<>();
        for (GeneName synonym : allSynonyms) {
            CandidateCacheKey cacheKey = new CandidateCacheKey(synonym, null);
            cacheKey.setGeneIdsFilter(ids);
            cacheKey.setQueryGenerator(qg);
            cacheKey.setMaxHits(1000);
            cacheKey.setLoadSynHitFields(synhit2namesFunc != null);
            for (SynHit sh : getCandidatesFromIndex(cacheKey)) {
                scores.merge(sh.getId(), sh.getLexicalScore(), (s1, s2) -> s1 + s2);
                if (synhit2namesFunc != null) {
                    List<String> names = ids2synonyms.compute(sh.getId(), (k, v) -> v != null ? v : new ArrayList<>());
                    String[] newnames = synhit2namesFunc.apply((GeneRecordHit) sh);
                    if (newnames != null) {
                        for (String name : newnames)
                            if (name != null)
                                names.add(name);
                    }
                }
            }
        }
        return new ImmutablePair<>(scores, ids2synonyms);
    }

    public List<SynHit> scoreIdsByExactSynonyms(Collection<String> allSynonyms, Set<String> geneIds) {
        Stream<TermQuery> exactSynonymTermQueries = allSynonyms.stream().map(s -> new TermQuery(new Term(LOOKUP_SYN_FIELD_EXACT, s)));
        Stream<TermQuery> idTermQueries = geneIds.stream().map(id -> new TermQuery(new Term(ID_FIELD, id)));
        BooleanQuery.Builder qb = new BooleanQuery.Builder();
        exactSynonymTermQueries.forEach(q -> qb.add(q, Occur.SHOULD));
        BooleanQuery.Builder filterBuilder = new BooleanQuery.Builder();
        idTermQueries.forEach(q -> filterBuilder.add(q, Occur.SHOULD));
        BooleanQuery.Builder exactSynonymQuery = qb.add(filterBuilder.build(), Occur.FILTER);
        List<SynHit> allHits = new ArrayList<>();
        try {
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs topdocs = indexSearcher.search(exactSynonymQuery.build(), 1000);
            for (ScoreDoc doc : topdocs.scoreDocs) {
                Document d = indexSearcher.doc(doc.doc);
                String indexNormalizedName = d.getField(LOOKUP_SYN_FIELD).stringValue();
                List<String> taxIds = Collections.emptyList();
                String source = null;
                String entityType = null;
                List<String> ids = new ArrayList<>();
//                List<Number> priorities = new ArrayList<>();
//                String source = d.getField(SynonymIndexFieldNames.SOURCE).stringValue();
//                String entityType = d.getField(SynonymIndexFieldNames.ENTITY_TYPE).stringValue();
                Arrays.stream(d.getFields(ID_FIELD)).map(IndexableField::stringValue).map(idAndSyn -> idAndSyn.split(NAME_PRIO_DELIMITER)).forEach(split -> {
                    ids.add(split[0]);
//                    priorities.add(Integer.valueOf(split[1]));
                });
//                List<String> taxIds = Arrays.stream(d.getFields(SynonymIndexFieldNames.TAX_ID_FIELD)).map(IndexableField::stringValue).collect(Collectors.toList());
                SynHit m = new SynHit(indexNormalizedName, doc.score, ids, source, entityType, taxIds);
//                m.setSynonymPriorities(priorities);
                allHits.add(m);
            }

        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
        return allHits;
    }


    public List<String> getPriorityNames(String id, int priority) {
        try {
            List<String> ret = Collections.emptyList();
            BooleanClause ic = new BooleanClause(new TermQuery(new Term(ID_FIELD, id + LuceneCandidateRetrieval.NAME_PRIO_DELIMITER + priority)), Occur.FILTER);
            BooleanQuery query = new BooleanQuery.Builder().add(ic).build();
            int maxRet = 1;
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs result = indexSearcher.search(query, maxRet);
            if (result.totalHits.value > 0) {
                ret = new ArrayList<>(maxRet);
                for (int i = 0; i < result.scoreDocs.length; ++i) {
                    Document doc = indexSearcher.doc(result.scoreDocs[i].doc);
                    String name = doc.getField(LOOKUP_SYN_FIELD).stringValue();
                    ret.add(name);
                }
            }
            return ret;
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
    }

    public Map<String, String> getPriorityNamesMap(Collection<String> ids, int priority) {
        try {
            Map<String, String> ret = Collections.emptyMap();
            BooleanQuery.Builder qb = new BooleanQuery.Builder();
            ids.stream().map(id -> new TermQuery(new Term(ID_FIELD, id + LuceneCandidateRetrieval.NAME_PRIO_DELIMITER + priority))).forEach(q -> qb.add(q, Occur.SHOULD));
            BooleanQuery query = new BooleanQuery.Builder().add(qb.build(), Occur.FILTER).build();
            IndexSearcher indexSearcher = getNameCentricIndexSearcher();
            TopDocs result = indexSearcher.search(query, ids.size());
            if (result.totalHits.value > 0) {
                ret = new HashMap<>(ids.size());
                for (int i = 0; i < result.scoreDocs.length; ++i) {
                    Document doc = indexSearcher.doc(result.scoreDocs[i].doc);
                    String id = null;
                    id = Stream.of(doc.getFields(ID_FIELD)).map(IndexableField::stringValue).map(x -> x.split(NAME_PRIO_DELIMITER)[0]).filter(x -> ids.contains(x)).findFirst().get();
                    String name = doc.getField(LOOKUP_SYN_FIELD).stringValue();
                    ret.put(id, name);
                }
            }
            return ret;
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
    }

    public List<String> getPriorityNames(Collection<String> ids, int priority) throws IOException {
        final Stream.Builder<List<String>> builder = Stream.builder();
        for (String id : ids) {
            builder.accept(getPriorityNames(id, priority));
        }
        return builder.build().flatMap(Collection::stream).collect(Collectors.toList());
    }

    public void close() {
        try {
            if (nameCentricIndexReader != null)
                nameCentricIndexReader.close();
            if (geneRecordIndexReader != null)
                geneRecordIndexReader.close();
        } catch (IOException e) {
            log.error("Could not close lucene indices", e);
        }
    }

    @Override
    public List<SynHit> getFamilyNames(GeneMention gm, QueryGenerator queryGenerator) {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    @Override
    public List<SynHit> getOriginalNamesIndexRecords(Collection<String> geneIds, GeneName geneName) {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    @Override
    public List<SynHit> getOriginalNamesIndexRecords(Collection<String> geneIds) {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    @Override
    public TFIDFScorer getTFIDFOnGeneRecordNames() {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    @Override
    public void setFulltextFieldsToRecordHits(Collection<? extends SynHit> recordHits, Collection<String> recordContextFieldNames) {
        throw new NotImplementedException("This method should be implemented when needed.");
    }

    @Override
    public Pair<Map<String, Double>, Map<String, Set<String>>> scoreSynonymsRecordIndex(String queryType, Map<String, Collection<GeneName>> ids2entities, Function<GeneRecordHit, String[]> synhit2namesFunc, QueryGenerator qg) {
        return null;
    }

    private IndexSearcher getGeneRecordIndexSearcher() {
        return geneRecordIndexSearcher.compute(Thread.currentThread(), (k, v) -> v != null ? v : new IndexSearcher(geneRecordIndexReader, executorService));
    }

    private IndexSearcher getNameCentricIndexSearcher() {
        IndexSearcher indexSearcher = mentionIndexSearcher.compute(Thread.currentThread(), (k, v) -> v != null ? v : new IndexSearcher(nameCentricIndexReader, executorService));
        // The default since Lucene 5 is BM25. But for our purposes, the
        // classic
        // Lucene Similarity works better (checked with IGN train).
        indexSearcher.setSimilarity(new ClassicSimilarity());
        return indexSearcher;
    }

    private List<SynHit> getCandidatesFromRecordIndex(CandidateCacheKey key) throws IOException {
        try {
            Query query = key.generateQuery();

            TopScoreDocCollector resultsCollector = TopScoreDocCollector.create(key.getMaxHits(), key.getMaxHits());
            IndexSearcher indexSearcher = getGeneRecordIndexSearcher();
            indexSearcher.search(query, resultsCollector);
            TopDocs topDocs = resultsCollector.topDocs();
            List<SynHit> ret = new ArrayList<>();
            boolean loadSynHitFields = key.isLoadSynHitFields();
            GeneName geneName = key.getGeneName();
            for (ScoreDoc doc : topDocs.scoreDocs) {
                Document document = indexSearcher.doc(doc.doc);
                GeneRecordHit sh = getRecordHit(loadSynHitFields, geneName, doc, document);
                ret.add(sh);
            }
            return ret;
        } catch (BooleanQuery.TooManyClauses e) {
            log.warn("Got too many clauses exception from gene name \"{}\". Assuming that this is a tagging error and not returning any candidates.", key.getGeneName().getText());
        }
        return Collections.emptyList();
    }

    @NotNull
    private GeneRecordHit getRecordHit(boolean loadSynHitFields, GeneName geneName, ScoreDoc doc, Document document) {
        String id = document.getField(ID_FIELD).stringValue();
        String taxId = null;
        if (loadSynHitFields)
            taxId = document.getField(TAX_ID_FIELD).stringValue();

        IndexableField symbolField = document.getField(SYMBOL);
        String symbol = symbolField != null ? symbolField.stringValue() : "";
        GeneRecordHit sh = new GeneRecordHit(symbol, doc.score, id, "<no source specified>");
        sh.setMappedMention(geneName != null ? geneName.getText() : "none");
        sh.setLuceneScore(doc.score);
        if (taxId != null) {
            sh.setTaxIds(Collections.singletonList(taxId));
            sh.setTaxId(taxId);
        }
        if (loadSynHitFields) {
            sh.setSymbol(symbol);
            Optional.ofNullable(document.getField(SYMBOL_FROM_NOMCENCLATURE)).ifPresent(f -> sh.setNomenclature(f.stringValue()));
            Optional.ofNullable(document.getField(CHROMOSOME)).ifPresent(f -> sh.setChromosome(f.stringValue()));
            Optional.ofNullable(document.getField(MAPLOCATION)).ifPresent(f -> sh.setMapLocation(f.stringValue()));
            sh.setSynonyms(Arrays.stream(document.getFields(SYNONYMS)).map(IndexableField::stringValue).toArray(String[]::new));
            sh.setFullNames(Arrays.stream(document.getFields(FULL_NAMES)).map(IndexableField::stringValue).toArray(String[]::new));
            sh.setOtherDesignations(Arrays.stream(document.getFields(OTHER_DESIGNATIONS)).map(IndexableField::stringValue).toArray(String[]::new));
            sh.setXrefs(Arrays.stream(document.getFields(XREFS)).map(IndexableField::stringValue).toArray(String[]::new));
            sh.setUniprotNames(Arrays.stream(document.getFields(UNIPROT_NAMES)).map(IndexableField::stringValue).toArray(String[]::new));
            sh.setBioThesaurusNames(Arrays.stream(document.getFields(BIO_THESAURUS)).map(IndexableField::stringValue).toArray(String[]::new));
            for (String synonymField : GeneRecordQueryGenerator.SYNONYM_FIELDS) {
                Optional<String> anyExactMatch = Arrays.stream(document.getFields(synonymField)).map(IndexableField::stringValue).filter(f -> f.equals(geneName.getNormalizedText())).findAny();
                if (anyExactMatch.isPresent()) {
                    sh.setLexicalScore(PERFECT_SCORE);
                    sh.setSynonym(anyExactMatch.get());
                }
            }
        }
        return sh;
    }

    public Set<GeneRecordHit> getGeneRecords(Collection<String> ids) {
        try {
            Set<GeneRecordHit> hits = new HashSet<>();
            IndexSearcher indexSearcher = getGeneRecordIndexSearcher();
            BooleanQuery.Builder mainBuilder = new BooleanQuery.Builder();
            mainBuilder.add(new MatchAllDocsQuery(), Occur.MUST);
            BooleanQuery.Builder filterBuilder = new BooleanQuery.Builder();
            ids.stream().forEach(id -> filterBuilder.add(new TermQuery(new Term(ID_FIELD, id)), Occur.SHOULD));
            mainBuilder.add(filterBuilder.build(), Occur.FILTER);


            TopDocs topdocs = indexSearcher.search(mainBuilder.build(), ids.size());
            for (ScoreDoc sd : topdocs.scoreDocs) {
                Document doc = indexSearcher.doc(sd.doc);
                GeneRecordHit recordHit = getRecordHit(true, null, sd, doc);
                hits.add(recordHit);
            }
            return hits;
        } catch (IOException e) {
            throw new GeneExpRuntimeException(e);
        }
    }


    private Map<String, Float> getFieldWeightsFromParameters(Map<String, Object> parameterMap) {
        if (parameterMap == null)
            parameterMap = Collections.emptyMap();
        Map<String, Float> fieldWeights = new HashMap<>();
        Object tieBreaker = parameterMap.get(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, Configuration.PARAM_DISMAX_TIE_BREAKER));
        if (tieBreaker == null)
            tieBreaker = globalFieldWeights.getOrDefault(Configuration.PARAM_DISMAX_TIE_BREAKER, 0.3f);
        else
            tieBreaker = Float.parseFloat((String) tieBreaker);
        fieldWeights.put(Configuration.PARAM_DISMAX_TIE_BREAKER, (Float) tieBreaker);
        for (String field : GeneRecordQueryGenerator.ALL_FIELDS) {
            Float defaultValue = globalFieldWeights.getOrDefault(field, 1f);

            String parameterValue = (String) parameterMap.get(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, field));
            float finalValue;
            if (parameterValue != null)
                finalValue = Float.parseFloat(parameterValue);
            else
                finalValue = defaultValue;
            fieldWeights.put(field, finalValue);
        }
        for (String field : GeneRecordQueryGenerator.SYNONYM_FIELDS) {
            String exactFieldName = field + "_exact";
            Float defaultValue = globalFieldWeights.getOrDefault(exactFieldName, 1f);

            String parameterValue = (String) parameterMap.get(Configuration.dot(Configuration.PREFIX_CANDIDATE_RETRIEVAL, exactFieldName));
            float finalValue;
            if (parameterValue != null)
                finalValue = Float.parseFloat(parameterValue);
            else
                finalValue = defaultValue;
            fieldWeights.put(exactFieldName, finalValue);
        }
        return fieldWeights;
    }
}
