/*
 * Decompiled with CFR 0.152.
 */
package de.julielab.bioportal.ontologies;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import de.julielab.bioportal.ontologies.HttpHandler;
import de.julielab.bioportal.ontologies.OntologyListRetriver;
import de.julielab.bioportal.ontologies.data.OntologyClassMapping;
import de.julielab.bioportal.ontologies.data.OntologyMetaData;
import de.julielab.bioportal.util.BioPortalOntologyToolsException;
import de.julielab.bioportal.util.BioPortalToolUtils;
import de.julielab.bioportal.util.ResourceNotFoundException;
import de.julielab.java.utilities.FileUtilities;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.ParseException;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MappingDownloader {
    private static final Logger log = LoggerFactory.getLogger(MappingDownloader.class);
    private static final Logger errors = LoggerFactory.getLogger((String)(MappingDownloader.class.getCanonicalName() + ".downloaderrors"));
    private Gson gson;
    private ExecutorService executorService;
    private Pattern jsonSyntaxErrorPattern = Pattern.compile("[A-Za-z _.:]+[0-9]+[A-Za-z ]+([0-9]+)");
    private Matcher jsonSyntaxErrorMatcher = this.jsonSyntaxErrorPattern.matcher("");
    private HttpHandler httpHandler;
    private OntologyListRetriver ontologyListRetriver;

    public MappingDownloader(String apiKey) {
        this.gson = new Gson();
        this.httpHandler = new HttpHandler(apiKey);
        this.ontologyListRetriver = new OntologyListRetriver(this.httpHandler);
        this.executorService = Executors.newFixedThreadPool(6);
    }

    public void downloadOntologyMappings(File mappingsDir, File ontosDir, Set<String> ontologiesToDownload) throws ParseException, IOException, BioPortalOntologyToolsException {
        int i;
        errors.info("------- Error report for download beginning at " + new Date() + " ---------\n", (Object)"UTF-8", (Object)true);
        if (!mappingsDir.exists()) {
            mappingsDir.mkdirs();
        }
        List<OntologyMetaData> ontologiesMetaData = this.ontologyListRetriver.getOntologiesMetaData(null, ontologiesToDownload);
        this.removeAlreadyDownloadedButLast(ontologiesMetaData, mappingsDir);
        this.removeNotInOntologiesDirectory(ontologiesMetaData, ontosDir);
        log.info("Starting the download of mappings for {} ontologies.", (Object)ontologiesMetaData.size());
        ArrayList futures = new ArrayList();
        for (i = 0; i < ontologiesMetaData.size(); ++i) {
            OntologyMetaData ontologyMetaData = ontologiesMetaData.get(i);
            try {
                log.info("Downloading mappings for ontology {}.", (Object)(i + 1 + "/" + ontologiesMetaData.size()));
                DownloadWorker downloadWorker = new DownloadWorker(ontologyMetaData, mappingsDir);
                Future<?> future = this.executorService.submit(downloadWorker);
                futures.add(future);
                continue;
            }
            catch (JsonSyntaxException jsonSyntaxException) {
                log.warn("Mappings of ontology {} could not be downloaded due to a JSON parsing error: {}", (Object)ontologyMetaData.acronym, (Object)jsonSyntaxException.getMessage());
            }
        }
        i = 1;
        for (Future future : futures) {
            try {
                future.get();
                log.info("{} of {} ontology mappings successfully downloaded.", (Object)i++, (Object)futures.size());
            }
            catch (InterruptedException | ExecutionException e) {
                e.printStackTrace();
            }
        }
        this.executorService.shutdown();
    }

    private void removeNotInOntologiesDirectory(List<OntologyMetaData> ontologiesMetaData, File ontosDir) {
        if (ontosDir == null) {
            return;
        }
        log.info("Removing ontologies from the mapping download list that are not found in directory {}", (Object)ontosDir);
        File[] ontologyFiles = ontosDir.listFiles(f -> !f.getName().equals(".DS_Store"));
        HashSet<String> foundAcronyms = new HashSet<String>();
        for (int i = 0; i < ontologyFiles.length; ++i) {
            File file = ontologyFiles[i];
            String acronym = BioPortalToolUtils.getAcronymFromFileName(file);
            foundAcronyms.add(acronym);
        }
        log.debug("Found ontologies: {}", foundAcronyms);
        Iterator<OntologyMetaData> it = ontologiesMetaData.iterator();
        while (it.hasNext()) {
            OntologyMetaData metaData = it.next();
            System.out.println(metaData.acronym);
            if (foundAcronyms.contains(metaData.acronym)) continue;
            log.debug("Removing ontology {} for mapping download because it was not found in directory {}", (Object)metaData.acronym, (Object)ontosDir);
            it.remove();
        }
    }

    private void storeOntologyMappings(OntologyMetaData ontologyMetaData, File ontosDir) throws ParseException, IOException, ResourceNotFoundException {
        String includedAttributes = "pagesize=500&no_context=true&no_links=true";
        String mappingsUrl = ontologyMetaData.apiUrl() + "/mappings?" + includedAttributes;
        File mappingsFile = new File(ontosDir.getAbsolutePath() + File.separatorChar + ontologyMetaData.acronym + ".map.json" + ".gz");
        if (mappingsFile.exists() && mappingsFile.length() != 0L) {
            log.info("Mapping file {} already exists and is not empty. Not downloading again mappings for ontology {}", (Object)mappingsFile, (Object)ontologyMetaData.acronym);
            return;
        }
        try (BufferedOutputStream os = FileUtilities.getOutputStreamToFile((File)mappingsFile);){
            log.info("Mappings of ontology {} are being downloaded (API URL: {}).", (Object)ontologyMetaData.bioportalPurl(), (Object)ontologyMetaData.apiUrl());
            ArrayList<OntologyClassMapping> mappings = new ArrayList<OntologyClassMapping>();
            String responseString = null;
            try {
                responseString = EntityUtils.toString((HttpEntity)this.httpHandler.sendGetRequest(mappingsUrl));
            }
            catch (ResourceNotFoundException e) {
                throw new ResourceNotFoundException(ontologyMetaData.name);
            }
            OntologyMappingsPage ontologyMappingsPage = this.parseResponse(responseString);
            mappings.addAll(ontologyMappingsPage.collection);
            log.info("Page {} of {} has been downloaded successfully for ontology {}.", new Object[]{ontologyMappingsPage.page, ontologyMappingsPage.pageCount, ontologyMetaData.acronym});
            while (ontologyMappingsPage.links.nextPage != null && ontologyMappingsPage.page <= ontologyMappingsPage.pageCount) {
                responseString = EntityUtils.toString((HttpEntity)this.httpHandler.sendGetRequest(ontologyMappingsPage.links.nextPage));
                ontologyMappingsPage = this.parseResponse(responseString);
                if (null != ontologyMappingsPage) {
                    mappings.addAll(ontologyMappingsPage.collection);
                    if (ontologyMappingsPage.collection.isEmpty()) {
                        log.warn("Page {} of {} was downloaded empty for ontology {}. The response string was: {}", new Object[]{ontologyMappingsPage.page, ontologyMappingsPage.pageCount, ontologyMetaData.acronym, responseString});
                        continue;
                    }
                    log.info("Page {} of {} has been downloaded successfully for ontology {}.", new Object[]{ontologyMappingsPage.page, ontologyMappingsPage.pageCount, ontologyMetaData.acronym});
                    continue;
                }
                log.warn("Current page of ontology \"{}\" could not be downloaded. Skipping this ontology.", (Object)ontologyMetaData.acronym);
            }
            if (ontologyMappingsPage.page < ontologyMappingsPage.pageCount) {
                log.warn("Only {} of {} pages of mappings have been downloaded for ontology {}.", new Object[]{ontologyMappingsPage.page, ontologyMappingsPage.pageCount, ontologyMetaData.name});
            }
            IOUtils.write((String)"[", (OutputStream)os, (String)"UTF-8");
            for (int i = 0; i < mappings.size(); ++i) {
                OntologyClassMapping mapping = (OntologyClassMapping)mappings.get(i);
                IOUtils.write((String)mapping.toString(), (OutputStream)os, (String)"UTF-8");
                if (i < mappings.size() - 1) {
                    IOUtils.write((String)",", (OutputStream)os, (String)"UTF-8");
                }
                IOUtils.write((String)"\n", (OutputStream)os, (String)"UTF-8");
            }
            IOUtils.write((String)"]", (OutputStream)os, (String)"UTF-8");
            log.info("{} mappings of ontology \"{}\" have been downloaded.", (Object)mappings.size(), (Object)ontologyMetaData.name);
        }
        catch (Exception e) {
            String msg = "File \"" + mappingsFile.getAbsolutePath() + "\" is deleted because the ontology mapping could not be downloaded completely due to error: " + e.getMessage();
            errors.info(msg + "\n", (Object)"UTF-8", (Object)true);
            log.info(msg);
            mappingsFile.delete();
        }
    }

    private OntologyMappingsPage parseResponse(String responseString) {
        OntologyMappingsPage ontologyMappingsPage = null;
        try {
            ontologyMappingsPage = (OntologyMappingsPage)this.gson.fromJson(responseString, OntologyMappingsPage.class);
        }
        catch (JsonSyntaxException e) {
            log.error("Error message: \"{}\".", (Object)e.getMessage());
            this.jsonSyntaxErrorMatcher.reset(e.getMessage());
            if (this.jsonSyntaxErrorMatcher.matches()) {
                int column = Integer.parseInt(this.jsonSyntaxErrorMatcher.group(1));
                String snippet = responseString.substring(Math.max(0, column - 40), Math.min(column + 40, responseString.length()));
                log.error("Error snippet: \"{}\"", (Object)snippet);
            } else {
                log.warn("Tried to extract column number from error output but failed (error message was not of expected format).");
            }
            throw e;
        }
        return ontologyMappingsPage;
    }

    private void removeAlreadyDownloadedButLast(List<OntologyMetaData> ontologiesMetaData, File ontosDir) {
        log.info("Removing already downloaded ontology mappings from ToDo list.");
        File[] downloadedOntologies = ontosDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".map.json.gz");
            }
        });
        if (downloadedOntologies.length == 0) {
            log.info("No already downladed mapping files with extension .map.json.gz found.");
            return;
        }
        HashSet<String> ontologyNamesToRemoveFromDownloadList = new HashSet<String>(downloadedOntologies.length);
        for (File f : downloadedOntologies) {
            String filename = f.getName();
            String ontoAcronym = filename.substring(0, filename.length() - ".map.json.gz".length());
            ontologyNamesToRemoveFromDownloadList.add(ontoAcronym);
        }
        ArrayList<String> removedOntologies = new ArrayList<String>();
        Iterator<OntologyMetaData> it = ontologiesMetaData.iterator();
        while (it.hasNext()) {
            OntologyMetaData meta = it.next();
            if (!ontologyNamesToRemoveFromDownloadList.contains(meta.acronym)) continue;
            it.remove();
            removedOntologies.add(meta.acronym);
        }
        log.info("Removed the following {} ontologies from download-list: {}.", (Object)removedOntologies.size(), (Object)StringUtils.join(removedOntologies, (String)", "));
    }

    public void shutdown() {
        this.executorService.shutdown();
    }

    private class DownloadWorker
    implements Runnable {
        private OntologyMetaData ontologyMetaData;
        private File mappingsDir;

        public DownloadWorker(OntologyMetaData ontologyMetaData, File mappingsDir) {
            this.ontologyMetaData = ontologyMetaData;
            this.mappingsDir = mappingsDir;
        }

        @Override
        public void run() {
            try {
                MappingDownloader.this.storeOntologyMappings(this.ontologyMetaData, this.mappingsDir);
            }
            catch (ResourceNotFoundException | IOException | ParseException e) {
                log.error("Exception while downloading mappings for ontology " + this.ontologyMetaData.acronym, e);
            }
        }
    }

    static class OntologyMappingsPage {
        int page;
        int pageCount;
        OntolgoyClassesPageLinks links;
        List<OntologyClassMapping> collection;

        OntologyMappingsPage() {
        }

        class OntolgoyClassesPageLinks {
            String nextPage;

            OntolgoyClassesPageLinks() {
            }
        }
    }
}

