/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.MQTTConnection;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.subscriptions.Subscription;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Session {
    private static final Logger LOG = LoggerFactory.getLogger(Session.class);
    private final String clientId;
    private boolean clean;
    private Will will;
    private final Queue<SessionRegistry.EnqueuedMessage> sessionQueue;
    private final AtomicReference<SessionStatus> status = new AtomicReference<SessionStatus>(SessionStatus.DISCONNECTED);
    private MQTTConnection mqttConnection;
    private List<Subscription> subscriptions = new ArrayList<Subscription>();
    private final Map<Integer, SessionRegistry.EnqueuedMessage> inflightWindow = new HashMap<Integer, SessionRegistry.EnqueuedMessage>();
    private final DelayQueue<InFlightPacket> inflightTimeouts = new DelayQueue();
    private final Map<Integer, MqttPublishMessage> qos2Receiving = new HashMap<Integer, MqttPublishMessage>();
    private final AtomicInteger inflightSlots = new AtomicInteger(10);

    Session(String clientId, boolean clean, Will will, Queue<SessionRegistry.EnqueuedMessage> sessionQueue) {
        this(clientId, clean, sessionQueue);
        this.will = will;
    }

    Session(String clientId, boolean clean, Queue<SessionRegistry.EnqueuedMessage> sessionQueue) {
        if (sessionQueue == null) {
            throw new IllegalArgumentException("sessionQueue parameter can't be null");
        }
        this.clientId = clientId;
        this.clean = clean;
        this.sessionQueue = sessionQueue;
    }

    void update(boolean clean, Will will) {
        this.clean = clean;
        this.will = will;
    }

    void markConnecting() {
        this.assignState(SessionStatus.DISCONNECTED, SessionStatus.CONNECTING);
    }

    boolean completeConnection() {
        return this.assignState(SessionStatus.CONNECTING, SessionStatus.CONNECTED);
    }

    void bind(MQTTConnection mqttConnection) {
        this.mqttConnection = mqttConnection;
    }

    public boolean disconnected() {
        return this.status.get() == SessionStatus.DISCONNECTED;
    }

    public boolean connected() {
        return this.status.get() == SessionStatus.CONNECTED;
    }

    public String getClientID() {
        return this.clientId;
    }

    public List<Subscription> getSubscriptions() {
        return new ArrayList<Subscription>(this.subscriptions);
    }

    public void addSubscriptions(List<Subscription> newSubscriptions) {
        this.subscriptions.addAll(newSubscriptions);
    }

    public boolean hasWill() {
        return this.will != null;
    }

    public Will getWill() {
        return this.will;
    }

    boolean assignState(SessionStatus expected, SessionStatus newState) {
        return this.status.compareAndSet(expected, newState);
    }

    public void closeImmediately() {
        this.mqttConnection.dropConnection();
    }

    public void disconnect() {
        boolean res = this.assignState(SessionStatus.CONNECTED, SessionStatus.DISCONNECTING);
        if (!res) {
            return;
        }
        this.mqttConnection = null;
        this.will = null;
        this.assignState(SessionStatus.DISCONNECTING, SessionStatus.DISCONNECTED);
    }

    boolean isClean() {
        return this.clean;
    }

    public void processPubRec(int pubRecPacketId) {
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(pubRecPacketId);
        if (removed == null) {
            LOG.warn("Received a PUBREC with not matching packetId");
            return;
        }
        if (removed instanceof SessionRegistry.PubRelMarker) {
            LOG.info("Received a PUBREC for packetId that was already moved in second step of Qos2");
            return;
        }
        this.inflightWindow.put(pubRecPacketId, new SessionRegistry.PubRelMarker());
        this.inflightTimeouts.add(new InFlightPacket(pubRecPacketId, 5000L));
        MqttMessage pubRel = MQTTConnection.pubrel(pubRecPacketId);
        this.mqttConnection.sendIfWritableElseDrop(pubRel);
        this.drainQueueToConnection();
    }

    public void processPubComp(int messageID) {
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(messageID);
        if (removed == null) {
            LOG.warn("Received a PUBCOMP with not matching packetId");
            return;
        }
        removed.release();
        this.inflightSlots.incrementAndGet();
        this.drainQueueToConnection();
    }

    public void sendPublishOnSessionAtQos(Topic topic, MqttQoS qos, ByteBuf payload) {
        switch (qos) {
            case AT_MOST_ONCE: {
                if (!this.connected()) break;
                this.mqttConnection.sendPublishNotRetainedQos0(topic, qos, payload);
                break;
            }
            case AT_LEAST_ONCE: {
                this.sendPublishQos1(topic, qos, payload);
                break;
            }
            case EXACTLY_ONCE: {
                this.sendPublishQos2(topic, qos, payload);
                break;
            }
            case FAILURE: {
                LOG.error("Not admissible");
            }
        }
    }

    private void sendPublishQos1(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (!this.connected() && this.isClean()) {
            return;
        }
        if (this.canSkipQueue()) {
            this.inflightSlots.decrementAndGet();
            int packetId = this.mqttConnection.nextPacketId();
            payload.retain();
            SessionRegistry.EnqueuedMessage old = this.inflightWindow.put(packetId, new SessionRegistry.PublishedMessage(topic, qos, payload));
            if (old != null) {
                old.release();
                this.inflightSlots.incrementAndGet();
            }
            this.inflightTimeouts.add(new InFlightPacket(packetId, 5000L));
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(topic.toString(), qos, payload, packetId);
            this.mqttConnection.sendPublish(publishMsg);
            LOG.debug("Write direct to the peer, inflight slots: {}", (Object)this.inflightSlots.get());
            if (this.inflightSlots.get() == 0) {
                this.mqttConnection.flush();
            }
        } else {
            SessionRegistry.PublishedMessage msg = new SessionRegistry.PublishedMessage(topic, qos, payload);
            msg.retain();
            this.sessionQueue.add(msg);
            LOG.debug("Enqueue to peer session");
        }
    }

    private void sendPublishQos2(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (this.canSkipQueue()) {
            this.inflightSlots.decrementAndGet();
            int packetId = this.mqttConnection.nextPacketId();
            payload.retain();
            SessionRegistry.EnqueuedMessage old = this.inflightWindow.put(packetId, new SessionRegistry.PublishedMessage(topic, qos, payload));
            if (old != null) {
                old.release();
                this.inflightSlots.incrementAndGet();
            }
            this.inflightTimeouts.add(new InFlightPacket(packetId, 5000L));
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(topic.toString(), qos, payload, packetId);
            this.mqttConnection.sendPublish(publishMsg);
            this.drainQueueToConnection();
        } else {
            SessionRegistry.PublishedMessage msg = new SessionRegistry.PublishedMessage(topic, qos, payload);
            msg.retain();
            this.sessionQueue.add(msg);
        }
    }

    private boolean canSkipQueue() {
        return this.sessionQueue.isEmpty() && this.inflightSlots.get() > 0 && this.connected() && this.mqttConnection.channel.isWritable();
    }

    private boolean inflighHasSlotsAndConnectionIsUp() {
        return this.inflightSlots.get() > 0 && this.connected() && this.mqttConnection.channel.isWritable();
    }

    void pubAckReceived(int ackPacketId) {
        SessionRegistry.EnqueuedMessage removed = this.inflightWindow.remove(ackPacketId);
        if (removed == null) {
            LOG.warn("Received a PUBACK with not matching packetId");
            return;
        }
        removed.release();
        this.inflightSlots.incrementAndGet();
        this.drainQueueToConnection();
    }

    public void flushAllQueuedMessages() {
        this.drainQueueToConnection();
    }

    public void resendInflightNotAcked() {
        ArrayList<InFlightPacket> expired = new ArrayList<InFlightPacket>(10);
        this.inflightTimeouts.drainTo(expired);
        this.debugLogPacketIds(expired);
        for (InFlightPacket notAckPacketId : expired) {
            SessionRegistry.EnqueuedMessage msg = this.inflightWindow.get(notAckPacketId.packetId);
            if (msg == null) continue;
            if (msg instanceof SessionRegistry.PubRelMarker) {
                MqttMessage pubRel = MQTTConnection.pubrel(notAckPacketId.packetId);
                this.inflightTimeouts.add(new InFlightPacket(notAckPacketId.packetId, 5000L));
                this.mqttConnection.sendIfWritableElseDrop(pubRel);
                continue;
            }
            SessionRegistry.PublishedMessage pubMsg = (SessionRegistry.PublishedMessage)msg;
            Topic topic = pubMsg.topic;
            MqttQoS qos = pubMsg.publishingQos;
            ByteBuf payload = pubMsg.payload;
            ByteBuf copiedPayload = payload.retainedDuplicate();
            MqttPublishMessage publishMsg = this.publishNotRetainedDuplicated(notAckPacketId, topic, qos, copiedPayload);
            this.inflightTimeouts.add(new InFlightPacket(notAckPacketId.packetId, 5000L));
            this.mqttConnection.sendPublish(publishMsg);
        }
    }

    private void debugLogPacketIds(Collection<InFlightPacket> expired) {
        if (!LOG.isDebugEnabled() || expired.isEmpty()) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        for (InFlightPacket packet : expired) {
            sb.append(packet.packetId).append(", ");
        }
        LOG.debug("Resending {} in flight packets [{}]", (Object)expired.size(), (Object)sb);
    }

    private MqttPublishMessage publishNotRetainedDuplicated(InFlightPacket notAckPacketId, Topic topic, MqttQoS qos, ByteBuf payload) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, true, qos, false, 0);
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic.toString(), notAckPacketId.packetId);
        return new MqttPublishMessage(fixedHeader, varHeader, payload);
    }

    private void drainQueueToConnection() {
        while (!this.sessionQueue.isEmpty() && this.inflighHasSlotsAndConnectionIsUp()) {
            SessionRegistry.EnqueuedMessage msg = this.sessionQueue.poll();
            if (msg == null) {
                return;
            }
            this.inflightSlots.decrementAndGet();
            int sendPacketId = this.mqttConnection.nextPacketId();
            SessionRegistry.EnqueuedMessage old = this.inflightWindow.put(sendPacketId, msg);
            if (old != null) {
                old.release();
                this.inflightSlots.incrementAndGet();
            }
            this.inflightTimeouts.add(new InFlightPacket(sendPacketId, 5000L));
            SessionRegistry.PublishedMessage msgPub = (SessionRegistry.PublishedMessage)msg;
            MqttPublishMessage publishMsg = MQTTConnection.notRetainedPublishWithMessageId(msgPub.topic.toString(), msgPub.publishingQos, msgPub.payload, sendPacketId);
            this.mqttConnection.sendPublish(publishMsg);
        }
    }

    public void writabilityChanged() {
        this.drainQueueToConnection();
    }

    public void sendQueuedMessagesWhileOffline() {
        LOG.trace("Republishing all saved messages for session {}", (Object)this);
        this.drainQueueToConnection();
    }

    void sendRetainedPublishOnSessionAtQos(Topic topic, MqttQoS qos, ByteBuf payload) {
        if (qos != MqttQoS.AT_MOST_ONCE) {
            this.mqttConnection.sendPublishRetainedWithPacketId(topic, qos, payload);
        } else {
            this.mqttConnection.sendPublishRetainedQos0(topic, qos, payload);
        }
    }

    public void receivedPublishQos2(int messageID, MqttPublishMessage msg) {
        ReferenceCountUtil.retain((Object)msg);
        MqttPublishMessage old = this.qos2Receiving.put(messageID, msg);
        ReferenceCountUtil.release((Object)old);
        this.mqttConnection.sendPublishReceived(messageID);
    }

    public void receivedPubRelQos2(int messageID) {
        MqttPublishMessage removedMsg = this.qos2Receiving.remove(messageID);
        ReferenceCountUtil.release((Object)removedMsg);
    }

    Optional<InetSocketAddress> remoteAddress() {
        if (this.connected()) {
            return Optional.of(this.mqttConnection.remoteAddress());
        }
        return Optional.empty();
    }

    public String toString() {
        return "Session{clientId='" + this.clientId + '\'' + ", clean=" + this.clean + ", status=" + this.status + ", inflightSlots=" + this.inflightSlots + '}';
    }

    static final class Will {
        final String topic;
        final ByteBuf payload;
        final MqttQoS qos;
        final boolean retained;

        Will(String topic, ByteBuf payload, MqttQoS qos, boolean retained) {
            this.topic = topic;
            this.payload = payload;
            this.qos = qos;
            this.retained = retained;
        }
    }

    static enum SessionStatus {
        CONNECTED,
        CONNECTING,
        DISCONNECTING,
        DISCONNECTED;

    }

    static class InFlightPacket
    implements Delayed {
        final int packetId;
        private long startTime;

        InFlightPacket(int packetId, long delayInMilliseconds) {
            this.packetId = packetId;
            this.startTime = System.currentTimeMillis() + delayInMilliseconds;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long diff = this.startTime - System.currentTimeMillis();
            return unit.convert(diff, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            if (this.startTime - ((InFlightPacket)o).startTime == 0L) {
                return 0;
            }
            if (this.startTime - ((InFlightPacket)o).startTime > 0L) {
                return 1;
            }
            return -1;
        }
    }
}

