/*
 * Decompiled with CFR 0.152.
 */
package de.otto.synapse.aws.s3;

import de.otto.edison.aws.s3.S3Service;
import de.otto.synapse.aws.s3.SnapshotUtils;
import de.otto.synapse.aws.s3.TempFileHelper;
import de.otto.synapse.configuration.aws.SnapshotProperties;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.s3.model.S3Object;

public class SnapshotReadService {
    private static final Logger LOG = LoggerFactory.getLogger(SnapshotReadService.class);
    private static final int ONE_MB = 0x100000;
    private final S3Service s3Service;
    private final String snapshotBucketName;
    private final TempFileHelper tempFileHelper;
    private File forcedSnapshotFile = null;

    public SnapshotReadService(SnapshotProperties properties, S3Service s3Service) {
        this(properties, s3Service, new TempFileHelper());
    }

    public SnapshotReadService(SnapshotProperties properties, S3Service s3Service, TempFileHelper tempFileHelper) {
        this.s3Service = s3Service;
        this.snapshotBucketName = properties.getBucketName();
        this.tempFileHelper = tempFileHelper;
    }

    public void setSnapshotFile(File file) {
        Objects.requireNonNull(file, "file must not be null");
        if (!file.exists() || !file.canRead()) {
            throw new IllegalArgumentException("snapshot file does not exists or is not readable");
        }
        this.forcedSnapshotFile = file;
    }

    public Optional<File> retrieveLatestSnapshot(String channelName) {
        if (this.forcedSnapshotFile != null) {
            LOG.info("Use local Snapshot file: {}", (Object)this.forcedSnapshotFile);
            return Optional.of(this.forcedSnapshotFile);
        }
        LOG.info("Start downloading snapshot from S3");
        this.infoDiskUsage();
        Optional<File> latestSnapshot = this.getLatestSnapshot(channelName);
        if (latestSnapshot.isPresent()) {
            LOG.info("Finished downloading snapshot {}", (Object)latestSnapshot.get().getName());
            this.infoDiskUsage();
        } else {
            LOG.warn("No snapshot found.");
        }
        return latestSnapshot;
    }

    Optional<File> getLatestSnapshot(String channelName) {
        Optional<S3Object> s3Object = this.fetchSnapshotMetadataFromS3(this.snapshotBucketName, channelName);
        if (s3Object.isPresent()) {
            String latestSnapshotKey = s3Object.get().key();
            Path snapshotFile = this.tempFileHelper.getTempFile(latestSnapshotKey);
            if (this.tempFileHelper.existsAndHasSize(snapshotFile, s3Object.get().size())) {
                LOG.info("Snapshot on disk is same as in S3, keep it and use it: {}", (Object)snapshotFile.toAbsolutePath().toString());
                return Optional.of(snapshotFile.toFile());
            }
            this.tempFileHelper.removeTempFiles("*-snapshot-*.json.zip");
            LOG.info("Downloading snapshot file to {}", (Object)snapshotFile.getFileName().toAbsolutePath().toString());
            if (this.s3Service.download(this.snapshotBucketName, latestSnapshotKey, snapshotFile)) {
                return Optional.of(snapshotFile.toFile());
            }
            return Optional.empty();
        }
        return Optional.empty();
    }

    Optional<S3Object> fetchSnapshotMetadataFromS3(String bucketName, String channelName) {
        return this.s3Service.listAll(bucketName).stream().filter(o -> o.key().startsWith(SnapshotUtils.getSnapshotFileNamePrefix(channelName))).filter(o -> o.key().endsWith(".json.zip")).sorted(Comparator.comparing(S3Object::lastModified, Comparator.reverseOrder())).findFirst();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void infoDiskUsage() {
        File file = null;
        try {
            file = File.createTempFile("tempFileForDiskUsage", ".txt");
            float usableSpace = (float)file.getUsableSpace() / 1024.0f / 1024.0f / 1024.0f;
            float freeSpace = (float)file.getFreeSpace() / 1024.0f / 1024.0f / 1024.0f;
            LOG.info(String.format("Available DiskSpace: usable %.3f GB / free %.3f GB", Float.valueOf(usableSpace), Float.valueOf(freeSpace)));
            String tempDirContent = Files.list(Paths.get(System.getProperty("java.io.tmpdir"), new String[0])).filter(path -> path.toFile().isFile()).filter(path -> path.toFile().length() > 0x100000L).map(path -> String.format("%s %dmb", path.toString(), path.toFile().length() / 0x100000L)).collect(Collectors.joining("\n"));
            LOG.info("files in /tmp > 1mb: \n {}", (Object)tempDirContent);
        }
        catch (IOException e) {
            LOG.info("Error calculating disk usage: " + e.getMessage());
        }
        finally {
            try {
                if (file != null) {
                    LOG.info("delete file {}", (Object)file.toPath().toString());
                    Files.delete(file.toPath());
                }
            }
            catch (IOException e) {
                LOG.error("Error deleting temp file while calculating disk usage:" + e.getMessage());
            }
        }
    }

    public void deleteOlderSnapshots(String channelName) {
        List oldestFiles;
        if (this.forcedSnapshotFile != null) {
            LOG.info("Skip deleting local Snapshot file: {}", (Object)this.forcedSnapshotFile);
            return;
        }
        String snapshotFileNamePrefix = SnapshotUtils.getSnapshotFileNamePrefix(channelName);
        String snapshotFileSuffix = ".json.zip";
        try {
            oldestFiles = Files.find(Paths.get(this.tempFileHelper.getTempDir(), new String[0]), 1, (path, basicFileAttributes) -> path.getFileName().toString().startsWith(snapshotFileNamePrefix) && path.getFileName().toString().endsWith(snapshotFileSuffix), new FileVisitOption[0]).sorted((path1, path2) -> (int)(path2.toFile().lastModified() - path1.toFile().lastModified())).map(Path::toFile).collect(Collectors.toList());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        if (oldestFiles.size() > 1) {
            oldestFiles.subList(1, oldestFiles.size()).forEach(this::deleteSnapshotFile);
        }
    }

    private void deleteSnapshotFile(File snapshotFile) {
        boolean success = snapshotFile.delete();
        if (success) {
            LOG.info("deleted {}", (Object)snapshotFile.getName());
        } else {
            LOG.warn("deletion of {} failed", (Object)snapshotFile.getName());
        }
    }
}

