/*
 * Decompiled with CFR 0.152.
 */
package de.otto.synapse.compaction.s3;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.annotations.VisibleForTesting;
import de.otto.synapse.channel.ChannelPosition;
import de.otto.synapse.channel.StartFrom;
import de.otto.synapse.compaction.s3.SnapshotFileHelper;
import de.otto.synapse.configuration.aws.SnapshotProperties;
import de.otto.synapse.helper.s3.S3Helper;
import de.otto.synapse.logging.ProgressLogger;
import de.otto.synapse.state.StateRepository;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import software.amazon.awssdk.services.s3.S3Client;

public class SnapshotWriteService {
    private static final Logger LOG = LoggerFactory.getLogger(SnapshotWriteService.class);
    private static final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH-mmX").withZone(ZoneOffset.UTC);
    private static final String ZIP_ENTRY = "data";
    private static final String DATA_FIELD_NAME = "data";
    private static final String START_SEQUENCE_NUMBERS_FIELD_NAME = "startSequenceNumbers";
    private static final String SHARD_FIELD_NAME = "shard";
    private static final String SEQUENCE_NUMBER_FIELD_NAME = "sequenceNumber";
    private static final int NUM_SNAPSHOTS_TO_KEEP = 6;
    private final S3Helper s3Helper;
    private final String snapshotBucketName;
    private final JsonFactory jsonFactory = new JsonFactory();
    private final Marker marker;

    public SnapshotWriteService(S3Client s3Client, SnapshotProperties properties) {
        this(s3Client, properties, null);
    }

    public SnapshotWriteService(S3Client s3Client, SnapshotProperties properties, Marker marker) {
        this.s3Helper = new S3Helper(s3Client);
        this.snapshotBucketName = properties.getBucketName();
        this.marker = marker;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String writeSnapshot(String channelName, ChannelPosition position, StateRepository<String> stateRepository) throws IOException {
        File snapshotFile = null;
        try {
            LOG.info(this.marker, "Start creating new snapshot");
            snapshotFile = this.createSnapshot(channelName, position, stateRepository);
            LOG.info(this.marker, "Finished creating snapshot file: {}", (Object)snapshotFile.getAbsolutePath());
            this.uploadSnapshot(this.snapshotBucketName, snapshotFile);
            LOG.info(this.marker, "Finished uploading snapshot file to s3");
            this.deleteOlderSnapshots(channelName);
        }
        finally {
            if (snapshotFile != null) {
                LOG.info(this.marker, "delete file {}", (Object)snapshotFile.toPath().toString());
                this.deleteFile(snapshotFile);
            }
        }
        return snapshotFile.getName();
    }

    @VisibleForTesting
    File createSnapshot(String channelName, ChannelPosition currentChannelPosition, StateRepository<String> stateRepository) throws IOException {
        File snapshotFile = SnapshotWriteService.createSnapshotFile(channelName);
        try (FileOutputStream fos = new FileOutputStream(snapshotFile);
             BufferedOutputStream bos = new BufferedOutputStream(fos);
             ZipOutputStream zipOutputStream = new ZipOutputStream(bos);){
            ZipEntry zipEntry = new ZipEntry("data");
            zipEntry.setMethod(8);
            zipOutputStream.putNextEntry(zipEntry);
            JsonGenerator jGenerator = this.jsonFactory.createGenerator((OutputStream)zipOutputStream, JsonEncoding.UTF8);
            jGenerator.writeStartObject();
            this.writeSequenceNumbers(currentChannelPosition, jGenerator);
            jGenerator.writeArrayFieldStart("data");
            ProgressLogger processedLogger = new ProgressLogger(LOG, stateRepository.size(), this.marker);
            stateRepository.consumeAll((key, entry) -> {
                try {
                    processedLogger.incrementAndLog();
                    if (!"".equals(entry)) {
                        jGenerator.writeStartObject();
                        jGenerator.writeStringField(key, entry);
                        jGenerator.writeEndObject();
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
            jGenerator.writeEndArray();
            jGenerator.writeEndObject();
            jGenerator.flush();
            zipOutputStream.closeEntry();
        }
        catch (Exception e) {
            LOG.info(this.marker, "delete file {}", (Object)snapshotFile.toPath().toString());
            this.deleteFile(snapshotFile);
            throw e;
        }
        finally {
            System.gc();
        }
        return snapshotFile;
    }

    private void deleteOlderSnapshots(String channelName) {
        String snapshotFileNamePrefix = SnapshotFileHelper.getSnapshotFileNamePrefix(channelName);
        String snapshotFileSuffix = ".json.zip";
        BiPredicate<Path, BasicFileAttributes> matchSnapshotFilePattern = (path, basicFileAttributes) -> path.getFileName().toString().startsWith(snapshotFileNamePrefix) && path.getFileName().toString().endsWith(snapshotFileSuffix);
        try (Stream<Path> pathStream = Files.find(Paths.get(SnapshotFileHelper.getTempDir(), new String[0]), 1, matchSnapshotFilePattern, new FileVisitOption[0]);){
            List oldestFiles = pathStream.sorted((path1, path2) -> (int)(path2.toFile().lastModified() - path1.toFile().lastModified())).map(Path::toFile).collect(Collectors.toList());
            if (oldestFiles.size() > 6) {
                oldestFiles.subList(6, oldestFiles.size()).forEach(this::deleteSnapshotFile);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void deleteSnapshotFile(File snapshotFile) {
        boolean success = snapshotFile.delete();
        if (success) {
            LOG.info(this.marker, "deleted {}", (Object)snapshotFile.getName());
        } else {
            LOG.warn(this.marker, "deletion of {} failed", (Object)snapshotFile.getName());
        }
    }

    private void deleteFile(File file) {
        boolean success;
        if (file != null && !(success = file.delete())) {
            LOG.error(this.marker, "failed to delete snapshot {}", (Object)file.getName());
        }
    }

    private static File createSnapshotFile(String channelName) throws IOException {
        return File.createTempFile(String.format("%s%s-", SnapshotFileHelper.getSnapshotFileNamePrefix(channelName), dateTimeFormatter.format(Instant.now())), ".json.zip");
    }

    private void uploadSnapshot(String bucketName, File snapshotFile) {
        this.s3Helper.upload(bucketName, snapshotFile);
    }

    private void writeSequenceNumbers(ChannelPosition currentChannelPosition, JsonGenerator jGenerator) throws IOException {
        jGenerator.writeArrayFieldStart(START_SEQUENCE_NUMBERS_FIELD_NAME);
        currentChannelPosition.shards().forEach(shardName -> {
            try {
                jGenerator.writeStartObject();
                jGenerator.writeStringField(SHARD_FIELD_NAME, shardName);
                if (currentChannelPosition.shard(shardName).startFrom() == StartFrom.HORIZON) {
                    jGenerator.writeStringField(SEQUENCE_NUMBER_FIELD_NAME, "0");
                } else {
                    jGenerator.writeStringField(SEQUENCE_NUMBER_FIELD_NAME, currentChannelPosition.shard(shardName).position());
                }
                jGenerator.writeEndObject();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
        jGenerator.writeEndArray();
    }
}

