/*
 * Decompiled with CFR 0.152.
 */
package de.otto.synapse.helper.s3;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.Bucket;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CreateBucketRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadResponse;
import software.amazon.awssdk.services.s3.model.Delete;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;

public class S3Helper {
    private static final Logger LOG = LoggerFactory.getLogger(S3Helper.class);
    private static final long AWS_S3_FILE_LIMIT_OF_5GB_IN_BYTES = 0x140000000L;
    public static final int PART_SIZE_IN_BYTES = 0x6400000;
    private final S3Client s3Client;

    public S3Helper(S3Client s3Client) {
        this.s3Client = s3Client;
    }

    public void createBucket(String bucketName) {
        if (!this.listBucketNames().contains(bucketName)) {
            this.s3Client.createBucket((CreateBucketRequest)CreateBucketRequest.builder().bucket(bucketName).build());
        }
    }

    public List<String> listBucketNames() {
        return this.s3Client.listBuckets().buckets().stream().map(Bucket::name).collect(Collectors.toList());
    }

    public void upload(String bucketName, File file) {
        if (file.length() >= 0x140000000L) {
            this.uploadAsMultipart(bucketName, file, 0x6400000);
            return;
        }
        try (FileInputStream fis = new FileInputStream(file);){
            PutObjectResponse putObjectResponse = this.s3Client.putObject((PutObjectRequest)PutObjectRequest.builder().bucket(bucketName).key(file.getName()).build(), RequestBody.fromInputStream((InputStream)fis, (long)file.length()));
            LOG.debug("upload {} to bucket {}: {}", new Object[]{file.getName(), bucketName, putObjectResponse.toString()});
        }
        catch (IOException e) {
            LOG.error("Error while uploading {} to bucket {}", new Object[]{file.getName(), bucketName, e});
        }
    }

    public void uploadAsMultipart(String bucketName, File file, int partSizeInBytes) {
        String filename = file.getName();
        CreateMultipartUploadRequest createMultipartUploadRequest = (CreateMultipartUploadRequest)CreateMultipartUploadRequest.builder().bucket(bucketName).key(filename).build();
        CreateMultipartUploadResponse response = this.s3Client.createMultipartUpload(createMultipartUploadRequest);
        String uploadId = response.uploadId();
        ArrayList<CompletedPart> parts = new ArrayList<CompletedPart>();
        try (RandomAccessFile randomAccessFile = new RandomAccessFile(file, "r");
             FileChannel inputChannel = randomAccessFile.getChannel();){
            ByteBuffer byteBuffer = ByteBuffer.allocate(partSizeInBytes);
            int partNumber = 0;
            while (inputChannel.read(byteBuffer) > 0) {
                byteBuffer.flip();
                UploadPartRequest uploadPartRequest = (UploadPartRequest)UploadPartRequest.builder().bucket(bucketName).key(filename).uploadId(uploadId).partNumber(Integer.valueOf(++partNumber)).build();
                String etag = this.s3Client.uploadPart(uploadPartRequest, RequestBody.fromByteBuffer((ByteBuffer)byteBuffer)).eTag();
                CompletedPart part = (CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(partNumber)).eTag(etag).build();
                parts.add(part);
            }
            CompletedMultipartUpload completedMultipartUpload = (CompletedMultipartUpload)CompletedMultipartUpload.builder().parts(parts).build();
            CompleteMultipartUploadRequest completeMultipartUploadRequest = (CompleteMultipartUploadRequest)CompleteMultipartUploadRequest.builder().bucket(bucketName).key(filename).uploadId(uploadId).multipartUpload(completedMultipartUpload).build();
            this.s3Client.completeMultipartUpload(completeMultipartUploadRequest);
        }
        catch (IOException e) {
            LOG.error("Something went wrong during multipart upload!!!", (Throwable)e);
        }
    }

    public boolean download(String bucketName, String fileName, Path destination) {
        try {
            if (Files.exists(destination, new LinkOption[0])) {
                Files.delete(destination);
            }
        }
        catch (IOException e) {
            LOG.error("could not delete temp snapshotfile {}: {}", (Object)destination.toString(), (Object)e.getMessage());
            return false;
        }
        try {
            GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().bucket(bucketName).key(fileName).build();
            GetObjectResponse getObjectResponse = this.s3Client.getObject(request, destination);
            LOG.debug("download {} from bucket {}: {}", new Object[]{fileName, bucketName, getObjectResponse.toString()});
            return true;
        }
        catch (RuntimeException e) {
            LOG.error("Failed to download {} from bucket {}: {}", new Object[]{fileName, bucketName, e.getMessage()});
            return false;
        }
    }

    public void deleteAllObjectsInBucket(String bucketName) {
        try {
            LOG.debug("deleting all objects in bucket {}", (Object)bucketName);
            this.deleteAllObjectsWithPrefixInBucket(bucketName, "");
            LOG.debug("files in bucket: {}", this.listAllFiles(bucketName));
        }
        catch (RuntimeException e) {
            LOG.error("Error while deleting files from bucket: " + e.getMessage(), (Throwable)e);
        }
    }

    public void deleteAllObjectsWithPrefixInBucket(String bucketName, String prefix) {
        ListObjectsV2Response response;
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(bucketName).build();
        do {
            if ((response = this.s3Client.listObjectsV2(request)).hasContents()) {
                DeleteObjectsRequest deleteObjectsRequest = (DeleteObjectsRequest)DeleteObjectsRequest.builder().bucket(bucketName).delete((Delete)Delete.builder().objects(this.convertS3ObjectsToObjectIdentifiers(response, prefix)).build()).build();
                DeleteObjectsResponse deleteObjectsResponse = this.s3Client.deleteObjects(deleteObjectsRequest);
                LOG.debug("Deleted {} objects in bucket {} with prefix {}: {}", new Object[]{deleteObjectsResponse.deleted().size(), bucketName, prefix, deleteObjectsResponse});
            } else {
                LOG.debug("deleteAllObjectsWithPrefixInBucket listObjects found no keys in bucket {} with prefix {}: {}", new Object[]{bucketName, prefix, response});
            }
            String token = response.nextContinuationToken();
            request = (ListObjectsV2Request)request.toBuilder().continuationToken(token).build();
        } while (response.isTruncated().booleanValue());
    }

    public List<String> listAllFiles(String bucketName) {
        return this.listAll(bucketName).stream().map(S3Object::key).collect(Collectors.toList());
    }

    public List<S3Object> listAll(String bucketName) {
        ListObjectsV2Response response;
        ListObjectsV2Request request = (ListObjectsV2Request)ListObjectsV2Request.builder().bucket(bucketName).build();
        ArrayList s3Objects = new ArrayList();
        do {
            if ((response = this.s3Client.listObjectsV2(request)).hasContents()) {
                s3Objects.addAll(response.contents());
            }
            String token = response.nextContinuationToken();
            request = (ListObjectsV2Request)request.toBuilder().continuationToken(token).build();
        } while (response.isTruncated().booleanValue());
        return Collections.unmodifiableList(s3Objects);
    }

    private List<ObjectIdentifier> convertS3ObjectsToObjectIdentifiers(ListObjectsV2Response listObjectsV2Response, String prefix) {
        return listObjectsV2Response.contents().stream().filter(o -> o.key() != null && o.key().startsWith(prefix)).map(o -> (ObjectIdentifier)ObjectIdentifier.builder().key(o.key()).build()).collect(Collectors.toList());
    }
}

