/*
 * Decompiled with CFR 0.152.
 */
package de.pauleff.formats.snbt;

import de.pauleff.formats.snbt.SNBTException;

public class SNBTTokenizer {
    private final String input;
    private final int length;
    private int position;

    public SNBTTokenizer(String input) {
        this.input = input != null ? input : "";
        this.position = 0;
        this.length = this.input.length();
    }

    public Token nextToken() throws SNBTException {
        while (this.position < this.length && Character.isWhitespace(this.input.charAt(this.position))) {
            ++this.position;
        }
        if (this.position >= this.length) {
            return new Token(TokenType.EOF, "", this.position);
        }
        char current = this.input.charAt(this.position);
        int tokenStart = this.position++;
        switch (current) {
            case '{': {
                return new Token(TokenType.LBRACE, "{", tokenStart);
            }
            case '}': {
                ++this.position;
                return new Token(TokenType.RBRACE, "}", tokenStart);
            }
            case '[': {
                ++this.position;
                return new Token(TokenType.LBRACKET, "[", tokenStart);
            }
            case ']': {
                ++this.position;
                return new Token(TokenType.RBRACKET, "]", tokenStart);
            }
            case ',': {
                ++this.position;
                return new Token(TokenType.COMMA, ",", tokenStart);
            }
            case ':': {
                ++this.position;
                return new Token(TokenType.COLON, ":", tokenStart);
            }
            case ';': {
                ++this.position;
                return new Token(TokenType.SEMICOLON, ";", tokenStart);
            }
            case '\"': {
                return this.parseQuotedString('\"', tokenStart);
            }
            case '\'': {
                return this.parseQuotedString('\'', tokenStart);
            }
        }
        if (Character.isDigit(current) || current == '-' || current == '+') {
            return this.parseNumber(tokenStart);
        }
        if (Character.isLetter(current) || current == '_') {
            return this.parseUnquotedString(tokenStart);
        }
        ++this.position;
        return new Token(TokenType.UNKNOWN, String.valueOf(current), tokenStart);
    }

    private Token parseQuotedString(char quote, int tokenStart) throws SNBTException {
        StringBuilder sb = new StringBuilder();
        ++this.position;
        while (this.position < this.length) {
            char c = this.input.charAt(this.position);
            if (c == quote) {
                ++this.position;
                return new Token(TokenType.STRING, sb.toString(), tokenStart);
            }
            if (c == '\\') {
                ++this.position;
                if (this.position >= this.length) {
                    throw new SNBTException("Unterminated escape sequence", this.position, this.input);
                }
                char escaped = this.input.charAt(this.position);
                switch (escaped) {
                    case '\"': {
                        sb.append('\"');
                        break;
                    }
                    case '\'': {
                        sb.append('\'');
                        break;
                    }
                    case '\\': {
                        sb.append('\\');
                        break;
                    }
                    case 'n': {
                        sb.append('\n');
                        break;
                    }
                    case 't': {
                        sb.append('\t');
                        break;
                    }
                    case 'r': {
                        sb.append('\r');
                        break;
                    }
                    case 'b': {
                        sb.append('\b');
                        break;
                    }
                    case 'f': {
                        sb.append('\f');
                        break;
                    }
                    case 'u': {
                        if (this.position + 4 >= this.length) {
                            throw new SNBTException("Incomplete unicode escape sequence", this.position, this.input);
                        }
                        try {
                            String hex = this.input.substring(this.position + 1, this.position + 5);
                            int codePoint = Integer.parseInt(hex, 16);
                            sb.append((char)codePoint);
                            this.position += 4;
                            break;
                        }
                        catch (NumberFormatException e) {
                            throw new SNBTException("Invalid unicode escape sequence", this.position, this.input, e);
                        }
                    }
                    default: {
                        sb.append(escaped);
                    }
                }
                ++this.position;
                continue;
            }
            sb.append(c);
            ++this.position;
        }
        throw new SNBTException("Unterminated string", tokenStart, this.input);
    }

    private Token parseNumber(int tokenStart) throws SNBTException {
        String numberStr;
        char suffix;
        StringBuilder sb = new StringBuilder();
        if (this.position < this.length && (this.input.charAt(this.position) == '-' || this.input.charAt(this.position) == '+')) {
            sb.append(this.input.charAt(this.position));
            ++this.position;
        }
        boolean hasDecimal = false;
        while (this.position < this.length) {
            char c = this.input.charAt(this.position);
            if (Character.isDigit(c)) {
                sb.append(c);
                ++this.position;
                continue;
            }
            if (c != '.' || hasDecimal) break;
            sb.append(c);
            hasDecimal = true;
            ++this.position;
        }
        if (this.position < this.length && this.isTypeSuffix(suffix = this.input.charAt(this.position))) {
            sb.append(suffix);
            ++this.position;
        }
        if ((numberStr = sb.toString()).isEmpty() || numberStr.equals("-") || numberStr.equals("+")) {
            throw new SNBTException("Invalid number format", tokenStart, this.input);
        }
        return new Token(TokenType.NUMBER, numberStr, tokenStart);
    }

    private Token parseUnquotedString(int tokenStart) {
        char c;
        StringBuilder sb = new StringBuilder();
        while (this.position < this.length && (Character.isLetterOrDigit(c = this.input.charAt(this.position)) || c == '_' || c == '.' || c == '-')) {
            sb.append(c);
            ++this.position;
        }
        String value = sb.toString();
        if (value.length() == 1 && (value.equals("B") || value.equals("I") || value.equals("L"))) {
            int nextPos;
            for (nextPos = this.position; nextPos < this.length && Character.isWhitespace(this.input.charAt(nextPos)); ++nextPos) {
            }
            if (nextPos < this.length && this.input.charAt(nextPos) == ';') {
                return new Token(TokenType.ARRAY_PREFIX, value, tokenStart);
            }
        }
        return new Token(TokenType.STRING, value, tokenStart);
    }

    private boolean isTypeSuffix(char c) {
        return c == 'b' || c == 'B' || c == 's' || c == 'S' || c == 'l' || c == 'L' || c == 'f' || c == 'F' || c == 'd' || c == 'D';
    }

    public String getInput() {
        return this.input;
    }

    public static class Token {
        private final TokenType type;
        private final String value;
        private final int position;

        public Token(TokenType type, String value, int position) {
            this.type = type;
            this.value = value;
            this.position = position;
        }

        public TokenType getType() {
            return this.type;
        }

        public String getValue() {
            return this.value;
        }

        public int getPosition() {
            return this.position;
        }

        public String toString() {
            return String.format("Token{%s, '%s', pos=%d}", new Object[]{this.type, this.value, this.position});
        }
    }

    public static enum TokenType {
        LBRACE,
        RBRACE,
        LBRACKET,
        RBRACKET,
        COMMA,
        COLON,
        SEMICOLON,
        STRING,
        NUMBER,
        ARRAY_PREFIX,
        EOF,
        UNKNOWN;

    }
}

