package de.pfabulist.kleinod.errors;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

/***** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2014, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***** END LICENSE BLOCK *****/

                      
/**
 * Wrapper for checked exceptions to be used in methods without throws
 * good for closures and slim code
 */
public class Unchecked {


    private Unchecked() {}

    public static RuntimeException runtime( Exception e ) {
        if ( e instanceof RuntimeException ) {
            return (RuntimeException)e;
        }

        if ( e instanceof FileNotFoundException ) {
            return new FileNotFoundRuntimeException((FileNotFoundException)e);
        }

        if ( e instanceof IOException) {
            return new IORuntimeException((IOException)e);
        }

        return new RuntimeException(e);
    }

    public static <A> Predicate<A> predicate( PredicateE<A> f ) {
        return (A a) -> {
            try {
                return f.call(a);
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static <A,B,R> BiFunction<A,B,R> biFunction( BiFunctionE<A,B,R> f ) {
        return (A a, B b) -> {
            try {
                return f.apply(a,b);
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static <A,R> Function<A,R> function( FunctionE<A, R> f ) {
        return (A a ) -> {
            try {
                return f.apply(a);
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static <A> Consumer<A> consumer( ConsumerE<A> f ) {
        return (A a ) -> {
            try {
                f.accept(a);
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static Runnable runnable( RunnableE f ) {
        return () -> {
            try {
                f.run();
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static <A> Supplier<A> supplier( SupplierE<A> f ) {
        return () -> {
            try {
                return f.get();
            } catch (Exception e) {
                throw runtime( e );
            }
        };
    }

    public static <T> T val( SupplierE<T> f ) {
        try {
            return f.get();
        } catch (Exception e) {
            throw runtime(e);
        }
    }

    public static void statement( RunnableE f ) {
        try {
            f.run();
        } catch (Exception e) {
            throw runtime(e);
        }
    }
}
