package de.pfabulist.kleinod.collection;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;


/***** BEGIN LICENSE BLOCK *****
* BSD License (2 clause)
* Copyright (c) 2006 - 2014, Stephan Pfab
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*     * Redistributions of source code must retain the above copyright
*       notice, this list of conditions and the following disclaimer.
*     * Redistributions in binary form must reproduce the above copyright
*       notice, this list of conditions and the following disclaimer in the
*       documentation and/or other materials provided with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***** END LICENSE BLOCK *****/


public final class Iterators {

    private Iterators() {}

    /**
     * Split on standard word boundary, i.e. space tab line ends
     * @param str Any string
     * @return An Array of Strings
     */
    public static String[] words( String str ) {
        return str.split("( |\t|\r|\n)+");
    }


    /**
     * Allows for-loop iteration over the lines of a text
     * @param str Any String
     * @return An Array of Strings
     */
    public static String[] lines( String str ) {
        return str.split( "(\n|\r)+");
    }
//
//
//
//    public static <T> Iterable<T> iterate( final Enumeration<T> en ) {
//        return new Iterable<T>() {
//            @Override
//            public Iterator<T> iterator() {
//                return new Iterator<T>() {
//                    @Override
//                    public boolean hasNext() {
//                        return en.hasMoreElements();
//                    }
//
//                    @Override
//                    public T next() {
//                        return en.nextElement();
//                    }
//
//                    @Override
//                    public void remove() {
//                        throw new UnsupportedOperationException( "remove not supported" );
//                    }
//                };
//            }
//        };
//    }
//
//    public static <T> T getAt( Iterable<T> it , int i ) {
//        for ( T t : it ) {
//            if ( i == 0 ) {
//                return t;
//            }
//            i--;
//        }
//
//        return null;
//    }
//
//    // from an iterator back to an iterable so someone can apply iterate on it
//    public static <T> Iterable<T> iterable( final Iterator<T> it ) {
//        return new Iterable<T>() {
//            @Override
//            public Iterator<T> iterator() {
//                return it;
//            }
//        };
//    }
//
//
    public static <T> Iterator<T> makeImmutable( final Iterator<T> it )  {
        return new Iterator<T>() {
            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public T next() {
                return it.next();
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException( "remove not allowed on immutable iterator" );
            }
        };
    }

    public static <T> Iterator<T> closeable( final Iterator<T> it, final BooleanSupplier isClosed ) {
        return new Iterator<T>() {
            @Override
            public boolean hasNext() {
                if ( isClosed.getAsBoolean() ) {
                    return false;
                }
                return it.hasNext();
            }

            @Override
            public T next() {
                return it.next();
            }

            @Override
            public void remove() {
                it.remove();
            }
        };
    }

    public static <T> List<T> toList( Iterable<T> it ) {
        List<T> ret = new ArrayList<>();
        for ( T elem : it ) {
            ret.add(elem);
        }

        return ret;
    }

    public static <T> Stream<T> stream(Iterable<T> in) {
        return StreamSupport.stream(in.spliterator(), false);
    }

    public static <T> Stream<T> parallelStream(Iterable<T> in) {
        return StreamSupport.stream(in.spliterator(), true);
    }

}
