package de.pfabulist.kleinod.paths;

import de.pfabulist.unchecked.Filess;

import java.io.IOException;
import java.net.URI;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static de.pfabulist.unchecked.Unchecked.u;

/**
 * ** BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2015, Stephan Pfab
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * **** END LICENSE BLOCK ****
 */
public class Pathss {

    public static final String DEFAULT_FILESYSTEM_SCHEME = "file";

    private Pathss() {}


    public static Path getTmpDir(String prefix) {

        return FileSystems.getDefault().getPath(System.getProperty(
                        "java.io.tmpdir"),
                prefix + "-" + fsFriendlyName(Clock.systemUTC().instant()));
    }


    public static String fsFriendlyName(Instant inst) {
        return getDateTimeString( inst );//dtf.format(inst);
    }

    public static String getDateTimeString(final Instant instant) {
        DateTimeFormatter formatter =
                DateTimeFormatter.ofPattern("yyyy-MM-dd--HH-mm-ss--SSS").withZone(ZoneId.systemDefault());
        return formatter.format(instant);
    }


    public static FileSystem getOrCreate( URI uri, Map<String, ?> env ) {
        if ( uri.getScheme().equals( DEFAULT_FILESYSTEM_SCHEME )) {
            return FileSystems.getDefault();
        }
        try {
            return FileSystems.getFileSystem( uri );
        } catch( FileSystemNotFoundException exp  ) {
            try {
                return FileSystems.newFileSystem( uri, env );
            } catch (IOException e) {
                throw u( e );
            }
        }
    }

    public static Path get( URI uri, Map<String,?> env ) {
        try {
            // first try to get it the nice way, because getOrCreate might not work even if the FS is already there
            return Paths.get( uri );
        } catch( FileSystemNotFoundException e ) {
            getOrCreate( uri, env  ); // create the filesystem if necessary and possible
            return Paths.get(uri);
        }
    }

    public static Path getPath( String str, Map<String,?> env ) {
        URI uri = URI.create(str);

        if ( uri.getScheme() == null ) {
            throw new IllegalArgumentException( "not a uri with scheme " + uri );
        }
        
        return Pathss.get(uri, env);
    }

    public static void deleteRecursive(Path path) {
        if ( !Files.exists(path)) {
            return;
        }

        if ( !Files.isDirectory( path )) {
            Filess.delete( path );
            return;
        }

        final List<Path> toDel = new ArrayList<>();


        try {
            Files.walkFileTree(path, new SimpleFileVisitor<Path>() {

                @Override
                public FileVisitResult visitFile(Path file,
                                                 BasicFileAttributes attrs) throws IOException {
                    toDel.add( file );
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir,
                                                          IOException exc) throws IOException {
                    if (exc == null) {
                        toDel.add( dir );
                        return FileVisitResult.CONTINUE;
                    } else {
                        throw exc;
                    }
                }

            });
        } catch (IOException e) {
            throw u( e );
        }

        toDel.forEach(Filess::delete);
    }


    public static void copyRecursive( Path src, Path target, CopyOption ... options  ) {
        if ( !Files.exists(src)) {
            return;
        }

        if ( !Files.isDirectory( src )) {
            Filess.copy( src, target );
            return;
        }

        try {
            Files.walkFileTree(src, new SimpleFileVisitor<Path>() {

                @Override
                public FileVisitResult visitFile(Path file,
                                                 BasicFileAttributes attrs) throws IOException {
                    Path to = target.resolve( src.relativize( file ));
                    Files.copy( file, to );
                    Files.setLastModifiedTime( to, Files.getLastModifiedTime( src )); // TODO others

                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    Path to = target.resolve(src.relativize(dir));
                    Files.copy( dir, to );
                    Files.setLastModifiedTime( to, Files.getLastModifiedTime( src )); // TODO others

                    return FileVisitResult.CONTINUE;
                }

            });
        } catch (IOException e) {
            throw u( e );
        }

    }


    public static boolean isEmpty( Path dir ) {
        try( DirectoryStream<Path> stream = Files.newDirectoryStream(dir)) {
            return !stream.iterator().hasNext();
        } catch( IOException e ) {
            throw u( e );
        }
    }


    public static boolean isRoot( Path path ) {
        return path.isAbsolute() && path.getParent() == null;
    }

}
