package de.pfabulist.kleinod.collection;

import de.pfabulist.unchecked.functiontypes.SupplierE;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import static de.pfabulist.nonnullbydefault.NonnullCheck._nn;
import static de.pfabulist.unchecked.Unchecked.u;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

public class KleinodMap<K, V> {

    private final Map<K, V> inner;

    public KleinodMap( ConcurrentMap<K, V> inner ) {
        this.inner = inner;
    }

    public KleinodMap() {
        this( new ConcurrentHashMap<>() );
    }

    public Optional<V> get( K key ) {
        return Optional.ofNullable( inner.get( key ) );
    }

    public V getOrThrow( K key ) {
        return getOrThrow( key, new IllegalArgumentException( "no value for key " + key ) );
    }

    public V getOrThrow( K key, Exception ex ) {
        @Nullable V v = inner.get( key );
        if( v == null ) {
            throw u( ex );
        }

        return v;
    }

    public V getOrThrow( K key, SupplierE<Exception, Exception> ex ) {
        @Nullable V v = inner.get( key );
        if( v == null ) {
            throw u( ex._get() );
        }

        return v;
    }

    public boolean contains( K key ) {
        return inner.containsKey( key );
    }

    public KleinodMap<K, V> put( K key, V value ) {
        inner.put( key, value );
        return this;
    }

    public Collection<V> values() {
        return inner.values();
    }

    public Set<K> keys() {
        return inner.keySet();
    }

    // only non null values are allowed => _nn is right
    public V compteIfAbsent( K key, Function<K, V> comp ) {
        return _nn( inner.computeIfAbsent( key, comp ) );
    }
}
