package de.pfabulist.kleinod.nio;

import de.pfabulist.kleinod.os.OS;

import java.nio.charset.StandardCharsets;
import java.util.Locale;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

public class PathLimits {

    private final OS os;
    private final static String one4ByteUnicode = new String( Character.toChars( 0x10400 ) );     // NOPMD
    private final static String one2ByteUnicode = "\uaaaa";
    private final static String oneOSXCount = one2ByteUnicode;
    private final static String oneWindowsCount = one2ByteUnicode;
    private final static String oneUnixCount = "a";

    public PathLimits( OS os ) {
        this.os = os;
    }

    public int getMaxFilenameLength() {
        return 255;
    }

    public int filenameCount( String fname ) {
        if( os.isOSX() ) {
            return fname.length();
        }

        if( os.isUnix() ) {
            return getBytes( fname ).length;
        }

        if( os.isWindows() ) {
            return fname.codePointCount( 0, fname.length() );
        }

        throw new UnsupportedOperationException( "unsupported os " + os );
    }

//    public int pathCount( String fname ) {
//        if( os.isOSX() ) {
//            return getBytes( fname ).length; // UTF8 where filename is UTF16 codepoints
//        }
//
//        if( os.isUnix() ) {
//            return getBytes( fname ).length;
//        }
//
//        if( os.isWindows() ) {
//            return fname.length();
//        }
//
//        throw new UnsupportedOperationException( "unsupported os " + os );
//    }

    public enum CharType {
        UTF8,
        UNICODE1,
        FULL_UNICODE
    }

    public String getBigChar() {
        if( os.isOSX() ) {
            return oneOSXCount;
        }

        if( os.isWindows() ) {
            return oneWindowsCount;
        }

        return oneUnixCount;
    }

    public int getMaxPathLength() {
        if( os.isWindows() ) {
            //return 32767;  but java 8 impl says 32000
            return 32000;
        }

        if( os.isOSX() ) {
            return 1016; //900; ///1016;  // ????????????????????????????????????
        }

        if( os.isUnix() ) {
            return 4095; // linux max path - null terminator
        }

        throw new UnsupportedOperationException( "os not supported: " + os );
    }

    public boolean bigUnicodeCountsAs1() {
        return os.isWindows();
    }

    static byte[] getBytes( String str ) {
        return str.getBytes( StandardCharsets.UTF_8 );
    }

    public int compareTo( String a, String b ) {
        if( os.isOSX() || os.isWindows() ) {
            return a.toLowerCase( Locale.getDefault() ).compareTo( b.toLowerCase( Locale.getDefault() ) );
        }

        if( os.isUnix() ) {
            return a.compareTo( b );
        }

        throw new UnsupportedOperationException( "os not supported: " + os );
    }

    public int hash( String str ) {
        if( os.isOSX() || os.isWindows() ) {
            return str.toLowerCase( Locale.getDefault() ).hashCode();
        }

        if( os.isUnix() ) {
            return str.hashCode();
        }

        throw new UnsupportedOperationException( "os not supported: " + os );
    }

}
