package de.pfabulist.kleinod.collection;

import javax.annotation.Nullable;
import java.util.Iterator;
import java.util.function.BooleanSupplier;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

@SuppressWarnings( "PMD.TooManyMethods" ) // todo pmd bug or config? this class does not have many methods
public final class Iterators {

    private Iterators() {
    }

    /**
     * Split on standard word boundary, i.e. space tab line ends
     *
     * @param str Any string
     * @return An Array of Strings
     */
    public static String[] words( String str ) {
        return str.split( "( |\t|\r|\n)+" );
    }

    /**
     * Allows for-loop iteration over the lines of a text
     *
     * @param str Any String
     * @return An Array of Strings
     */
    public static String[] lines( String str ) {
        return str.split( "(\n|\r)+" );
    }

    public static <T> Iterator<T> makeImmutable( final Iterator<T> it ) {
        return new Iterator<T>() {
            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            @Nullable public T next() {
                return it.next();
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException( "remove not allowed on immutable iterator" );
            }
        };
    }

    public static <T> Iterator<T> closeable( final Iterator<T> it, final BooleanSupplier isClosed ) {
        return new Iterator<T>() {
            @Override
            public boolean hasNext() {
                if( isClosed.getAsBoolean() ) {
                    return false;
                }
                return it.hasNext();
            }

            @Override
            @Nullable public T next() {
                return it.next();
            }

            @Override
            public void remove() {
                it.remove();
            }
        };
    }

    public static <T> Stream<T> stream( Iterable<T> in ) {
        return StreamSupport.stream( in.spliterator(), false );
    }

    public static <T> Stream<T> parallelStream( Iterable<T> in ) {
        return StreamSupport.stream( in.spliterator(), true );
    }

}
