package de.pfabulist.kleinod.nio;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

public class BigIO {

    private final long upto;
    private final InputStream is = new InputStream() {

        private long count = 0;

        @Override
        public int read() throws IOException {
            if ( count >= upto ) {
                return -1;
            }

            count++;

            return extract0To255( count );
        }

    };


    private static abstract class BigOutputStream extends OutputStream {
        abstract long getCount();
    }

    private final BigOutputStream os = new BigOutputStream() {
        @Override
        long getCount() {
            return count;
        }

        private long count = 0;

        @Override
        public void write( int b ) throws IOException {
            if( count >= upto ) {
                throw new IOException( "more bytes than expected (>=" + upto );
            }

            count++;
            int val = getLower8Bit( b );
            int expected = extract0To255( count );
            if( expected != val ) {
                throw new IOException( "unexpected value. expected " + expected + " got " + val + " at position " + count );
            }
        }
    };

    public BigIO( long upto ) {
        this.upto = upto;
    }

    public InputStream getInputStream() {
        return is;
    }

    public OutputStream getOutputStream() {
        return os;
    }

    public boolean consumed() {
        return os.getCount() >= upto;
    }

    private int getLower8Bit( int b ) {
        return b >= 0 ? b : (b + 256);
    }

    private int extract0To255( long count ) {
        return (int)(count & 0xFF);
    }


}
