package de.pfabulist.kleinod.nio;

import de.pfabulist.frex.CharacterClass;
import de.pfabulist.frex.Frex;

import java.nio.file.InvalidPathException;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static de.pfabulist.frex.Frex.alpha;
import static de.pfabulist.frex.Frex.any;
import static de.pfabulist.frex.Frex.anyBut;
import static de.pfabulist.frex.Frex.or;
import static de.pfabulist.frex.Frex.startsWith;
import static de.pfabulist.frex.Frex.txt;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

public class PathSpecWindows implements PathSpec {

    private static final String UNCPATH = "uncpath";
    private static final String UNCSERVER = "uncserver";
    private static final String UNCSHARE = "uncShare";
    private final Pattern drives = startsWith( alpha().andThen( txt( ":" ) ) ).buildPattern();

    private final CharacterClass slash = txt( '\\' ).or( txt( '/' ) );
    private final Pattern unc =
            slash.atLeast( 2 ).
                    andThen( anyBut( slash ).oneOrMore().group( UNCSERVER ) ).
                    andThen( slash.atLeast( 1 ) ).
                    andThen( anyBut( slash ).oneOrMore().group( UNCSHARE ) ).
                    andThen( slash.
                             andThen( any().zeroOrMore() ).zeroOrOnce().group( UNCPATH ) ).
                    buildPattern();

    private final Pattern doubleSlashStart = startsWith( slash.times( 2 ) ).buildPattern();

    private static final Pattern separatorPat = Frex.or( txt('\\'), txt( '/' )).buildPattern();


    @Override
    public int getMaxPathLength() {
        return 32000;
    }

    @Override
    public int getPathLength( String str ) {
        return str.length();
    }

    @Override
    public int getMaxFilenameLength() {
        return 255;
    }

    @Override
    public int getFilenameLength( String str ) {
        return str.codePointCount( 0, str.length() );
    }

    @Override
    public int compareTo( String a, String b ) {
        return getNormalForm( a ).compareTo( getNormalForm( b ) );
    }

    @Override
    public RC getRootComponent( String str ) {
        if( drives.matcher( str ).matches() ) {
            return new RC( Optional.of( getNormalForm( str.substring( 0, 2 ) ) ), str.substring( 2 ) );
        }

        if ( !doubleSlashStart.matcher( str ).matches() ) {
            return new RC( Optional.empty(), str ); // relative path or single slash
        }

        Matcher matcher = unc.matcher( str );

        if( !matcher.matches() ) {
            throw new InvalidPathException( str, "illegal path (not unc)" );
        }

        String path   = Optional.ofNullable( matcher.group( UNCPATH )).orElseThrow( () -> new IllegalStateException( "bad group name" ));
        String server = getNormalForm( Optional.ofNullable( matcher.group( UNCSERVER )).orElseThrow( () -> new IllegalStateException( "bad group name" )));
        String share  = getNormalForm( Optional.ofNullable( matcher.group( UNCSHARE )).orElseThrow( () -> new IllegalStateException( "bad group name" )));

        return new RC( Optional.of( "\\\\" + server + "\\" + share ),
                       path.isEmpty() ? "\\" : path );
    }

    @Override
    public String getNormalForm( String str ) {
        return str.toLowerCase( Locale.getDefault() ).replace( '/', '\\' );
    }

    @Override
    public String getSeparator() {
        return "\\";
    }

    @Override
    public Pattern getAllSeparators() {
        return separatorPat;
    }

    @Override
    public Optional<Pattern> getIllegalCharacters() {
        return Optional.of(
                Frex.contains( or( txt( ':' ), txt( '?' ), txt( '<' ), txt( '>' ), txt( '*' ), txt( '|' ), txt( '"' ) ) ).
                        buildPattern());

    }
}
