package de.pfabulist.kleinod.os;

import java.util.regex.Pattern;

import static de.pfabulist.frex.Frex.contains;
import static de.pfabulist.frex.Frex.or;
import static de.pfabulist.frex.Frex.startsWith;
import static de.pfabulist.frex.Frex.txt;

/*****
 * BEGIN LICENSE BLOCK *****
 * BSD License (2 clause)
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * All rights reserved.
 * <p>
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Stephan Pfab BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *****/

public class OS {

    public static final OS UNIX = new OS( "unix" );
    public static final OS WINDOWS = new OS( "windows" );
    public static final OS OSX = new OS( "Mac OS X" );


    private final static Pattern UNIX_PATTERN = or( txt( "Linux" ),
                                                    txt( "Solaris" ),
                                                    txt( "SunOS" ),
                                                    txt( "HP-UX" ),
                                                    txt( "AIX" ),
                                                    txt( "FreeBSD" ),
                                                    txt( "Irix" ),
                                                    txt( "Digital Unix" ),
                                                    txt( "OSF1" ),
                                                    startsWith( "Mac OS X" ),
                                                    contains( "BSD" ),
                                                    contains( "nix" ),
                                                    contains( "nux" ) ).buildCaseInsensitivePattern();
    private final static Pattern WINDOWS_PATTERN = startsWith( "windows" ).buildCaseInsensitivePattern();
    private final static Pattern OSX_PATTERN = startsWith( "mac os x" ).buildCaseInsensitivePattern();
    private final static Pattern LINUX_PATTERN = contains( "linux" ).buildCaseInsensitivePattern();

    private final String osName;

    public OS() {
        this( System.getProperty( "os.name" ) );
    }

    public OS( String osName ) {
        this.osName = osName;
    }

    /**
     * Is this Windows ?
     *
     * @return true iff this system is windows
     */
    public boolean isWindows() {
        return WINDOWS_PATTERN.matcher( osName ).matches();
    }

//    /**
//     * Is this Windows Vista or later
//     * @return true iff this system is windows vista or later
//     */
//    public  boolean isWindowsVistaPlus() {
//        return "Windows Vista".equals( osName ) || "Windows 7".equals( osName );
//    }

    /**
     * Is this Linux ?
     *
     * @return true iff this system is Linux
     */
    public boolean isLinux() {
        return LINUX_PATTERN.matcher( osName ).matches();
    }

    /**
     * Is this OSX ?
     *
     * @return true iff this system is OSX
     */
    public boolean isOSX() {
        return OSX_PATTERN.matcher( osName ).matches();
    }

    /**
     * Is this Unix ?
     *
     * @return true iff this system is Unix
     */
    public boolean isUnix() {
        return UNIX_PATTERN.matcher( osName ).matches();
    }

    @Override
    public String toString() {
        return osName;
    }
}
