package de.pfabulist.kleinod.os;


import java.util.regex.Pattern;

import static de.pfabulist.kleinod.frex.Frex.contains;
import static de.pfabulist.kleinod.frex.Frex.or;
import static de.pfabulist.kleinod.frex.Frex.startsWith;
import static de.pfabulist.kleinod.frex.Frex.txt;
import static de.pfabulist.roast.types.NonnullCheck.n_;

/**
 * Copyright (c) 2006 - 2016, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

public class OS {

    public static final OS UNIX = new OS( "unix" );
    public static final OS WINDOWS = new OS( "windows" );
    public static final OS OSX = new OS( "Mac OS X" );

    private final static Pattern UNIX_PATTERN = or( txt( "Linux" ),
                                                    txt( "Solaris" ),
                                                    txt( "SunOS" ),
                                                    txt( "HP-UX" ),
                                                    txt( "AIX" ),
                                                    txt( "FreeBSD" ),
                                                    txt( "Irix" ),
                                                    txt( "Digital Unix" ),
                                                    txt( "OSF1" ),
                                                    startsWith( "Mac OS X" ),
                                                    contains( "BSD" ),
                                                    contains( "nix" ),
                                                    contains( "nux" ) ).buildCaseInsensitivePattern();
    private final static Pattern WINDOWS_PATTERN = startsWith( "windows" ).buildCaseInsensitivePattern();
    private final static Pattern OSX_PATTERN = startsWith( "mac os x" ).buildCaseInsensitivePattern();
    private final static Pattern LINUX_PATTERN = contains( "linux" ).buildCaseInsensitivePattern();

    private final String osName;

    public OS() {
        this( n_( System.getProperty( "os.name" ) ) );
    }

    public OS( String osName ) {
        this.osName = osName;
    }

    /**
     * Is this Windows ?
     *
     * @return true iff this system is windows
     */
    public boolean isWindows() {
        return WINDOWS_PATTERN.matcher( osName ).matches();
    }

//    /**
//     * Is this Windows Vista or later
//     * @return true iff this system is windows vista or later
//     */
//    public  boolean isWindowsVistaPlus() {
//        return "Windows Vista".equals( osName ) || "Windows 7".equals( osName );
//    }

    /**
     * Is this Linux ?
     *
     * @return true iff this system is Linux
     */
    public boolean isLinux() {
        return LINUX_PATTERN.matcher( osName ).matches();
    }

    /**
     * Is this OSX ?
     *
     * @return true iff this system is OSX
     */
    public boolean isOSX() {
        return OSX_PATTERN.matcher( osName ).matches();
    }

    /**
     * Is this Unix ?
     *
     * @return true iff this system is Unix
     */
    public boolean isUnix() {
        return UNIX_PATTERN.matcher( osName ).matches();
    }

    @Override
    public String toString() {
        return osName;
    }
}
