package de.pfabulist.roast.nio;

import de.pfabulist.roast.collection.Map_;

import javax.annotation.Nullable;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.UserPrincipal;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import static de.pfabulist.roast.NonnullCheck.n_;
import static de.pfabulist.roast.functiontypes.Runnable_.e_;
import static de.pfabulist.roast.functiontypes.Runnable_.v_;
import static de.pfabulist.roast.functiontypes.Supplier_.vn_;
import static de.pfabulist.roast.unchecked.Unchecked.u;

/**
 * Copyright (c) 2006 - 2017, Stephan Pfab
 * SPDX-License-Identifier: BSD-2-Clause
 */

/**
 * This class consists exclusively r_ static methods that operate on files,
 * directories, or other types r_ files.
 * <p>
 * In most cases, the methods defined here will delegate to the associated
 * file system provider to perform the file operations.
 */

@SuppressWarnings( { "PMD.ExcessivePublicCount", "PMD.ExcessiveClassLength", "PMD.GodClass", "PMD.TooManyMethods" } )
public final class Files_ {

    private Files_() {
    }

    // createDirectory -----------------

    public static Path_ createDirectory__( Path dir, FileAttribute<?>... attrs ) {
        return Path_.r_( createDirectory_( dir, attrs ) );
    }

    public static Path createDirectory( @Nullable Path tgt, FileAttribute<?>... attrs ) {
        return createDirectory_( n_( tgt ), attrs );
    }

    public static Path createDirectory_( Path dir, FileAttribute<?>... attrs ) {
        return vn_( () -> Files.createDirectory( dir, attrs ) );
    }

    public static void createDirectories( @Nullable Path dir, FileAttribute<?>... attrs ) {
        createDirectories_( n_( dir ), attrs );
    }

    public static void createDirectories_( Path dir, FileAttribute<?>... attrs ) {
        vn_( () -> Files.createDirectories( dir, attrs ) );
    }

    // exists

    public static boolean exists_( Path path, LinkOption... options ) {
        return vn_( () -> Files.exists( path, options ) );
    }

    public static boolean exists( @Nullable Path path, LinkOption... options ) {
        return exists_( n_(path), options  );
    }
    // write ---

    public static Path write( @Nullable Path path, @Nullable byte[] bytes, OpenOption... options ) {
        return write_( n_( path ), n_( bytes ), options );
    }

    public static Path write_( Path path, byte[] bytes, OpenOption... options ) {
        return vn_( () -> Files.write( path, bytes, options ) );
    }

    // list

    public static Stream<Path> list( @Nullable Path dir ) {
        return list_( n_( dir ) );
    }

    public static Stream<Path> list_( Path dir ) {
        return vn_( () -> Files.list( dir ) );
    }

    // newDirectoryStream

    public static DirectoryStream<Path> newDirectoryStream( @Nullable Path dir ) {
        return newDirectoryStream_( n_( dir ) );
    }

    public static DirectoryStream<Path> newDirectoryStream_( Path dir ) {
        return vn_( () -> Files.newDirectoryStream( dir ) );
    }

    public static DirectoryStream_<Path> newDirectoryStream__( Path dir ) {

        return DirectoryStream_.r_( newDirectoryStream( dir ) );

        //return provider_(dir).newDirectoryStream( dir, Files.AcceptAllFilter.FILTER);

        //return vn_( () -> Files.newDirectoryStream( dir ) );
    }

    // isDirectory

    public static boolean isDirectory_( Path path, LinkOption... options ) {
        return vn_( () -> Files.isDirectory( path, options ) );
    }

    // getLastModifiedTime

    public static
    @Nullable
    FileTime getLastModifiedTime( @Nullable Path path, LinkOption... options ) {
        return getLastModifiedTime_( n_( path ), options );
    }

    public static FileTime getLastModifiedTime_( Path path, LinkOption... options ) {
        return vn_( () -> Files.getLastModifiedTime( path, options ) );
    }

    // getLastAccessTime

    public static FileTime getLastAccessTime( @Nullable Path path ) {
        return getLastAccessTime_( n_( path ) );
    }

    public static FileTime getLastAccessTime_( Path path, LinkOption... options ) {
        return vn_( () -> n_( Files_.readAttributes_( path, BasicFileAttributes.class, options ).lastAccessTime() ) );
    }

    // getCreationTime

    public static FileTime getCreationTime( @Nullable Path path, LinkOption... options ) {
        return getCreationTime_( n_( path ), options );
    }

    public static FileTime getCreationTime_( Path path, LinkOption... options ) {
        return vn_( () -> n_( Files_.readAttributes_( path, BasicFileAttributes.class, options ).creationTime() ) );
    }

    // readAllBytes ---

    public static <A extends BasicFileAttributes> A readAttributes( @Nullable Path path, @Nullable Class<A> type, @Nullable LinkOption... options ) {
        return readAttributes_( n_( path ), n_( type ), n_( options ) );
    }

    public static <A extends BasicFileAttributes> A readAttributes_( Path path, Class<A> type, LinkOption... options ) {
        return vn_( () -> Files.readAttributes( path, type, options ) );
    }

    public static Map<String, Object> readAttributes( @Nullable Path path, @Nullable String attributes, LinkOption... options ) {
        return readAttributes_( n_( path ), n_( attributes ), options );
    }

    public static Map<String, Object> readAttributes_( Path path, String attributes, LinkOption... options ) {
        return n_( vn_( () -> Files.readAttributes( path, attributes, options ) ) );
    }

    public static Map_<String, Object> readAttributes__( Path path, String attributes, LinkOption... options ) {
        return Map_.r_( n_( vn_( () -> Files.readAttributes( path, attributes, options ) ) ) );
    }

    // readAllBytes ---

    public static byte[] readAllBytes( @Nullable Path path ) {
        return readAllBytes_( n_( path ) );
    }

    public static byte[] readAllBytes_( Path path ) {
        return vn_( () -> Files.readAllBytes( path ) );

    }

    // newByteChannel --------------

    public static SeekableByteChannel newByteChannel( @Nullable Path path, @Nullable Set<StandardOpenOption> standardOpenOptions, FileAttribute<?>... attrs ) {
        return newByteChannel_( n_( path ), n_( standardOpenOptions ), attrs );
    }

    public static SeekableByteChannel newByteChannel_( Path path, Set<StandardOpenOption> standardOpenOptions, FileAttribute<?>... attrs ) {
        return vn_( () -> Files.newByteChannel( path, standardOpenOptions, attrs ) );
    }

    public static SeekableByteChannel_ newByteChannel__( Path path, Set<StandardOpenOption> standardOpenOptions, FileAttribute<?>... attrs ) {
        return SeekableByteChannel_.r_( newByteChannel_( path, standardOpenOptions, attrs ) );
    }

    public static SeekableByteChannel newByteChannel( @Nullable Path path, OpenOption... options ) {
        return newByteChannel_( n_( path ), options );
    }

    public static SeekableByteChannel newByteChannel_( Path path, OpenOption... options ) {
        return vn_( () -> Files.newByteChannel( path, options ) );
    }

    public static SeekableByteChannel_ newByteChannel__( Path path, OpenOption... options ) {
        return SeekableByteChannel_.r_( newByteChannel_( path, options ) );
    }

    // size_ -------------------------

    public static long size( @Nullable Path target ) {
        return size_( n_( target ) );
    }

    public static long size_( Path target ) {
        return vn_( () -> Files.size( target ) );
    }

    // copy ---------------------

    public static Path copy( @Nullable Path src, @Nullable Path tgt, CopyOption... options ) {
        return copy_( n_( src ), n_( tgt ), options );
    }

    public static Path copy_( Path src, Path tgt, CopyOption... options ) {
        return vn_( () -> Files.copy( src, tgt, options ) );
    }

    public static long copy_( Path source, OutputStream out ) {
        return vn_( () -> Files.copy( source, out ) );
    }

    public static long copy( @Nullable Path source, @Nullable OutputStream out ) {
        return copy_( n_(source), n_(out) );
    }


        // delete ------------------

    public static void delete( @Nullable Path path ) {
        delete_( n_( path ) );
    }

    public static void delete_( Path path ) {
        v_( () -> Files.delete( path ) );
    }

    // move ----

    public static void move( @Nullable Path src, @Nullable Path tgt, CopyOption... options ) {
        move_( n_( src ), n_( tgt ), options );
    }

    public static void move_( Path src, Path tgt, CopyOption... options ) {
        vn_( () -> Files.move( src, tgt, options ) );
    }

    public static boolean deleteIfExists( @Nullable Path path ) {
        return deleteIfExists_( n_( path ) );
    }

    public static boolean deleteIfExists_( Path path ) {
        return vn_( () -> Files.deleteIfExists( path ) );
    }

    //
    // extra ---
    //

    public static boolean isKid( @Nullable Path dir, @Nullable Path kid ) {
        return isKid_( n_( dir ), n_( kid ) );
    }

    public static boolean isKid_( Path dir, Path kid ) {
        return list_( dir ).
                anyMatch( k -> k.equals( kid ) );
    }

    public static boolean isSymbolicLink( Path path ) {
        return Files.isSymbolicLink( path );
    }

    public static void setLastModifiedTime( @Nullable Path file, @Nullable FileTime past ) {
        setLastModifiedTime_( n_( file ), n_( past ) );
    }

    public static void setLastModifiedTime_( Path file, FileTime past ) {
        vn_( () -> Files.setLastModifiedTime( file, past ) );
    }

    public static boolean isSameFile( @Nullable Path file, @Nullable Path file1 ) {
        return isSameFile_( n_( file ), n_( file1 ) );
    }

    public static boolean isSameFile_( Path file, Path file1 ) {
        return vn_( () -> Files.isSameFile( file, file1 ) );
    }

    public static Path createLink( @Nullable Path link, @Nullable Path existing ) {
        return createLink_( n_( link ), n_( existing ) );
    }

    public static Path createLink_( Path link, Path existing ) {
        return vn_( () -> Files.createLink( link, existing ) );
    }

    public static FileStore getFileStore( @Nullable Path path ) {
        return getFileStore_( n_( path ) );
    }

    private static FileStore getFileStore_( Path path ) {
        return vn_( () -> Files.getFileStore( path ) );
    }

//    // provider --
//
//    public static FileSystemProvider provider() {
//        return Files.provider();
//    }

    // copy

    public static long copy( @Nullable InputStream is, @Nullable Path path, CopyOption... options ) {
        return copy_( n_( is ), n_( path ), options );
    }

    public static long copy_( InputStream is, Path path, CopyOption... options ) {
        return vn_( () -> Files.copy( is, path, options ) );
    }

    public static InputStream newInputStream( Path jar, OpenOption... options ) {
        return vn_( () -> Files.newInputStream( jar, options ) );
    }

    /**
     * similar to
     * see {@link Files#lines(Path, Charset)}
     */
    public static Stream<String> lines( InputStream is, Charset cs ) {
        CharsetDecoder decoder = cs.newDecoder();
        Reader reader = new InputStreamReader( is, decoder );

        BufferedReader br = new BufferedReader( reader );
        try {
            return br.lines().onClose( e_( br::close ) );//asUncheckedRunnable( br ) );
        } catch( Error | RuntimeException e ) {
            try {
                br.close();
            } catch( IOException ex ) {
                try {
                    e.addSuppressed( ex );
                } catch( Throwable ignore ) {
                }
            }
            throw e;
        }
    }

    public static void unzipToPath( InputStream is, Path root ) {
        try( ZipInputStream zin = new ZipInputStream( is ) ) {

            @Nullable ZipEntry ze;
            while( ( ze = zin.getNextEntry() ) != null ) {
                String name = ze.getName();
                Path target = n_( root.resolve( name ) );

                if( name.endsWith( "/" ) ) {
                    Files_.createDirectories_( target );
                } else {
                    Files_.copy( zin, target );
                }
                zin.closeEntry();
            }
        } catch( IOException e ) {
            throw u( e );
        }

    }

    public static boolean isEmpty_( Path dir ) {
        try( DirectoryStream<Path> stream = Files.newDirectoryStream( dir ) ) {
            return !stream.iterator().hasNext();
        } catch( IOException e ) {
            throw u( e );
        }
    }

    public static boolean isRoot_( Path path ) {
        return path.isAbsolute() && path.getParent() == null;
    }

    public static boolean isHidden_( Path resolve ) {
        return vn_( () -> Files.isHidden( resolve ) );
    }

    public static UserPrincipal getOwner_( Path path, LinkOption... options ) {
        return vn_( () -> Files.getOwner( path, options ) );
    }

    public static void setOwner_( Path path, UserPrincipal user ) {
        vn_( () -> Files.setOwner( path, user ) );
    }

    public static Path setAttribute_( Path path, String attribute, Object value, LinkOption... options ) {
        vn_( () -> Files.setAttribute( path, attribute, value, options ) );
        return path;
    }

    public static <V extends FileAttributeView> Optional<V> getFileAttributeView_o( Path path, Class<V> type, LinkOption... options ) {
        return Optional.ofNullable( Files.getFileAttributeView( path, type, options ) );
    }

    public static <V extends FileAttributeView> V getFileAttributeView_ot( Path path, Class<V> type, LinkOption... options ) {
        return getFileAttributeView_o( path, type, options ).orElseThrow( () -> new IllegalArgumentException( "no such Attribute View " + type ) );
    }

    public static void deleteRecursive_( Path path ) {
        if( !exists_( path ) ) {
            return;
        }

        final Deque<Path> toDel = new ArrayDeque<>();

        if( isDirectory_( path ) ) {
            Files_.walk_( path ).forEach( toDel::addFirst );
        } else {
            toDel.add( path );
        }

        toDel.forEach( Files_::delete );
    }

    // walk

    public static Stream<Path> walk_( Path start, FileVisitOption... options ) {
        return vn_( () -> n_( Files.walk( start, options ) ) );
    }

    public static Stream<Path> walk( @Nullable Path start, FileVisitOption... options ) {
        return walk_( n_( start ), options );
    }

    // copyRecursive_

    public static void copyRecursive_( Path src, Path target, CopyOption... options ) {
        if( !Files_.exists_( src ) ) {
            return;
        }

        if( !Files_.isDirectory_( src ) ) {
            Files_.copy( src, target, options );
            return;
        }

        walk_( src ).forEach( kid ->
                              {
                                  Path to = n_( target.resolve( src.relativize( kid ) ) );
                                  Files_.copy( kid, to, options );
                              }
        );

    }

    // createSymbolicLink_ ---------------

    public static Path createSymbolicLink_( Path link, Path target, FileAttribute<?>... attrs ) {
        return vn_( () -> Files.createSymbolicLink( link, target, attrs ) );
    }

    public static Path createSymbolicLink( @Nullable Path link, @Nullable Path target, FileAttribute<?>... attrs ) {
        return createSymbolicLink_( n_( link ), n_( target ), attrs );
    }

    // readSymbolicLink ---------------

    public static Path readSymbolicLink_( Path path ) {
        return vn_( () -> Files.readSymbolicLink( path ));
    }
    public static Path readSymbolicLink( @Nullable Path path ) {
        return readSymbolicLink_( n_( path) );
    }

    // createFile

    public static Path createFile_(Path path, FileAttribute<?>... attrs) {
        return vn_( () -> Files.createFile( path, attrs ));
    }

    public static Path createFile( @Nullable Path path, FileAttribute<?>... attrs) {
        return createFile_( n_(path), attrs );
    }

    // newOutputStream

    public static OutputStream newOutputStream_( Path path, OpenOption... options) {
        return vn_( () -> Files.newOutputStream( path, options ));
    }

    public static OutputStream newOutputStream( @Nullable Path path, OpenOption... options) {
        return newOutputStream_( n_( path ), options );
    }
}
