package de.pheasn.pluginupdater;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Timer;
import java.util.TimerTask;

import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.JSONValue;

public class Updater {

	private static final String API_NAME_VALUE = "name";
	private static final String API_LINK_VALUE = "downloadUrl";
	private static final String API_RELEASE_TYPE_VALUE = "releaseType";
	private static final String API_QUERY = "/servermods/files?projectIds=";
	private static final String API_HOST = "https://api.curseforge.com";

	/**
	 * 5 minutes
	 */
	private static final int MIN_INTERVAL = 5;
	/**
	 * 18 minutes
	 */
	private static final int DEFAULT_INTERVAL = 18;

	private final Updatable plugin;
	private final int pluginId;
	private final String apiKey;
	private final ReleaseChannel channel;
	private final TimerTask timerTask;

	private Timer timer;
	private boolean updatePending = false;

	public static class Builder {
		private final Updatable plugin;
		private final int pluginId;

		private String apiKey = null;
		private ReleaseChannel channel = ReleaseChannel.STABLE;
		private int interval = DEFAULT_INTERVAL;

		public Builder(Updatable plugin, int pluginId) {
			if (plugin == null)
				throw new NullPointerException("Plugin can't be null");
			this.plugin = plugin;
			this.pluginId = pluginId;
		}

		public Builder apiKey(String apiKey) {
			this.apiKey = apiKey;
			return this;
		}

		public Builder channel(ReleaseChannel channel) {
			if (channel == null)
				throw new NullPointerException("ReleaseChannel can't be null");
			this.channel = channel;
			return this;
		}

		/**
		 * Set the update check interval in minutes
		 *
		 * @param interval
		 *            the interval in minutes. Must be greater or equal 5
		 *            minutes.
		 * @return this Builder
		 * @throws IllegalArgumentException
		 *             if the interval is less than 5 minutes
		 */
		public Builder interval(int interval) throws IllegalArgumentException {
			if (interval < MIN_INTERVAL)
				throw new IllegalArgumentException("Interval must be greater or equal 5 minutes");
			this.interval = interval;
			return this;
		}

		public Updater build() {
			return new Updater(plugin, pluginId, apiKey, channel, interval);
		}
	}

	private Updater(Updatable plugin, int pluginId, String apiKey, ReleaseChannel channel, int interval) {
		this.plugin = plugin;
		this.pluginId = pluginId;
		this.apiKey = apiKey;
		this.channel = channel;
		this.timer = new Timer();
		this.timerTask = new TimerTask() {
			@Override
			public void run() {
				checkForUpdate();
			}
		};

		setInterval(interval);
	}

	/**
	 * Sets the update check interval in minutes.
	 *
	 * @param interval
	 *            the new check interval. Must be greater or equal 5
	 */
	public void setInterval(int interval) {
		long msInterval = (interval < MIN_INTERVAL ? DEFAULT_INTERVAL : interval) * 60 * 1000;
		cancel();
		timer.schedule(getTimerTask(), msInterval, msInterval);
	}

	/**
	 * Cancels all scheduled update checks
	 */
	public void cancel() {
		timer.cancel();
		timer = new Timer();
	}

	void checkForUpdate() {
		if (updatePending)
			return;
		updatePending = true;

		URL url = null;
		try {
			url = new URL(API_HOST + API_QUERY + pluginId);
		} catch (MalformedURLException e) {
			updatePending = false;
			return;
		}

		try {
			URLConnection con = url.openConnection();
			if (apiKey != null) {
				con.addRequestProperty("X-API-Key", apiKey);
			}
			con.addRequestProperty("User-Agent", "PluginUpdater by Pheasn");

			BufferedReader reader = new BufferedReader(new InputStreamReader(con.getInputStream()));
			String response = reader.readLine();
			JSONArray array = (JSONArray) JSONValue.parse(response);
			if (!array.isEmpty()) {
				for (int i = array.size() - 1; i >= 0; i--) {
					JSONObject version = (JSONObject) array.get(i);
					String versionLink = (String) version.get(API_LINK_VALUE);
					String versionName = (String) version.get(API_NAME_VALUE);
					String releaseTypeName = ((String) version.get(API_RELEASE_TYPE_VALUE)).toUpperCase();

					ReleaseChannel pendingChannel = ReleaseChannel.parse(releaseTypeName);

					if (pendingChannel.compareTo(channel) >= 0) {
						PluginVersion pendingVersion = new PluginVersion(versionName, pendingChannel);
						PluginVersion currentVersion = new PluginVersion(plugin.getVersion(),
								plugin.getReleaseChannel());
						if (pendingVersion.compareTo(currentVersion) == 1) {
							File file = new File("./plugins/update", plugin.getFile().getName());
							file.getParentFile().mkdirs();
							if (file.exists() && !file.delete()) {
								updatePending = false;
								break;
							}
							if (!file.createNewFile()) {
								updatePending = false;
								break;
							}

							URL dwnurl = new URL(versionLink);
							InputStream in = dwnurl.openStream();
							FileOutputStream out = new FileOutputStream(file);
							int fetched;
							byte[] buffer = new byte[4096];
							while ((fetched = in.read(buffer)) != -1) {
								out.write(buffer, 0, fetched);
							}
							out.close();
							in.close();
							plugin.updateNotify(versionName);
							break;
						} else {
							updatePending = false;
							break;
						}
					}
				}
			}
			reader.close();
		} catch (Exception e) {
			updatePending = false;
		}
	}

	private TimerTask getTimerTask() {
		return timerTask;
	}
}
