package de.pheasn.pluginupdater;

import java.util.Arrays;

public final class PluginVersion implements Comparable<PluginVersion> {

  private final int[] versionArr;
  private final String versionString;
  private final ReleaseChannel releaseChannel;


  /**
   * Instantiates a new PluginVersion object.
   *
   * @param versionString a String containing the version similar to "version 3.1.23.1" all
   *        non-numeric characters will be removed from the String
   * @param releaseChannel the release type of this version
   */
  public PluginVersion(String versionString, ReleaseChannel releaseChannel) {
    if (versionString == null) {
      throw new NullPointerException("versionString null");
    }
    versionString = removeNonNumeric(versionString);
    this.releaseChannel = releaseChannel;

    if (versionString.isEmpty()) {
      throw new IllegalArgumentException();
    }

    String[] split = versionString.split("\\.");
    versionArr = new int[split.length];
    StringBuilder sb = new StringBuilder(split.length * 3);
    try {
      for (int i = 0; i < split.length; i++) {
        if (split[i].isEmpty()) {
          throw new IllegalArgumentException();
        }
        versionArr[i] = Integer.parseInt(split[i]);
        sb.append(versionArr[i] + ".");
      }
    } catch (NumberFormatException e) {
      throw new IllegalArgumentException();
    }
    sb.deleteCharAt(sb.length() - 1);
    this.versionString = sb.toString();
  }

  /**
   * Gets the version in the form i[.i]*
   * 
   * @return the version
   */
  public String getVersion() {
    return versionString;
  }

  public ReleaseChannel getReleaseChannel() {
    return releaseChannel;
  }

  private String removeNonNumeric(String inputString) {
    StringBuilder output = new StringBuilder();
    for (char c : inputString.toCharArray()) {
      if (Character.isDigit(c) || c == '.') {
        output.append(c);
      }
    }
    return output.toString();
  }

  @Override
  public int compareTo(PluginVersion v2) {
    int index = 0;
    while (index < versionArr.length && index < v2.versionArr.length) {
      if (versionArr[index] > v2.versionArr[index]) {
        return 1;
      } else if (versionArr[index] < v2.versionArr[index]) {
        return -1;
      }
      index++;
    }
    // if both array index limits have been reached, check for release type
    if (index == v2.versionArr.length && index == versionArr.length) {
      if (releaseChannel == null) {
        return -1;
      } else if (v2.releaseChannel == null) {
        return 1;
      } else {
        return releaseChannel.compareTo(v2.releaseChannel);
      }
    } else {
      // if only one version array's limit has been reached, the other version
      // may be newer
      int index2 = index;
      while (index2 < versionArr.length) {
        if (versionArr[index2] != 0) {
          return 1;
        }
        index2++;
      }
      index2 = index;
      while (index2 < v2.versionArr.length) {
        if (v2.versionArr[index2] != 0) {
          return -1;
        }
        index2++;
      }
      // If there additional array elements are 0's only, the version
      // numbers are
      // the same, check for release type
      if (releaseChannel == null) {
        return -1;
      } else if (v2.releaseChannel == null) {
        return 1;
      } else {
        return releaseChannel.compareTo(v2.releaseChannel);
      }
    }
  }

  /**
   * Checks whether version is significantly newer than version2.
   *
   * @param version2 the other version
   * @param significance 0-based significance
   * @return True, if any of the version numbers with index equal or smaller i is bigger than the
   *         version number of version2 with the same index
   */
  public boolean isSignificantlyNewerThan(PluginVersion version2, int significance) {
    int index = 0;
    while (index <= significance && index < versionArr.length
        && index < version2.versionArr.length) {
      if (versionArr[index] > version2.versionArr[index]) {
        return true;
      }
      index++;
    }
    while (index <= significance && index < versionArr.length) {
      if (versionArr[index] != 0) {
        return true;
      }
      index++;
    }
    return false;
  }

  @Override
  public int hashCode() {
    final int prime = 61;
    int result = 7;
    result = prime * result + ((releaseChannel == null) ? 0 : releaseChannel.hashCode());
    boolean nonZeroFound = false;
    for (int i = versionArr.length - 1; i >= 0; i--) {
      if (versionArr[i] != 0) {
        nonZeroFound = true;
      }
      if (nonZeroFound) {
        result = prime * result + versionArr[i];
      }
    }
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    PluginVersion other = (PluginVersion) obj;
    if (this.compareTo(other) != 0) {
      return false;
    }
    return true;
  }

  @Override
  public String toString() {
    return "PluginVersion [versionArr=" + Arrays.toString(versionArr) + ", releaseChannel="
        + releaseChannel + "]";
  }
}
