package de.pheasn.pluginupdater;

import com.google.auto.value.AutoValue;
import com.google.common.annotations.VisibleForTesting;

import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.JSONValue;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.Timer;
import java.util.TimerTask;

import javax.annotation.Nullable;

@AutoValue
public abstract class Updater {

  @VisibleForTesting
  static boolean debug = false;
  @VisibleForTesting
  UpdateCheckFinishedEvent test = null;


  private static final String API_NAME_VALUE = "name";
  private static final String API_LINK_VALUE = "downloadUrl";
  private static final String API_RELEASE_TYPE_VALUE = "releaseType";
  private static final String API_QUERY = "/servermods/files?projectIds=";
  private static final String API_HOST = "https://api.curseforge.com";

  /**
   * 5 minutes.
   */
  private static final long MIN_INTERVAL = 5;
  /**
   * 18 minutes.
   */
  private static final long DEFAULT_INTERVAL = 18;

  private final TimerTask timerTask;

  private Timer timer;
  private boolean updatePending = false;

  public static Builder builder() {
    return new AutoValue_Updater.Builder().interval(DEFAULT_INTERVAL)
        .releaseChannel(ReleaseChannel.STABLE);
  }

  @AutoValue.Builder
  public abstract static class Builder {

    /**
     * Sets the desired release channel. Default: STABLE
     * 
     * @param channel the ReleaseChannel
     * @return this Builder
     */
    public abstract Builder releaseChannel(ReleaseChannel channel);

    /**
     * Sets the plugin which must implement the Updatable interface.
     * 
     * @param plugin the plugin
     * @return this Builder
     */
    public abstract Builder plugin(Updatable plugin);

    /**
     * Sets the Bukkit plugin ID.
     * 
     * @param id the ID
     * @return this Builder
     */
    public abstract Builder pluginId(int id);

    /**
     * Sets the update check interval in minutes.
     *
     * @param interval the interval in minutes. Must be greater or equal 5 minutes.
     * @return this Builder
     */
    public abstract Builder interval(long interval);

    /**
     * Sets the API-Key. Every server owner can obtain his Bukkit API-Key on bukkit.org. Default:
     * null
     * 
     * @param apiKey the API key
     * @return this Builder
     */
    @Nullable
    public abstract Builder apiKey(String apiKey);

    abstract Updater autoBuild();

    /**
     * Returns an instance of the Updater class.
     * 
     * @return the Updater
     */
    public Updater build() {
      Updater updater = autoBuild();
      if (!debug && updater.getInterval() < MIN_INTERVAL) {
        throw new IllegalArgumentException("Interval must be greater or equal 5 minutes");
      }
      updater.setInterval(updater.getInterval());
      return updater;
    }
  }

  /**
   * Returns the server owner's API key.
   *
   * @return the API key
   */
  @Nullable
  public abstract String getApiKey();

  /**
   * Returns the minimum ReleaseChannel.
   *
   * @return the ReleaseChannel
   */
  public abstract ReleaseChannel getReleaseChannel();

  /**
   * Returns the initial update check interval in minutes. The returned value doesn't reflect
   * changes on the interval that have been made by invoking setInterval().
   *
   * @return the interval
   */
  abstract long getInterval();

  /**
   * Returns the plugin's Bukkit ID.
   * 
   * @return the ID
   */
  public abstract int getPluginId();

  /**
   * Returns the updatable plugin.
   * 
   * @return the Updatable
   */
  public abstract Updatable getPlugin();

  Updater() {
    this.timer = new Timer();
    this.timerTask = new TimerTask() {
      @Override
      public void run() {
        checkForUpdate();
      }
    };
  }

  /**
   * Sets the update check interval in minutes.
   *
   * @param interval the new check interval. Must be greater or equal 5
   */
  public void setInterval(long interval) {
    long msInterval;
    if (!debug) {
      msInterval = (interval < MIN_INTERVAL ? DEFAULT_INTERVAL : interval) * 60 * 1000;
    } else {
      msInterval = interval;
    }
    cancel();
    timer.schedule(timerTask, msInterval, msInterval);
  }

  /**
   * Cancels all scheduled update checks.
   */
  public void cancel() {
    timer.cancel();
    timer = new Timer();
  }

  void checkForUpdate() {
    if (updatePending) {
      return;
    }
    updatePending = true;

    URL url = null;
    try {
      url = new URL(API_HOST + API_QUERY + getPluginId());
    } catch (MalformedURLException e) {
      updatePending = false;
      if (debug) {
        test.updateCheckFinishedEvent();
      }
      return;
    }

    FileOutputStream fileOutput = null;
    try {
      URLConnection con = url.openConnection();
      if (getApiKey() != null) {
        con.addRequestProperty("X-API-Key", getApiKey());
      }
      con.addRequestProperty("User-Agent", "PluginUpdater by Pheasn");

      BufferedReader reader = new BufferedReader(
          new InputStreamReader(con.getInputStream(), Charset.forName("US-ASCII")));
      String response = reader.readLine();
      reader.close();
      JSONArray array = (JSONArray) JSONValue.parse(response);
      if (!array.isEmpty()) {
        for (int i = array.size() - 1; i >= 0; i--) {
          JSONObject version = (JSONObject) array.get(i);
          String versionLink = (String) version.get(API_LINK_VALUE);
          String versionName = (String) version.get(API_NAME_VALUE);
          String releaseTypeName = ((String) version.get(API_RELEASE_TYPE_VALUE)).toUpperCase();

          ReleaseChannel pendingChannel = ReleaseChannel.parse(releaseTypeName);
          if (pendingChannel.compareTo(getReleaseChannel()) >= 0) {
            PluginVersion pendingVersion = new PluginVersion(versionName, pendingChannel);
            PluginVersion currentVersion =
                new PluginVersion(getPlugin().getVersion(), getPlugin().getReleaseChannel());
            if (pendingVersion.compareTo(currentVersion) > 0) {
              File file = new File(new File(getPlugin().getFile().getParentFile(), "update"),
                  getPlugin().getFile().getName());
              if (!file.getParentFile().exists() && !file.getParentFile().mkdirs()) {
                updatePending = false;
                break;
              }
              if (file.exists() && !file.delete()) {
                updatePending = false;
                break;
              }
              if (!file.createNewFile()) {
                updatePending = false;
                break;
              }

              URL dwnurl = new URL(versionLink);
              InputStream in = dwnurl.openStream();
              fileOutput = new FileOutputStream(file);
              int fetched;
              byte[] buffer = new byte[4096];
              while ((fetched = in.read(buffer)) != -1) {
                fileOutput.write(buffer, 0, fetched);
              }
              fileOutput.close();
              in.close();
              getPlugin().updateNotify(versionName);
              break;
            } else {
              updatePending = false;
              break;
            }
          }
        }
      }
    } catch (Exception e) {
      if (fileOutput != null) {
        try {
          fileOutput.close();
        } catch (IOException e1) {
          // Ignore this
        }
      }
      if (debug) {
        e.printStackTrace();
      }
      updatePending = false;
    }

    if (debug) {
      test.updateCheckFinishedEvent();
    }
  }

}
