package de.pheasn.pluginupdater;

import java.util.Arrays;

public class PluginVersion implements Comparable<PluginVersion> {

  private final int[] versionArr;
  private final ReleaseChannel releaseChannel;


  /**
   * Instantiates a new PluginVersion object.
   *
   * @param versionString a String containing the version similar to "version 3.1.23.1" all
   *        non-numeric characters will be removed from the String
   * @param releaseChannel the (minimum) release channel you want to receive updates from.
   */
  public PluginVersion(String versionString, ReleaseChannel releaseChannel) {
    versionString = removeNonNumeric(versionString);
    this.releaseChannel = releaseChannel;

    String[] splitted = versionString.split("\\.");
    versionArr = new int[splitted.length];
    for (int i = 0; i < splitted.length; i++) {
      versionArr[i] = Integer.parseInt(splitted[i]);
    }
  }

  private String removeNonNumeric(String inputString) {
    StringBuilder output = new StringBuilder();
    for (char c : inputString.toCharArray()) {
      if (Character.isDigit(c) || c == '.') {
        output.append(c);
      }
    }
    return output.toString();
  }

  @Override
  public int compareTo(PluginVersion v2) {
    int index = 0;
    while (index < versionArr.length && index < v2.versionArr.length) {
      if (versionArr[index] > v2.versionArr[index]) {
        return 1;
      } else if (versionArr[index] < v2.versionArr[index]) {
        return -1;
      }
      index++;
    }
    // if both array index limits have been reached, check for release type
    if (index == v2.versionArr.length && index == versionArr.length) {
      int compareResult = releaseChannel.compareTo(v2.releaseChannel);
      if (compareResult < 0) {
        return -1;
      } else if (compareResult > 0) {
        return 1;
      } else {
        return 0;
      }
    } else {
      // if only one version array's limit has been reached, the other version
      // may be newer
      int index2 = index;
      while (index2 < versionArr.length) {
        if (versionArr[index2] != 0) {
          return 1;
        }
        index2++;
      }
      index2 = index;
      while (index2 < v2.versionArr.length) {
        if (v2.versionArr[index2] != 0) {
          return -1;
        }
        index2++;
      }
      // If there additional array elements are 0's only, the version
      // numbers are
      // the same, check for release type
      int compareResult = releaseChannel.compareTo(v2.releaseChannel);
      if (compareResult < 0) {
        return -1;
      } else if (compareResult > 0) {
        return 1;
      } else {
        return 0;
      }
    }
  }

  /**
   * Checks whether version is significantly newer than version2.
   *
   * @param version2 the other version
   * @param significance 0-based significance
   * @return True, if any of the version numbers with index equal or smaller i is bigger than the
   *         version number of version2 with the same index
   */
  public boolean isSignificantlyNewerThan(PluginVersion version2, int significance) {
    int index = 0;
    while (index <= significance && index < versionArr.length
        && index < version2.versionArr.length) {
      if (versionArr[index] > version2.versionArr[index]) {
        return true;
      }
      index++;
    }
    while (index <= significance && index < versionArr.length) {
      if (versionArr[index] != 0) {
        return true;
      }
      index++;
    }
    return false;
  }

  @Override
  public int hashCode() {
    final int prime = 61;
    int result = 7;
    result = prime * result + ((releaseChannel == null) ? 0 : releaseChannel.hashCode());
    boolean nonZeroFound = false;
    for (int i = versionArr.length - 1; i >= 0; i--) {
      if (versionArr[i] != 0) {
        nonZeroFound = true;
      }
      if (nonZeroFound) {
        result = prime * result + versionArr[i];
      }
    }
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    PluginVersion other = (PluginVersion) obj;
    if (this.compareTo(other) != 0) {
      return false;
    }
    return true;
  }

  @Override
  public String toString() {
    return "PluginVersion [versionArr=" + Arrays.toString(versionArr) + ", releaseChannel="
        + releaseChannel + "]";
  }
}
