package de.philippkatz.maven.plugins.dependencyresolver;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Properties;
import java.util.jar.Manifest;
import java.util.stream.Collectors;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.graph.Exclusion;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactResult;
import org.eclipse.aether.resolution.DependencyRequest;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// code based on https://github.com/apache/maven-resolver/tree/master/maven-resolver-demos/maven-resolver-demo-maven-plugin

@Mojo(name = "resolve-dependencies", threadSafe = true)
public class DependencyResolver extends AbstractMojo {

	private static final Logger LOGGER = LoggerFactory.getLogger(DependencyResolver.class);

	/** Folder which contains the downloaded libs. */
	private static final String LIB_DOWNLOAD = "lib-download";

	@Parameter(readonly = true, defaultValue = "${project}")
	private MavenProject project;

	@Component
	private RepositorySystem repoSystem;

	@Parameter(defaultValue = "${repositorySystemSession}", readonly = true)
	private RepositorySystemSession repoSession;

	@Parameter(property = "dependencies", required = true)
	private List<String> dependencies;

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {

		var collectRequest = new CollectRequest();

		for (var dependency : dependencies) {
			// TODO - if dependency starts with !, do not get transitive dependencies - ugly
			// convention, make this more explicit!
			// !ws.palladian:palladian-core::jar-with-dependencies:3.0.0-SNAPSHOT
			var noTransitive = dependency.startsWith("!");
			var normalizedDepencendy = dependency.replaceAll("^!", "");
			if (noTransitive) {
				var exclusions = Arrays.asList(new Exclusion("*", "*", "*", "*"));
				collectRequest.addDependency(
						new Dependency(new DefaultArtifact(normalizedDepencendy), "compile", null, exclusions));
			} else {
				collectRequest.addDependency(new Dependency(new DefaultArtifact(normalizedDepencendy), "compile"));
			}
		}

		collectRequest.addRepository(
				new RemoteRepository.Builder("central", "default", "https://repo.maven.apache.org/maven2/").build());
		// TODO - make configurable, or take from project configuration?
		collectRequest.addRepository(new RemoteRepository.Builder("snapshots", "default",
				"https://oss.sonatype.org/content/repositories/snapshots/").build());

		var dependencyRequest = new DependencyRequest(collectRequest, null);

		List<ArtifactResult> artifactResults;
		try {
			artifactResults = repoSystem.resolveDependencies(repoSession, dependencyRequest).getArtifactResults();
		} catch (DependencyResolutionException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}

		var projectPath = project.getBasedir().toPath();
		var libDownloadPath = projectPath.resolve(LIB_DOWNLOAD);

		if (Files.isDirectory(libDownloadPath)) {
			try {
				LOGGER.info("Delete contents of {}", libDownloadPath);
				Files.walk(libDownloadPath) //
						.sorted(Comparator.reverseOrder()) //
						.map(Path::toFile) //
						.forEach(File::delete);
			} catch (IOException e) {
				throw new MojoExecutionException("Could not delete " + libDownloadPath, e);
			}
		}

		try {
			Files.createDirectories(libDownloadPath);
		} catch (IOException e) {
			throw new MojoExecutionException("Could not create " + libDownloadPath, e);
		}

		for (var artifactResult : artifactResults) {
			var artifact = artifactResult.getArtifact();
			var name = getArtifactFilename(artifact);
			var destinationPath = libDownloadPath.resolve(name);
			try {
				LOGGER.info("Copy to {}", destinationPath);
				Files.copy(artifact.getFile().toPath(), destinationPath, StandardCopyOption.REPLACE_EXISTING);
			} catch (IOException e) {
				throw new MojoExecutionException("Could not copy to " + destinationPath, e);
			}
		}

		// test that all downloaded dependencies are referenced in the MANIFEST.MF and
		// the build.properties file - if not let the build fail and have the user
		// update these files manually

		var pathEntries = artifactResults //
				.stream().map(r -> LIB_DOWNLOAD + "/" + getArtifactFilename(r.getArtifact())) //
				.collect(Collectors.toList());

		var bundleClassPath = parseBundleClassPath(projectPath.resolve(Paths.get("META-INF/MANIFEST.MF")));
		var missingBundeClassPath = pathEntries //
				.stream() //
				.filter(entry -> !bundleClassPath.contains(entry)) //
				.collect(Collectors.toList());
		if (!missingBundeClassPath.isEmpty()) {
			LOGGER.warn("Missing entries in Bundle-ClassPath in MANIFEST.MF: {}", missingBundeClassPath);
		}

		var binIncludes = parseBinIncludes(projectPath.resolve(Paths.get("build.properties")));
		var missingBinIncludes = pathEntries //
				.stream() //
				.filter(entry -> !binIncludes.contains(entry)) //
				.collect(Collectors.toList());
		if (!missingBinIncludes.isEmpty()) {
			LOGGER.warn("Missing entries in bin.includes in build.properties: {}", missingBundeClassPath);
		}
	}

	private static String getArtifactFilename(Artifact artifact) {
		// only use artifact ID, without group and version
		return artifact.getArtifactId() + "." + artifact.getExtension();
	}

	// TODO - which is the proper exception type?
	// MojoExecutionException vs. MojoFailureException

	static List<String> parseBundleClassPath(Path manifest) throws MojoExecutionException {
		Manifest parsedManifest;
		try {
			parsedManifest = new Manifest(new FileInputStream(manifest.toFile()));
		} catch (IOException e) {
			throw new MojoExecutionException("Could not read " + manifest);
		}
		var bundleClassPath = parsedManifest.getMainAttributes().getValue("Bundle-ClassPath");
		if (bundleClassPath == null) {
			throw new MojoExecutionException("Bundle-ClassPath is missing");
		}
		var classPathEntries = bundleClassPath.split(",");
		return Arrays.asList(classPathEntries);
	}

	static List<String> parseBinIncludes(Path buildProperties) throws MojoExecutionException {
		var properties = new Properties();
		try {
			properties.load(new FileInputStream(buildProperties.toFile()));
		} catch (IOException e) {
			throw new MojoExecutionException("Could not read " + buildProperties);
		}
		var binIncludes = properties.getProperty("bin.includes");
		if (binIncludes == null) {
			throw new MojoExecutionException("bin.includes is missing");
		}
		var binIncludeEntries = binIncludes.split(",");
		return Arrays.asList(binIncludeEntries);
	}

}
