/*
 * Decompiled with CFR 0.152.
 */
package de.pirckheimer_gymnasium.engine_pi;

import de.pirckheimer_gymnasium.engine_pi.Camera;
import de.pirckheimer_gymnasium.engine_pi.DebugInfo;
import de.pirckheimer_gymnasium.engine_pi.Scene;
import de.pirckheimer_gymnasium.engine_pi.Vector;
import de.pirckheimer_gymnasium.engine_pi.annotations.Internal;
import de.pirckheimer_gymnasium.engine_pi.event.EventListeners;
import de.pirckheimer_gymnasium.engine_pi.event.FrameUpdateListener;
import de.pirckheimer_gymnasium.engine_pi.graphics.RenderTarget;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

public final class GameLoop {
    private static final Color COLOR_FPS_BACKGROUND = new Color(255, 255, 255, 50);
    private static final Color COLOR_FPS_BORDER = new Color(0, 106, 214);
    private static final Color COLOR_BODY_COUNT_BORDER = new Color(0, 214, 84);
    private static final Color COLOR_BODY_COUNT_BACKGROUND = new Color(255, 255, 255, 50);
    private static final int DEBUG_INFO_HEIGHT = 20;
    private static final int DEBUG_INFO_LEFT = 10;
    private static final int DEBUG_INFO_TEXT_OFFSET = 16;
    private static final Color DEBUG_GRID_COLOR = new Color(255, 255, 255, 100);
    private static final int GRID_SIZE_IN_PIXELS = 150;
    private static final int GRID_SIZE_METER_LIMIT = 100000;
    private static final int DEBUG_TEXT_SIZE = 12;
    private static final double DESIRED_FRAME_DURATION = 0.016;
    private static final int NANOSECONDS_PER_SECOND = 1000000000;
    private final ExecutorService threadPoolExecutor = Executors.newCachedThreadPool();
    private final RenderTarget render;
    private final Supplier<Scene> currentScene;
    private final Supplier<Boolean> isDebug;
    private final Queue<Runnable> dispatchableQueue = new ConcurrentLinkedQueue<Runnable>();
    private final EventListeners<FrameUpdateListener> frameUpdateListeners = new EventListeners();
    private double frameDuration;

    public GameLoop(RenderTarget render, Supplier<Scene> currentScene, Supplier<Boolean> isDebug) {
        this.render = render;
        this.currentScene = currentScene;
        this.isDebug = isDebug;
    }

    public void enqueue(Runnable runnable) {
        this.dispatchableQueue.add(runnable);
    }

    public void run() {
        this.frameDuration = 0.016;
        long frameStart = System.nanoTime();
        while (!Thread.currentThread().isInterrupted()) {
            Scene scene = this.currentScene.get();
            try {
                double deltaSeconds = Math.min(0.032, this.frameDuration);
                scene.step(deltaSeconds, this.threadPoolExecutor::submit);
                this.frameUpdateListeners.invoke(listener -> listener.onFrameUpdate(deltaSeconds));
                scene.getCamera().onFrameUpdate();
                scene.invokeFrameUpdateListeners(deltaSeconds);
                Runnable runnable = this.dispatchableQueue.poll();
                while (runnable != null) {
                    runnable.run();
                    runnable = this.dispatchableQueue.poll();
                }
                this.render();
                long frameEnd = System.nanoTime();
                double duration = (double)(frameEnd - frameStart) / 1.0E9;
                if (duration < 0.016) {
                    try {
                        Thread.sleep((int)(1000.0 * (0.016 - duration)));
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        break;
                    }
                }
                frameEnd = System.nanoTime();
                this.frameDuration = (double)(frameEnd - frameStart) / 1.0E9;
                frameStart = frameEnd;
            }
            catch (InterruptedException e) {
                break;
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        this.threadPoolExecutor.shutdown();
        try {
            this.threadPoolExecutor.awaitTermination(3L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            return;
        }
    }

    public EventListeners<FrameUpdateListener> getFrameUpdateListener() {
        return this.frameUpdateListeners;
    }

    public void render(RenderTarget renderTarget) {
        renderTarget.render(this::render);
    }

    private void render() {
        this.render.render(this::render);
    }

    @Internal
    private void render(Graphics2D g, int width, int height) {
        Scene scene = this.currentScene.get();
        g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        g.setColor(scene.getBackgroundColor());
        g.fillRect(0, 0, width, height);
        g.setClip(0, 0, width, height);
        AffineTransform transform = g.getTransform();
        scene.render(g, width, height);
        g.setTransform(transform);
        if (this.isDebug.get().booleanValue()) {
            this.renderGrid(g, scene, width, height);
            this.renderInfo(g, new DebugInfo(this.frameDuration, this.currentScene.get().getWorldHandler().getWorld().getBodyCount()));
        }
        g.dispose();
    }

    @Internal
    private void renderGrid(Graphics2D g, Scene scene, int width, int height) {
        AffineTransform pre = g.getTransform();
        Camera camera = scene.getCamera();
        Vector position = camera.getPosition();
        double rotation = -camera.getRotation();
        g.setClip(0, 0, width, height);
        g.translate(width / 2, height / 2);
        double pixelPerMeter = camera.getMeter();
        g.rotate(Math.toRadians(rotation), 0.0, 0.0);
        g.translate(-position.getX() * pixelPerMeter, position.getY() * pixelPerMeter);
        int gridSizeInMeters = (int)Math.round(150.0 / pixelPerMeter);
        double gridSizeInPixels = (double)gridSizeInMeters * pixelPerMeter;
        double gridSizeFactor = gridSizeInPixels / (double)gridSizeInMeters;
        if (gridSizeInMeters > 0 && gridSizeInMeters < 100000) {
            int x;
            int windowSizeInPixels = Math.max(width, height);
            int startX = (int)(position.getX() - (double)(windowSizeInPixels / 2) / pixelPerMeter);
            int startY = (int)(-1.0 * position.getY() - (double)(windowSizeInPixels / 2) / pixelPerMeter);
            startX -= startX % gridSizeInMeters + gridSizeInMeters;
            startY -= startY % gridSizeInMeters + gridSizeInMeters;
            int stopX = (int)((double)(startX -= gridSizeInMeters) + (double)windowSizeInPixels / pixelPerMeter + (double)(gridSizeInMeters * 2));
            int stopY = (int)((double)startY + (double)windowSizeInPixels / pixelPerMeter + (double)(gridSizeInMeters * 2));
            g.setFont(new Font("Monospaced", 0, 12));
            g.setColor(DEBUG_GRID_COLOR);
            for (x = startX; x <= stopX; x += gridSizeInMeters) {
                g.fillRect((int)((double)x * gridSizeFactor) - 1, (int)((double)(startY - 1) * gridSizeFactor), 2, (int)((double)windowSizeInPixels + 3.0 * gridSizeInPixels));
            }
            for (int y = startY; y <= stopY; y += gridSizeInMeters) {
                g.fillRect((int)((double)(startX - 1) * gridSizeFactor), (int)((double)y * gridSizeFactor - 1.0), (int)((double)windowSizeInPixels + 3.0 * gridSizeInPixels), 2);
            }
            for (x = startX; x <= stopX; x += gridSizeInMeters) {
                for (int y = startY; y <= stopY; y += gridSizeInMeters) {
                    g.drawString(x + " / " + -y, (int)((double)x * gridSizeFactor + 5.0), (int)((double)y * gridSizeFactor - 5.0));
                }
            }
        }
        g.setTransform(pre);
    }

    @Internal
    private void renderInfo(Graphics2D g, DebugInfo debugInfo) {
        double frameDuration = debugInfo.getFrameDuration();
        int actorCount = debugInfo.getBodyCount();
        Font displayFont = new Font("Monospaced", 0, 12);
        FontMetrics fm = g.getFontMetrics(displayFont);
        int y = 10;
        String fpsMessage = "FPS: " + (Serializable)(frameDuration == 0.0 ? "\u221e" : Long.valueOf(Math.round(1.0 / frameDuration)));
        Rectangle2D bounds = fm.getStringBounds(fpsMessage, g);
        g.setColor(COLOR_FPS_BORDER);
        g.fillRect(10, y, (int)bounds.getWidth() + 20, (int)bounds.getHeight() + 16);
        g.setColor(COLOR_FPS_BACKGROUND);
        g.drawRect(10, y, (int)bounds.getWidth() + 20 - 1, (int)bounds.getHeight() + 16 - 1);
        g.setColor(Color.WHITE);
        g.setFont(displayFont);
        g.drawString(fpsMessage, 20, y + 8 + fm.getHeight() - fm.getDescent());
        String actorMessage = "Actors: " + actorCount;
        bounds = fm.getStringBounds(actorMessage, g);
        g.setColor(COLOR_BODY_COUNT_BORDER);
        g.fillRect(10, y += fm.getHeight() + 20, (int)bounds.getWidth() + 20, (int)bounds.getHeight() + 16);
        g.setColor(COLOR_BODY_COUNT_BACKGROUND);
        g.drawRect(10, y, (int)bounds.getWidth() + 20 - 1, (int)bounds.getHeight() + 16 - 1);
        g.setColor(Color.WHITE);
        g.setFont(displayFont);
        g.drawString(actorMessage, 20, y + 8 + fm.getHeight() - fm.getDescent());
    }
}

