/*
 * Copyright (c) 2019 Richard Hauswald - https://quantummaid.de/.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package de.quantummaid.httpmaid.chains.graph;

import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.RequiredArgsConstructor;
import lombok.ToString;

import static de.quantummaid.httpmaid.chains.graph.Color.BLACK;
import static de.quantummaid.httpmaid.chains.graph.Label.textLabel;
import static de.quantummaid.httpmaid.util.Validators.validateNotNull;
import static de.quantummaid.httpmaid.util.Validators.validateNotNullNorEmpty;
import static java.lang.String.format;

@ToString
@EqualsAndHashCode
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public final class Node {
    private final String name;
    private final Label label;
    private final Color color;

    public static Node node(final String name) {
        return node(name, BLACK);
    }

    public static Node node(final String name,
                            final Color color) {
        return node(name, textLabel(name), color);
    }

    public static Node node(final String name,
                            final Label label,
                            final Color color) {
        validateNotNullNorEmpty(name, "name");
        validateNotNull(label, "label");
        validateNotNull(color, "color");
        return new Node(name, label, color);
    }

    String plot() {
        return format("%s [label=%s; color=\"%s\"; shape=\"box\"];%n", name, label.plot(), color.color());
    }

    String name() {
        return name;
    }
}
