/*
 * Copyright (c) 2020 Richard Hauswald - https://quantummaid.de/.
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package de.quantummaid.httpmaid.undertow;

import de.quantummaid.httpmaid.HttpMaid;
import de.quantummaid.httpmaid.endpoint.RawHttpRequest;
import de.quantummaid.httpmaid.endpoint.RawHttpRequestBuilder;
import de.quantummaid.httpmaid.http.Headers;
import de.quantummaid.httpmaid.http.HeadersBuilder;
import io.undertow.server.HttpHandler;
import io.undertow.server.HttpServerExchange;
import io.undertow.util.HeaderMap;
import io.undertow.util.HttpString;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.RequiredArgsConstructor;
import lombok.ToString;

import java.io.InputStream;
import java.io.OutputStream;

import static de.quantummaid.httpmaid.http.HeadersBuilder.headersBuilder;

@ToString
@EqualsAndHashCode
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public final class UndertowHandler implements HttpHandler {
    private final HttpMaid httpMaid;

    public static UndertowHandler undertowHandler(final HttpMaid httpMaid) {
        return new UndertowHandler(httpMaid);
    }

    @Override
    public void handleRequest(final HttpServerExchange httpServerExchange) {
        if (httpServerExchange.isInIoThread()) {
            httpServerExchange.dispatch(this);
            return;
        }
        httpMaid.handleRequest(() -> {
            final RawHttpRequestBuilder builder = RawHttpRequest.rawHttpRequestBuilder();

            final Headers headers = extractHeaders(httpServerExchange);
            builder.withHeaders(headers);

            final String path = httpServerExchange.getRequestPath();
            builder.withPath(path);

            final HttpString requestMethod = httpServerExchange.getRequestMethod();
            builder.withMethod(requestMethod.toString());

            final String queryString = httpServerExchange.getQueryString();
            builder.withQueryString(queryString);

            httpServerExchange.startBlocking();
            final InputStream body = httpServerExchange.getInputStream();
            builder.withBody(body);

            return builder.build();
        }, response -> {
            final int status = response.status();
            httpServerExchange.setStatusCode(status);

            final HeaderMap responseHeaders = httpServerExchange.getResponseHeaders();
            response.headers().forEach((name, values) ->
                    responseHeaders.putAll(HttpString.tryFromString(name), values));

            final OutputStream outputStream = httpServerExchange.getOutputStream();
            response.streamBodyToOutputStream(outputStream);
        });
    }

    private static Headers extractHeaders(final HttpServerExchange httpServerExchange) {
        final HeadersBuilder headersBuilder = headersBuilder();
        final HeaderMap requestHeaders = httpServerExchange.getRequestHeaders();
        requestHeaders.forEach(header -> {
            final String name = header.getHeaderName().toString();
            headersBuilder.withAdditionalHeader(name, header);
        });
        return headersBuilder.build();
    }
}
