/**
 * Copyright 2018 Karl-Philipp Richter
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.richtercloud.selenium.tools;

import org.apache.commons.lang3.ArrayUtils;
import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.jupiter.api.Assertions;
import org.openqa.selenium.WebDriver;

public final class SeleniumAssertionHelper {

    public static <T> void assertThat(final T actual,
            final Matcher condition,
            final SeleniumHelper seleniumHelper,
            final WebDriver webDriver,
            final String description,
            final WaitExceptionAction... waitExceptionActions) throws WebDriverWaitException {
        if (condition == null) {
            throw new IllegalArgumentException("condition mustn't be null");
        }
        if (seleniumHelper == null) {
            throw new IllegalArgumentException("seleniumHelper mustn't be null");
        }
        if (webDriver == null) {
            throw new IllegalArgumentException("webDriver mustn't be null");
        }
        if (description == null) {
            throw new IllegalArgumentException("description mustn't be null");
        }
        if (waitExceptionActions == null || waitExceptionActions.length == 0) {
            throw new IllegalArgumentException("waitExceptionActions mustn't be null or empty");
        }
        if (ArrayUtils.contains(waitExceptionActions, null)) {
            throw new IllegalArgumentException("waitExceptionActions mustn't contain null");
        }
        try {
            MatcherAssert.assertThat(actual, condition);
        } catch (AssertionError err) {
            for(WaitExceptionAction waitExceptionAction : waitExceptionActions) {
                waitExceptionAction.perform(seleniumHelper, webDriver, description);
            }
            throw err;
        }
    }

    public static void assertEqualsHtml(final String expected, final String actual) {
        if(expected == null) {
            throw new IllegalArgumentException("expected mustn't be null");
        }
        if(actual == null) {
            throw new IllegalArgumentException("actual mustn't be null");
        }
        Assertions.assertEquals(expected.replaceAll("[ ]+", " "),
                actual.replaceAll("[ ]+", " "));
    }

    public static Matcher<String> equalToHtml(final String expected) {
        return new IsEqualHtml(expected);
    }

    private static class IsEqualHtml extends BaseMatcher<String> {
        private final String expectedValue;

        public IsEqualHtml(final String equalArg) {
            super();
            this.expectedValue = equalArg;
        }

        @Override
        public boolean matches(Object actualValue) {
            if (expectedValue == null && actualValue == null) {
                return true;
            }
            if (!(actualValue instanceof String)) {
                return false;
            }
            String actualValueCast = (String) actualValue;
            return areEqual(actualValueCast, expectedValue);
        }

        @Override
        public void describeTo(Description description) {
            description.appendValue(expectedValue);
        }

        private static boolean areEqual(String actual, String expected) {
            if (actual == null) {
                return expected == null;
            }
            return actual.replaceAll("[ ]+", " ").equals(expected);
        }
    }

    private SeleniumAssertionHelper() {
    }
}
