/**
 * Copyright 2018 Karl-Philipp Richter
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.richtercloud.selenium.tools;

import de.richtercloud.jsf.validation.service.*;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.zip.GZIPOutputStream;
import javax.imageio.ImageIO;
import javax.xml.parsers.ParserConfigurationException;
import org.apache.commons.collections4.iterators.PermutationIterator;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import static org.hamcrest.CoreMatchers.*;
import org.hamcrest.Matcher;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.openqa.selenium.By;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.support.ui.ExpectedCondition;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.FluentWait;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;
import ru.yandex.qatools.ashot.AShot;
import ru.yandex.qatools.ashot.Screenshot;
import ru.yandex.qatools.ashot.shooting.ShootingStrategies;

/*
internal implementation notes:
- passing browser at every method avoid handling of initialization of @Drone
fields
- most methods are non-static because they involve taking screenshots on
failures which involves non-static screenshot counter
*/
/**
 * Provides helper methods for Selenium functional tests.
 *
 * Wraps counters for screenshots and other states which require initialization.
 *
 * @author richter
 */
public class SeleniumHelper {
    private final static Logger LOGGER = LoggerFactory.getLogger(SeleniumHelper.class);
    private final static String SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME = "screenshotOutputMode";
    private final static String SCREENSHOT_OUTPUT_MODE_FILE = "file";
    private final static String SCREENSHOT_OUTPUT_MODE_BASE64 = "base64";
    /**
     * Compresses images in gzip format and provides the base64-encoded result
     * of the encryption in order to reduce log size on continuous integration
     * services.
     */
    private final static String SCREENSHOT_OUTPUT_MODE_BASE64_GZIP = "base64-gzip";
    private final static String NO_DESCRIPTION = "no description";
    private static final String DOCTYPE_TEMPLATE = "<!DOCTYPE html>";
    private static final String WHITESPACE_TEMPLATE = "\\s+";
    public final static String SCREENSHOT_START_MARKER_DEFAULT = "---";
    public final static String SCREENSHOT_END_MARKER_DEFAULT = "---";
    public final static String PAGE_SOURCE_START_MARKER_DEFAULT = "===";
    public final static String PAGE_SOURCE_END_MARKER_DEFAULT = "===";
    public final static int WEB_DRIVER_WAIT_TIMEOUT_DEFAULT = 5;
    private static final String LOG_DESCRIPTION_PAGE_SOURCE = "browser page source";
    private static final String LOG_DESCRIPTION_DOM_TREE = "DOM tree";

    /**
     * Accounts for the fact that some Selenium driver implementations don't include the doctype in their page source
     * return value, but we might want to evaluate the driver's page source rather than the real page's source.
     */
    public static final Matcher MISSING_DOCTYPE_MESSAGE_MATCHER = new MessageValidatorMessageMatcher("Start tag seen " +
            "without seeing a doctype first. Expected “<!DOCTYPE html>”.");
    private final File screenshotDir;
    private int screenshotCounter;
    /**
     * The marker added in a new line before logging the screenshot in any mode
     * which involves logging. Has no effect on screenshot file output mode. Can
     * be set to the empty string, but then an empty line is still added to the
     * output. Must never be {@code null}.
     */
    private final String screenshotStartMarker;
    /**
     * The marker added in a new line after logging the screenshot in any mode
     * which involves logging. See {@link #screenshotStartMarker} for further
     * details.
     */
    private final String screenshotEndMarker;
    /**
     * The marker added in a new line before logging the page source. See
     * {@link #screenshotStartMarker} for further details.
     */
    private final String pageSourceStartMarker;
    /**
     * The marker added in a new line after logging the page source. See
     * {@link #screenshotStartMarker} for further details.
     */
    private final String pageSourceEndMarker;
    /**
     * The time in seconds for
     * {@link #webDriverWait(org.openqa.selenium.WebDriver, java.util.function.Function) }
     * and {@link #webDriverWait(org.openqa.selenium.WebDriver, java.util.function.Function, java.lang.String) }
     * to wait for the passed condition.
     */
    private final int webDriverWaitTimeout;

    private final ValidationService validationService;

    /**
     * Creates a new {@code SeleniumHelper} using the defaults of
     * {@link #SeleniumHelper(int) } and
     * {@link #WEB_DRIVER_WAIT_TIMEOUT_DEFAULT} for the web driver wait timeout
     * @throws IOException if an I/O exception occured during creation of
     *     temporary directory for screenshot
     */
    public SeleniumHelper() throws IOException {
        this(WEB_DRIVER_WAIT_TIMEOUT_DEFAULT);
    }

    /**
     * Creates a new {@code SeleniumHelper} using default values
     * {@link #SCREENSHOT_START_MARKER_DEFAULT},
     * {@link #SCREENSHOT_END_MARKER_DEFAULT},
     * {@link #PAGE_SOURCE_START_MARKER_DEFAULT},
     * {@link #PAGE_SOURCE_END_MARKER_DEFAULT} and a temporary screenshot
     * directory
     * @param webDriverWaitTimeout the web driver wait timeout
     * @throws IOException if an I/O exception occured during creation of
     *     temporary directory for screenshot
     */
    public SeleniumHelper(int webDriverWaitTimeout) throws IOException {
        this(SCREENSHOT_START_MARKER_DEFAULT,
                SCREENSHOT_END_MARKER_DEFAULT,
                PAGE_SOURCE_START_MARKER_DEFAULT,
                PAGE_SOURCE_END_MARKER_DEFAULT,
                webDriverWaitTimeout,
                Files.createTempDirectory("selenium-helper-screenshots").toFile());
    }

    public SeleniumHelper(File screenshotDir) throws IOException {
        this(SCREENSHOT_START_MARKER_DEFAULT,
                SCREENSHOT_END_MARKER_DEFAULT,
                PAGE_SOURCE_START_MARKER_DEFAULT,
                PAGE_SOURCE_END_MARKER_DEFAULT,
                WEB_DRIVER_WAIT_TIMEOUT_DEFAULT,
                screenshotDir);
    }

    public SeleniumHelper(int webDriverWaitTimeout,
            File screenshotDir) throws IOException {
        this(SCREENSHOT_START_MARKER_DEFAULT,
                SCREENSHOT_END_MARKER_DEFAULT,
                PAGE_SOURCE_START_MARKER_DEFAULT,
                PAGE_SOURCE_END_MARKER_DEFAULT,
                webDriverWaitTimeout,
                screenshotDir);
    }

    /**
     * Creates a new {@code SeleniumHelper}.
     * @param screenshotStartMarker the start marker for logging text-encoded
     *     screenshots
     * @param screeenshotEndMarker the end marker for logging text-encoded
     *     screenshots
     * @param pageSourceStartMarker the start marker to make finding page logs
     *     easier
     * @param pageSourceEndMarker the end marker to make finding page logs
     *     easier
     * @param webDriverWaitTimeout the web driver wait timeout
     * @param screenshotDir the directory where to store screenshots (will be
     *     created if it doesn't exist and has to be an empty directory in case the
     *     path exists)
     * @throws IOException if an I/O exception occurs during the optional
     *     creation of {@code screenshotDir}
     * @throws IllegalArgumentException if {@code screenshotStartMarker},
     *     {@code screeenshotEndMarker}, {@code pageSourceStartMarker},
     *     {@code pageSourceEndMarker} or {@code screenshotDir} are {@code null}
     */
    public SeleniumHelper(String screenshotStartMarker,
            String screeenshotEndMarker,
            String pageSourceStartMarker,
            String pageSourceEndMarker,
            int webDriverWaitTimeout,
            File screenshotDir) throws IOException {
        this(screenshotStartMarker,
                screeenshotEndMarker,
                pageSourceStartMarker,
                pageSourceEndMarker,
                webDriverWaitTimeout,
                screenshotDir,
                new MemoryValidationService(true));
    }

    protected SeleniumHelper(String screenshotStartMarker,
            String screeenshotEndMarker,
            String pageSourceStartMarker,
            String pageSourceEndMarker,
            int webDriverWaitTimeout,
            File screenshotDir,
            ValidationService validationService) throws IOException {
        if(screenshotStartMarker == null) {
            throw new IllegalArgumentException("screenshotStartMarker mustn't be null");
        }
        this.screenshotStartMarker = screenshotStartMarker;
        if(screeenshotEndMarker == null) {
            throw new IllegalArgumentException("screeenshotEndMarker mustn't be null");
        }
        this.screenshotEndMarker = screeenshotEndMarker;
        if(pageSourceStartMarker == null) {
            throw new IllegalArgumentException("pageSourceStartMarker mustn't be null");
        }
        this.pageSourceStartMarker = pageSourceStartMarker;
        if(pageSourceEndMarker == null) {
            throw new IllegalArgumentException("pageSourceEndMarker mustn't be null");
        }
        this.pageSourceEndMarker = pageSourceEndMarker;
        if(screenshotDir == null) {
            throw new IllegalArgumentException("screenshotDir mustn't be null");
                //could trigger creation, but avoids accidental passing of null
                //like this
        }
        if(!screenshotDir.exists()) {
            Files.createDirectories(screenshotDir.toPath());
        }else if(screenshotDir.exists()) {
            if(!screenshotDir.isDirectory()) {
                throw new IllegalArgumentException(String.format("screenshot "
                        + "directory '%s' points to an existing path which is not a"
                        + " directory",
                        screenshotDir.getAbsolutePath()));
            }else if(screenshotDir.list().length != 0) {
                throw new IllegalArgumentException(String.format("screenshot "
                        + "directory '%s' points to an existing directory which is not empty",
                        screenshotDir.getAbsolutePath()));
            }
        }
        this.screenshotDir = screenshotDir;
        LOGGER.info(String.format("screenshot directory is '%s'",
                screenshotDir.getAbsolutePath()));
        this.webDriverWaitTimeout = webDriverWaitTimeout;
        this.validationService = validationService;
    }

    /**
     * Wrapper around random string generation which allows to quickly avoid
     * temporary problems like Chrome Selenium driver not accepting unicode
     * characters outside Basic Multilingual Plane (BMP) (asked
     * https://stackoverflow.com/questions/46950897/how-to-produce-random-basic-multilingual-plane-bmp-strings-in-java
     * for inputs)
     *
     * @param length the length of the produced string
     * @return the produced random string
     */
    public static String randomString(int length) {
        if(length <= 0) {
            throw new IllegalArgumentException("length mustn't be <= 0");
        }
        return RandomStringUtils.randomAlphanumeric(length);
    }

    public String retrieveMessagesText(WebDriver browser,
            WebElement messages) throws WebDriverWaitException {
        return retrieveMessagesText(browser,
                messages,
                null, //expectedText (null means no comparison to expected text)
                false //waitForExpectedText
        );
    }

    /**
     * Retrieves the text value of the nested elements inside a
     * {@code p:messages}.
     *
     * @param browser the web driver reference to use
     * @param messages the web element carrying the messages id whose child
     *     elements will be accessed to retrieve the message
     * @param expectedText allows to optionally wait until the nested element
     *     containing the text has the specified value if it's not {@code null}
     * @return the messages text ({@code expectedText} in case it's not
     *     {@code null}
     * @throws IOException if an I/O Exception during taking a screenshot on not
     *     finding expected elements occurs
     */
    @SuppressWarnings("PMD.PreserveStackTrace")
    private String retrieveMessagesText(WebDriver browser,
            WebElement messages,
            String expectedText,
            boolean waitForExpectedText) throws WebDriverWaitException {
        LOGGER.trace(String.format("retrieveMessagesText expectedText: %s",
                expectedText));
        LOGGER.trace(String.format("retrieveMessagesText messages.text: %s",
                messages.getText()));
        webDriverWait(browser, ExpectedConditions.visibilityOf(messages));
        //- approaches to query the li element which contains the different
        //messages in order to increase flexibility towards checking the
        //presence for different messages has been replaced with this approach
        //based on `WebElement.getText` which is easier and allows to check for
        //different message order with permutations of the text
        //- waiting for the text to appear is unnecessary and leads to extreme
        //delay when querying a large set of text permutations because wrong
        //innerHTML is probably a phantomjs bug and the waiting doesn't work
        //around it
        if(expectedText != null) {
            if(waitForExpectedText) {
                try {
                    webDriverWait(browser,
                            ExpectedConditions.textToBePresentInElement(messages,
                                    expectedText));
                }catch(TimeoutException ex) {
                    throw new MessageTextNotContainedException(expectedText,
                            messages.getText());
                }
            }
            if(!expectedText.equals(messages.getText())) {
                throw new MessageTextNotContainedException(expectedText,
                        messages.getText());
                    //re-using the exception internally
            }
        }
        String retValue = messages.getText();
        assert expectedText == null || expectedText.equals(retValue):
                String.format("expectedText '%s' is not equal to retValue '%s'",
                        expectedText,
                        retValue);
        return retValue;
    }

    /**
     * Allows to retrieve the text parameter passed to
     * {@link ExpectedConditions#textToBePresentInElement(org.openqa.selenium.WebElement, java.lang.String) }
     * as well as test the type which is useful for unit testing.
     */
    protected class TextToBePresentInElementWithText implements ExpectedCondition<Boolean> {
        private final WebElement element;
        private final String text;

        protected TextToBePresentInElementWithText(WebElement element, String text) {
            this.element = element;
            this.text = text;
        }

        public String getText() {
            return text;
        }

        @Override
        public Boolean apply(WebDriver driver) {
            return ExpectedConditions.textToBePresentInElement(element, text).apply(driver);
        }

        @Override
        public String toString() {
            return ExpectedConditions.textToBePresentInElement(element, text).toString();
        }
    }

    public static String retrievePlainText(WebDriver browser) {
        WebElement preElement = browser.findElement(By.xpath("/html/body/pre"));
        return preElement.getText();
    }

    public void assertMessagesContains(WebDriver browser,
            WebElement messages,
            String messageText) throws WebDriverWaitException {
        assertMessagesContains(browser,
                messages,
                messageText,
                false //waitForMessageText
        );
    }

    /*
    internal implementation notes:
    - Don't use JUnit 5 Assertions.fail or other methods since they confuse
    jqwik tests
    */
    public void assertMessagesContains(WebDriver browser,
            WebElement messages,
            String messageText,
            boolean waitForMessageText) throws WebDriverWaitException {
        webDriverWait(browser,
                ExpectedConditions.visibilityOf(messages));
        String messagesText = retrieveMessagesText(browser,
                messages,
                messageText,
                waitForMessageText);
        assert messageText.equals(messagesText);
    }

    /**
     * Checks whether any string in {@code text} is contained in the
     * {@code p:messages} element {@code messages} using
     * {@link #webDriverWait(org.openqa.selenium.WebDriver, java.util.function.Function) }
     * and returns after the first match or fails using
     * {@link MessageTextNotContainedException} if none of {@code texts} match.
     *
     * Depending on the Selenium configuration this might a TimeoutException for
     * every non-matching string in {@code texts} which can be ignored.
     *
     * @param browser the web driver to use
     * @param messages the messages element to investigate
     * @param texts the texts to check
     * @throws WebDriverWaitException if an I/O exception occurs during taking of
     *     screenshots in case expected elements aren't present in {@code messages}
     * @throws IllegalArgumentException if {@code texts} is {@code null} or
     *     empty in order to be able to detect unexpected text conditions early
     */
    /*
    internal implementation notes:
    - Don't use JUnit 5 Assertions.fail or other methods since they confuse
    jqwik tests
    */
    public void assertMessagesContainsAny(WebDriver browser,
            WebElement messages,
            Set<String> texts) throws WebDriverWaitException {
        if(browser == null) {
            throw new IllegalArgumentException("browser mustn't be null");
        }
        if(messages == null) {
            throw new IllegalArgumentException("messages mustn't be null");
        }
        if(texts == null) {
            throw new IllegalArgumentException("texts mustn't be null");
        }
        if(texts.isEmpty()) {
            throw new IllegalArgumentException("texts mustn't be empty");
        }
        if(texts.contains(null)) {
            throw new IllegalArgumentException("texts mustn't contain null");
        }
        //run with wait for the first element in order to catch all AJAX updates
        //and then assume that all updates have been done and thus...
        Iterator<String> textsItr = texts.iterator();
        String text = textsItr.next();
        String messagesText;
        try {
            retrieveMessagesText(browser,
                    messages,
                    text,
                    true //waitForExpectedText
            );
            return;
        }catch(MessageTextNotContainedException ex) {
            messagesText = retrieveMessagesText(browser,
                    messages,
                    null,
                    false //waitForExpectedText
            );
        }
        //...check the rest of tests without wait
        while(textsItr.hasNext()) {
            text = textsItr.next();
            if(text.equals(messagesText)) {
                return;
            }
        }
        LOGGER.trace(String.format("messages.innerHTML: %s",
                messages.getAttribute("innerHTML")));
        throw new MessageTextNotContainedException(texts,
                messages.getText());
    }

    /**
     * Creates the permutation of {@code texts} which might be useful in
     * {@link #assertMessagesContainsAny(org.openqa.selenium.WebDriver, org.openqa.selenium.WebElement, java.util.Set) }
     * in order to reflect that validation constraint violoations are generally
     * in undefined order since {@code Validator.validate} returns a
     * {@code Set}.This methods creates {@code n} faculty ({@code n!}) items for {@code n}
     * items in {@code text} which can take a long time to check (five items can
     * already take up to a minute).
     *
     * Note that the type of the collection used for {@code texts} determines
     * whether the return value contains the item more than once or not.
     *
     * An example for steam fetichist (involving SteamEx stream extension):
     * {@code
     * createMessagePermutation(StreamEx.of(entities.stream())
     *                  .map(prop -> String.format("invalid: key '%s' mustn't be mapped to null",
     *                          prop.getName()))
     *                  .append("invalid: mapping for this entity isn't valid. Details should have been given to you.")
     *                  .collect(Collectors.toSet()))
     *          .stream()
     *  .map(perm -> StreamEx.of(perm)
     *          .joining("\n",
     *                  String.format("%s\n"
     *                          + "The following constraints are violated:\n",
     *                          BackingBean.X_FAILED_ERROR_SUMMARY),
     *                  String.format("\nFix the corresponding values in the components.")))
     *              //joining isn't necessary because \n could have been
     *              //added to each of the permutation items, this was
     *              //used as a way to get to know the awesome StreamEx
     *              //library
     *  .collect(Collectors.toSet());
     * }
     *
     * @param texts the texts to permutate
     * @return the created permutation
     */
    public static Set<List<String>> createMessagePermutation(Collection<String> texts) {
        Set<List<String>> retValue = new HashSet<>();
        PermutationIterator<String> textsPermutationIterator = new PermutationIterator<>(texts);
        textsPermutationIterator.forEachRemaining(perm -> retValue.add(perm));
        return retValue;
    }

    public static Set<List<String>> createMessagePermutation(String... texts) {
        return createMessagePermutation(new HashSet<>(Arrays.asList(texts)));
    }

    private String compressGzipAndEncodeBytes(byte[] screenshotBytes) throws IOException {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        GZIPOutputStream gZIPOutputStream = new GZIPOutputStream(byteArrayOutputStream);
        ByteArrayInputStream screenshotBytesInputStream = new ByteArrayInputStream(screenshotBytes);
        IOUtils.copy(screenshotBytesInputStream,
                gZIPOutputStream);
        byte[] compressedScreenshotBytes = byteArrayOutputStream.toByteArray();
        return Base64.getEncoder().encodeToString(compressedScreenshotBytes);
    }

    /**
     * Takes a screenshot with description {@link #NO_DESCRIPTION}.
     *
     * @param browser the browser to take the screenshot from.
     * @throws WebDriverWaitException if the storing method throws an I/O exception
     * @see #screenshot(org.openqa.selenium.WebDriver, java.lang.String) how the
     *     screenshot is taken
     */
    public void screenshot(WebDriver browser) throws WebDriverWaitException {
        screenshot(browser,
                NO_DESCRIPTION);
    }

    /**
     * Takes a screenshot of {@code browser} and stores it according to the
     * value of the system property
     * {@link #SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME} while managing the
     * numbering and description in file names.
     *
     * @param browser the browser to take the screenshot from
     * @param description the description to use in the file name (mustn't be
     *     {@code null}, but is allowed to be empty because the {@code null} check
     *     is only performed in order to guard again unwanted {@code null} passing)
     * @throws WebDriverWaitException if the storing method throws an I/O exception
     * @throws IllegalArgumentException if {@code browser} or
     *     {@code description} is {@code null}
     */
    public void screenshot(WebDriver browser,
            String description) throws WebDriverWaitException {
        if(browser == null) {
            throw new IllegalArgumentException("browser mustn't be null");
        }
        if(!(browser instanceof TakesScreenshot)) {
            LOGGER.debug("browser doesn't support taking screenshots");
            return;
        }
        if(description == null) {
            throw new IllegalArgumentException("description mustn't be null");
        }
        String screenshotOutputMode = System.getProperty(SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME,
                SCREENSHOT_OUTPUT_MODE_FILE);
        assert screenshotOutputMode != null;
        switch(screenshotOutputMode) {
            case SCREENSHOT_OUTPUT_MODE_FILE:
                File scrFile = ((TakesScreenshot)browser).getScreenshotAs(OutputType.FILE);
                try {
                    FileUtils.copyFile(scrFile, new File(screenshotDir, generateScreenshotFilename(screenshotCounter, description)));
                }catch(IOException ex) {
                    throw new WebDriverWaitException(ex);
                }
                break;
            case SCREENSHOT_OUTPUT_MODE_BASE64:
                String screenshotBase64 = ((TakesScreenshot)browser).getScreenshotAs(OutputType.BASE64);
                LOGGER.info(String.format("base64 encoding of screenshot with description '%s':\n%s\n%s\n%s",
                        description,
                        screenshotStartMarker,
                        screenshotBase64,
                        screenshotEndMarker));
                break;
            case SCREENSHOT_OUTPUT_MODE_BASE64_GZIP:
                byte[] screenshotBytes = ((TakesScreenshot)browser).getScreenshotAs(OutputType.BYTES);
                String compressedScreenshotBase64;
                try {
                    compressedScreenshotBase64 = compressGzipAndEncodeBytes(screenshotBytes);
                }catch(IOException ex) {
                    throw new WebDriverWaitException(ex);
                }
                LOGGER.info(String.format("gzip compresed base64 encoding of screenshot with description '%s':\n%s\n%s\n%s",
                        description,
                        screenshotStartMarker,
                        compressedScreenshotBase64,
                        screenshotEndMarker));
                break;
            default:
                throw new IllegalArgumentException(String.format("Illegal "
                        + "value '%s' specified for property %s",
                        screenshotOutputMode,
                        SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME));
        }
        screenshotCounter++;
    }

    public void screenshotFullPage(WebDriver browser) throws WebDriverWaitException {
        screenshotFullPage(browser,
                NO_DESCRIPTION);
    }

    public void screenshotFullPage(WebDriver browser,
            String description) throws WebDriverWaitException {
        String screenshotOutputMode = System.getProperty(SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME,
                SCREENSHOT_OUTPUT_MODE_FILE);
        assert screenshotOutputMode != null;
        Screenshot fpScreenshot;
        switch(screenshotOutputMode) {
            case SCREENSHOT_OUTPUT_MODE_FILE:
                fpScreenshot = new AShot().shootingStrategy(ShootingStrategies.viewportPasting(1000)).takeScreenshot(browser);
                File outputFile = new File(screenshotDir, generateScreenshotFilename(screenshotCounter, description));
                LOGGER.info("screenshot output filename: {}",
                        outputFile.getName());
                try {
                    ImageIO.write(fpScreenshot.getImage(),"PNG", outputFile);
                }catch(IOException ex) {
                    throw new WebDriverWaitException(ex);
                }
                break;
            case SCREENSHOT_OUTPUT_MODE_BASE64:
                fpScreenshot = new AShot().shootingStrategy(ShootingStrategies.viewportPasting(1000)).takeScreenshot(browser);
                ByteArrayOutputStream imageOutputStream = new ByteArrayOutputStream();
                try {
                    ImageIO.write(fpScreenshot.getImage(),"PNG",imageOutputStream);
                }catch(IOException ex) {
                    throw new WebDriverWaitException(ex);
                }
                String screenshotBase64 = Base64.getEncoder().encodeToString(imageOutputStream.toByteArray());
                LOGGER.info(String.format("base64 encoding of screenshot with description '%s':\n%s\n%s\n%s",
                        description,
                        screenshotStartMarker,
                        screenshotBase64,
                        screenshotEndMarker));
                break;
            case SCREENSHOT_OUTPUT_MODE_BASE64_GZIP:
                fpScreenshot = new AShot().shootingStrategy(ShootingStrategies.viewportPasting(1000)).takeScreenshot(browser);
                imageOutputStream = new ByteArrayOutputStream();
                try {
                    ImageIO.write(fpScreenshot.getImage(), "PNG", imageOutputStream);
                    String compressedScreenshotBase64 = compressGzipAndEncodeBytes(imageOutputStream.toByteArray());
                    LOGGER.info(String.format("gzip compresed base64 encoding of screenshot with description '%s':\n%s\n%s\n%s",
                            description,
                            screenshotStartMarker,
                            compressedScreenshotBase64,
                            screenshotEndMarker));
                }catch(IOException ex) {
                    throw new WebDriverWaitException(ex);
                }
                break;
            default:
                throw new IllegalArgumentException(String.format("Illegal "
                        + "value '%s' specified for property %s",
                        screenshotOutputMode,
                        SCREENSHOT_OUTPUT_MODE_PROPERTY_NAME));
        }
        screenshotCounter++;
    }

    private String generateScreenshotFilename(int screenshotCounter,
            String description) {
        return String.format("%05d-%s.png", screenshotCounter, description);
    }

    public void validatePageSource(String pageSource,
            ValidatorMessageSeverity minimumLevel,
            Matcher<ValidatorMessage> ignore) throws SAXException,
            IOException,
            PageSourceInvalidException {
        List<ValidatorMessage> violations = validationService.validateMessages(new ByteArrayInputStream(pageSource.getBytes()),
                minimumLevel,
                ignore);
        if(!violations.isEmpty()) {
            throw new PageSourceInvalidException(ValidationService.transformValidatorResponse(violations));
        }
    }

    public void validatePageSource(String pageSource,
                                   ValidatorMessageSeverity minimumLevel) throws SAXException,
            IOException,
            PageSourceInvalidException {
        validatePageSource(pageSource, minimumLevel, MISSING_DOCTYPE_MESSAGE_MATCHER);
    }

    public void validatePageSource(String pageSource) throws SAXException,
            IOException,
            PageSourceInvalidException {
        validatePageSource(pageSource, ValidatorMessageSeverity.WARNING);
    }

    public void logPageSource(WebDriver browser,
            Logger logger) {
        logPageSource(LOG_DESCRIPTION_PAGE_SOURCE,
                browser.getPageSource(),
                logger);
    }

    public void logPageSourcePretty(WebDriver browser,
            Logger logger) throws WebDriverWaitException {
        logPageSourcePretty(browser,
                logger,
                ValidatorMessageSeverity.WARNING,
                not(any(ValidatorMessage.class)));
    }

    /**
     * Logs the page source retrieved from {@code browser} to {@code logger} as
     * info message.
     *
     * @param browser the web driver to get the page source from
     * @param logger the logger to send the info message to
     * @param validationMinimumLevel only validation message with this level can
     *     cause the page source to be considered invalid
     * @param ignore allows to specify specific messages which ought to be
     *     ignored or criteria to ignore a number of messages
     * @throws WebDriverWaitException wraps exceptions during logging, validation and setup of validation
     */
    public void logPageSourcePretty(WebDriver browser,
            Logger logger,
            ValidatorMessageSeverity validationMinimumLevel,
            Matcher<ValidatorMessage> ignore) throws WebDriverWaitException {
        String pageSource = browser.getPageSource();
        try {
            validatePageSource(pageSource,
                    validationMinimumLevel,
                    ignore);
            //always validate since there's no sense in passing invalid XHTML to
            //the pretty printer
        }catch(SAXException | IOException | PageSourceInvalidException ex) {
            throw new WebDriverWaitException(ex);
        }
        String pageSourcePretty = prettyPrintPageSource(pageSource);
        logPageSource(LOG_DESCRIPTION_PAGE_SOURCE,
                pageSourcePretty,
                logger);
    }

    public void logPageSourceURLPretty(URL url,
            Logger logger,
            ValidatorMessageSeverity validationMinimumLevel,
            Matcher<ValidatorMessage> ignore) throws IOException,
            SAXException,
            PageSourceInvalidException {
        String pageSource = IOUtils.toString(url,
                Charset.defaultCharset());
        validatePageSource(pageSource,
                validationMinimumLevel,
                ignore);
            //always validate since there's no sense in passing invalid XHTML to
            //the pretty printer
        String pageSourcePretty = prettyPrintPageSource(pageSource);
        logPageSource(LOG_DESCRIPTION_PAGE_SOURCE,
                pageSourcePretty,
                logger);
    }

    private void logPageSource(String description,
            String pageSource,
            Logger logger) {
        logger.info(String.format("%s:\n%s\n%s\n%s",
                description,
                pageSourceStartMarker,
                pageSource,
                pageSourceEndMarker));
    }

    /**
     * Logs the current state of the DOM tree by executing
     * {@code return document.childNodes[1].outerHTML;} in {@code browser}.
     *
     * @param browser the web driver to use
     * @param logger the target logger
     * @throws javax.xml.parsers.ParserConfigurationException if a parser
     *     configuration error occurs during setup of pretty printing
     *     dependencies
     * @throws org.xml.sax.SAXException if a SAX exception occurs during setup
     *     of pretty printing dependencies
     * @throws IOException if an I/O error occurs
     */
    public void logDOMTree(JavascriptExecutor browser,
            Logger logger) throws ParserConfigurationException,
            SAXException,
            IOException {
        String htmlOuterHtml = (String) browser.executeScript("return document.childNodes[1].outerHTML;");
            //asked
            //https://stackoverflow.com/questions/49692420/how-to-get-a-org-w3c-doc-document-or-node-reference-with-selenium-in-java
            //for how to obtain a org.w3c.dom.Document or Node reference
            //directly
        String pageSourcePretty = prettyPrintPageSource(htmlOuterHtml);
        logPageSource(LOG_DESCRIPTION_DOM_TREE,
                pageSourcePretty,
                logger);
    }

    private String prettyPrintPageSource(String pageSource) {
        //Pretty printer choice:
        //- JTidy is unmaintained and hard to configure to do nothing else but
        //pretty printing
        //- `LSSerializer` together with
        //`com.sun.org.apache.xerces.internal.dom.DOMImplementationSourceImpl`
        //(from
        //http://www.java2s.com/Tutorials/Java/XML_HTML_How_to/DOM/Pretty_print_XML.htm
        //) causes
        //`org.xml.sax.SAXParseException; lineNumber: 2; columnNumber: 39; The element type "link" must be terminated by the matching end-tag "</link>".`
        //for valid XHTML
        Document doc = Jsoup.parse(pageSource);
        doc.outputSettings().prettyPrint(true);
        String pageSourcePretty = doc.html();
                //- `LSSerializer` changes `<head></head>` to `<head/>`, but no
                //manipuation except whitespace ought to take place, asked
                //https://stackoverflow.com/questions/49555441/how-to-avoid-html-being-manipulated-during-pretty-printing-with-lsserializer
                //for input
        if(StringUtils.startsWithIgnoreCase(pageSource, DOCTYPE_TEMPLATE)) {
            assert StringUtils.startsWithIgnoreCase(pageSource, DOCTYPE_TEMPLATE);
            //check for `<html` instead of `<html>` in order to include HTML
            //elements with attributes like `xmlns`
            assert pageSource.contains("<html"):
                    String.format("pageSource doesn't contain '<html': %s",
                            pageSource);
            pageSource.substring(pageSource.indexOf("<html")).replaceAll(WHITESPACE_TEMPLATE, "")
                    .equals(pageSourcePretty.substring(pageSourcePretty.indexOf("<html")).replaceAll(WHITESPACE_TEMPLATE, ""));
        }else {
            pageSource.replaceAll(WHITESPACE_TEMPLATE, "")
                    .equals(pageSourcePretty.replaceAll(WHITESPACE_TEMPLATE, ""));
        }
            //- Matchers EqualToIgnoringWhiteSpace (Hamcrest 1.3) and
            //EqualsToCompressingWhitespace (renamed in 2.0) don't work on text
            //with newlines since they replace whitespace sequences with one
            //space instead of removing it (therefore custom implementation of
            //IsEqualIgnoringWhiteSpace has been created, see below)
            //- Since 2.0.0.0's IsEqualCompressingWhiteSpace only does
            //`String.replaceAll("\\s+", "")` there's need to depend on Hamcrest
        return pageSourcePretty;
    }

    public <V> void webDriverWait(WebDriver browser,
            Function<? super WebDriver, V> condition) throws WebDriverWaitException {
        webDriverWait(browser,
                condition,
                NO_DESCRIPTION);
    }

    public <V> void webDriverWait(WebDriver browser,
            Function<? super WebDriver, V> condition,
            String exceptionScreenshotDescription) throws WebDriverWaitException {
        webDriverWait(browser,
                condition,
                exceptionScreenshotDescription,
                webDriverWaitTimeout);
    }

    /**
     * Creates a {@link WebDriverWait} and allows the timeout configuration in
     * this instance to be overridden (while other configuration parameters of
     * this instance can still be used).
     * @param <V> the type of the function
     * @param browser the web driver to use
     * @param condition the condition to wait for
     * @param exceptionScreenshotDescription the timeout screenshot description
     * @param overrideWebDriverWaitTimout the wait timeout
     * @throws WebDriverWaitException in case taking the failure screenshot causes such an
     *     exception
     */
    @SuppressWarnings("PMD.UnnecessaryFullyQualifiedName")
    //suppress PMD.UnnecessaryFullyQualifiedName until https://github.com/pmd/pmd/issues/2029 is fixed
    public <V> void webDriverWait(WebDriver browser,
            Function<? super WebDriver, V> condition,
            String exceptionScreenshotDescription,
            int overrideWebDriverWaitTimout) throws WebDriverWaitException {
        webDriverWait(browser,
                condition,
                exceptionScreenshotDescription,
                overrideWebDriverWaitTimout,
                WaitExceptionActions.screenshot());
    }

    /**
     * Creates a {@link WebDriverWait} and allows the timeout configuration in
     * this instance to be overridden (while other configuration parameters of
     * this instance can still be used).
     * @param <V> the type of the function
     * @param browser the web driver to use
     * @param condition the condition to wait for
     * @param exceptionDescription the timeout screenshot description
     * @param waitExceptionActions the actions to take in case of an exception during wait
     * @throws WebDriverWaitException in case taking the failure screenshot causes such an
     *     exception
     */
    public <V> void webDriverWait(WebDriver browser,
                                  Function<? super WebDriver, V> condition,
                                  String exceptionDescription,
                                  WaitExceptionAction... waitExceptionActions) throws WebDriverWaitException {
        webDriverWait(browser,
                condition,
                exceptionDescription,
                webDriverWaitTimeout,
                waitExceptionActions);
    }

    /**
     * Creates a {@link WebDriverWait} and allows the timeout configuration in
     * this instance to be overridden (while other configuration parameters of
     * this instance can still be used).
     * @param <V> the type of the function
     * @param browser the web driver to use
     * @param condition the condition to wait for
     * @param exceptionDescription the timeout screenshot description
     * @param overrideWebDriverWaitTimout the wait timeout
     * @param waitExceptionActions the actions to take in case of an exception during wait
     * @throws WebDriverWaitException in case taking the failure screenshot causes such an
     *     exception
     */
    public <V> void webDriverWait(WebDriver browser,
                                  Function<? super WebDriver, V> condition,
                                  String exceptionDescription,
                                  int overrideWebDriverWaitTimout,
                                  WaitExceptionAction... waitExceptionActions) throws WebDriverWaitException {
        try {
            createWebDriverWait(browser,
                    overrideWebDriverWaitTimout).until(condition);
        }catch(TimeoutException ex) {
            LOGGER.trace(String.format("WebDriverWait timed out, performing actions %s, see nested "
                            + "exception for details",
                            waitExceptionActions),
                    ex);
            handleWaitException(browser,
                    String.format("web driver wait timeout exception for element '%s'",
                            exceptionDescription),
                    waitExceptionActions);
            throw ex;
        }catch(NoSuchElementException ex) {
            LOGGER.trace("NoSuchElementException occured, made screenshot, see "
                            + "nested exception for details",
                    ex);
            handleWaitException(browser,
                    String.format("web driver wait no-such-element exception for element '%s'",
                            exceptionDescription),
                    waitExceptionActions);
            throw ex;
        }
    }

    /* default */ void handleWaitException(WebDriver browser,
            String description,
            WaitExceptionAction... waitExceptionActions) throws WebDriverWaitException {
        for(WaitExceptionAction waitExceptionAction : waitExceptionActions) {
            waitExceptionAction.perform(this, browser, description);
        }
    }

    /**
     * Compares the response code of {@code uRLConnection} to
     * {@code responseCode}
     *
     * @param responseCode the expected response code
     * @param httpResponse the response to check
     * @throws IOException if an I/O Exception happens during constructions of
     *     the exception message
     * @throws ResponseCodeUnequalsException if the response code isn't the
     *     expected {@code responseCode}
     */
    /*
    internal implementation notes:
    - Don't use JUnit 5 Assertions.fail or other methods since they confuse
    jqwik tests
    */
    public static void assertResponseCodeEquals(int responseCode,
            HttpResponse httpResponse) throws IOException {
        if(responseCode != httpResponse.getStatusLine().getStatusCode()) {
            throw new ResponseCodeUnequalsException(httpResponse);
        }
    }

    /**
     * Compares the response code of {@code uRLConnection} to
     * {@code responseCode}
     *
     * @param responseCode the expected response code
     * @param uRLConnection the connection to check
     * @throws IOException if an I/O Exception happens during constructions of
     *     the exception message
     * @throws ResponseCodeUnequalsException if the response code isn't the
     *     expected {@code responseCode}
     */
    /*
    internal implementation notes:
    - Don't use JUnit 5 Assertions.fail or other methods since they confuse
    jqwik tests
    */
    public static void assertResponseCodeEquals(int responseCode,
            HttpURLConnection uRLConnection) throws IOException {
        if(responseCode != uRLConnection.getResponseCode()) {
            throw new ResponseCodeUnequalsException(uRLConnection);
        }
    }

    /**
     * Retrieves a specific menu item in the nested submenus opened based on the
     * specified indices assuming that all menu items are links.
     * @param slideMenu the slide menu element
     * @param indices the indices of submenus to open
     * @return the found web element
     */
    public static WebElement retrieveSlideMenuItem(WebElement slideMenu,
            int... indices) {
        String xPath = buildXPathForSlideMenuIndices(indices);
        return slideMenu.findElement(By.xpath(xPath));
    }

    /**
     * Retrieves all menu items in the nested submenus opened based on the
     * specified indices assuming that all menu items are links.
     * @param slideMenu the slide menu element
     * @param indices the indices of submenus to open
     * @return the found web elements
     */
    public static List<WebElement> retrieveSlideMenuItems(WebElement slideMenu,
            int... indices) {
        String xPath = buildXPathForSlideMenuIndices(indices);
        return slideMenu.findElements(By.xpath(xPath));
    }

    private static String buildXPathForSlideMenuIndices(int... indices) {
        StringBuilder xPathBuilder = new StringBuilder(128);
        xPathBuilder.append("div/div[1]/");
        for(int index : indices) {
            xPathBuilder.append(String.format("ul/li[%d]/",
                    index));
        }
        xPathBuilder.append('a');
        String xPath = xPathBuilder.toString();
        LOGGER.trace("buildXPathForSlideMenuIndices xPath: {}",
                xPath);
        return xPath;
    }

    /**
     * Allows to check for the absence of a web element. See
     * https://stackoverflow.com/questions/6533597/webdriver-how-to-check-if-an-page-object-web-element-exists
     * which is referencing
     * https://groups.google.com/forum/#!topic/webdriver/kJqbbLVo40E for an
     * explanation why this is a good if not the only way to perform the check.
     * @param webElement the web element whose absence to check
     * @return the created condition
     */
    public static ExpectedCondition<Boolean> absenceOfWebElement(WebElement webElement) {
        return (WebDriver f) -> {
            try {
                webElement.isDisplayed();
                return false;
            }catch(NoSuchElementException ex) {
                return true;
            }
                //there seems to be no method in ExpectedConditions which can
                //check absense since invisibility and even statenessOf fail due
                //to  `org.openqa.selenium.NoSuchElementException` which isn't
                //necessarily intuitive (all improvement requests regarding
                //documentation are bumped because Selenium devs think that the
                //existance of the W3C web driver specification makes Javadoc
                //unnecessary)
        };
    }

    /**
     * Factory method for tests which has to be used until support for mocking
     * constructors via instrumentalisation with PowerMockito works in JUnit 5
     * (see comments in SeleniumHelperTest.testAssertMessagesContainsAny for
     * details and an issue report link).
     *
     * @param webDriver the web driver to use
     * @return a new {@code FluentWait} reference
     */
    protected FluentWait<WebDriver> createFluentWait(WebDriver webDriver) {
        return new FluentWait<>(webDriver);
    }

    protected WebDriverWait createWebDriverWait(WebDriver webDriver,
            long timeOutInSeconds) {
        return new WebDriverWait(webDriver,
                timeOutInSeconds);
    }
}
