package de.rpgframework.jfx;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.ResourceBundle;
import java.util.function.Function;

import org.prelle.javafx.JavaFXConstants;

import de.rpgframework.ResourceI18N;
import de.rpgframework.genericrpg.data.ComplexDataItem;
import de.rpgframework.genericrpg.data.DataItem;
import de.rpgframework.genericrpg.data.DataItemValue;
import de.rpgframework.genericrpg.data.PageReference;
import de.rpgframework.genericrpg.modification.DataItemModification;
import de.rpgframework.genericrpg.modification.Modification;
import de.rpgframework.genericrpg.requirements.Requirement;
import de.rpgframework.jfx.attach.PDFViewerServiceFactory;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.geometry.Insets;
import javafx.scene.Node;
import javafx.scene.control.Label;
import javafx.scene.control.ScrollPane;
import javafx.scene.control.TextField;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.scene.text.TextFlow;

/**
 * @author prelle
 *
 */
public class GenericDescriptionVBox extends ADescriptionPane {

	private final static ResourceBundle RES = ResourceBundle.getBundle(GenericDescriptionVBox.class.getName());

	private Function<Requirement,String> requirementResolver;
	private Function<Modification,String> modificationResolver;
	private BooleanProperty showModificationsInDescription = new SimpleBooleanProperty(true);
	private BooleanProperty useScrollPane = new SimpleBooleanProperty(true);

	private TextFlow description;
	private TextField tfKey;
	private Label hdRequires;
	private Label requirements;
	private Label hdModifies;
	private Label modifications;

	protected VBox inner;

	//-------------------------------------------------------------------
	public GenericDescriptionVBox(Function<Requirement,String> requirementResolver, Function<Modification,String> modificationResolver) {
		this.requirementResolver = requirementResolver;
		this.modificationResolver= modificationResolver;
		if (requirementResolver==null) {
			requirementResolver = (req) -> req.toString();
		}
		if (modificationResolver==null) {
			modificationResolver = (req) -> req.toString();
		}

		initComponents();
		initLayout();
		initInteractivity();
	}

	//-------------------------------------------------------------------
	public GenericDescriptionVBox(Function<Requirement,String> requirementResolver, Function<Modification,String> modificationResolver, DataItem item) {
		this(requirementResolver, modificationResolver);
		setData(item);
	}

	//-------------------------------------------------------------------
	private void initComponents() {
//		descTitle = new Label();
		descTitle.setWrapText(true);
//		descTitle.getStyleClass().add(JavaFXConstants.STYLE_HEADING3);
//		descSources = new Label();
		descSources.setWrapText(true);

		description = new TextFlow();
		tfKey       = new TextField();
		tfKey.setVisible(false);
		tfKey.setManaged(false);
		requirements = new Label();
		requirements.setWrapText(true);
		modifications = new Label();
		modifications.setWrapText(true);
	}

	//-------------------------------------------------------------------
	protected void initLayout() {
		Label hdDescription = new Label(ResourceI18N.get(RES, "label.descr"));
		hdRequires    = new Label(ResourceI18N.get(RES, "label.requires"));
		hdModifies    = new Label(ResourceI18N.get(RES, "label.modifications"));
		hdDescription.getStyleClass().add(JavaFXConstants.STYLE_HEADING5);
		hdRequires.getStyleClass().add(JavaFXConstants.STYLE_HEADING5);
		hdModifies.getStyleClass().add(JavaFXConstants.STYLE_HEADING5);

		inner = new VBox(5,hdDescription, description, tfKey, hdModifies, modifications, hdRequires, requirements);
		VBox.setMargin(hdModifies, new Insets(10, 0, 0, 0));
		VBox.setMargin(hdRequires, new Insets(10, 0, 0, 0));

		setMaxHeight(Double.MAX_VALUE);
		setStyle("-fx-pref-width: 20em");
		setStyle("-fx-max-width: 30em");
		getChildren().addAll(descTitle, descSources, getWithOrWithoutScrollPane());
	}
	
	//-------------------------------------------------------------------
	private Node getWithOrWithoutScrollPane() {
		if (useScrollPane.get()) {
			ScrollPane scroll = new ScrollPane(inner);
			scroll.setFitToWidth(true);
			scroll.setMinHeight(200);
			scroll.setMaxHeight(Double.MAX_VALUE);
			VBox.setVgrow(scroll, Priority.ALWAYS);
			return scroll;
		} else {
			return inner;
		}
	}

	//-------------------------------------------------------------------
	protected void initInteractivity() {
		useScrollPaneProperty().addListener( (ov,o,n) -> {
			getChildren().setAll(descTitle, descSources, getWithOrWithoutScrollPane());
		});
	}

	//-------------------------------------------------------------------
	public void setData(DataItemValue<?> data) {
		if (data==null) {
			descTitle.setText(null);
			descSources.setText(null);
			requirements.setText(null);
			modifications.setText(null);
			return;
		}
		setData(data.getModifyable());
	}

	//-------------------------------------------------------------------
	public void setData(DataItem data) {
		if (data==null) {
			descTitle.setText(null);
			descSources.setText(null);
			requirements.setText(null);
			modifications.setText(null);
			return;
		}

		// Eventually open a PDF
		Optional<PageReference> optPageRef = data.getPageReferences()
				.stream()
				.filter(pr -> pr.getLanguage().equals(Locale.getDefault().getLanguage()))
				.findFirst();
		PageReference pageRef = optPageRef.isPresent()?optPageRef.get():null;
		if (pageRef!=null) {
			PDFViewerServiceFactory.create().ifPresent(service -> {
				service.show(pageRef.getProduct().getRules(), pageRef.getProduct().getID(),
						pageRef.getLanguage(), pageRef.getPage());
			});
		}

		descTitle.setText(data.getName(Locale.getDefault()));
		descSources.setText(RPGFrameworkJavaFX.createSourceText(data));
		tfKey.setVisible(!data.hasLicense(Locale.getDefault()));
		tfKey.setManaged(!data.hasLicense(Locale.getDefault()));
		tfKey.setText(data.getTypeString()+"."+data.getId().toLowerCase()+".desc");
		RPGFrameworkJavaFX.parseMarkupAndFillTextFlow(description, data.getDescription(Locale.getDefault()));

		// Eventually show requirements
		if ((data instanceof ComplexDataItem) && !((ComplexDataItem)data).getRequirements().isEmpty()) {
			hdRequires.setVisible(true);
			hdRequires.setManaged(true);
			requirements.setVisible(true);
			requirements.setManaged(true);
			List<String> list = new ArrayList<>();
			for (Requirement req : ((ComplexDataItem)data).getRequirements()) {
				if (requirementResolver!=null)
					list.add(requirementResolver.apply(req));
				else
					list.add(String.valueOf(req));
			}
			requirements.setText(String.join(", ", list));
		} else {
			hdRequires.setVisible(false);
			hdRequires.setManaged(false);
			requirements.setVisible(false);
			requirements.setManaged(false);
		}
		//Eventually show modifications
		if ((data instanceof ComplexDataItem) && !((ComplexDataItem)data).getOutgoingModifications().isEmpty() && showModificationsInDescription.get()) {
			hdModifies.setVisible(true);
			hdModifies.setManaged(true);
			modifications.setVisible(true);
			modifications.setManaged(true);
			List<String> list = new ArrayList<>();
			for (Modification req : ((ComplexDataItem)data).getOutgoingModifications()) {
				if ((req instanceof DataItemModification) && String.valueOf(((DataItemModification)req).getReferenceType()).equals("HOOK") ) continue;
				String toAdd = (modificationResolver!=null)
						?modificationResolver.apply(req)
						:String.valueOf(req);
				if (toAdd!=null) list.add(toAdd);
			}
			modifications.setText(String.join(", ", list));
		} else {
			hdModifies.setVisible(false);
			hdModifies.setManaged(false);
			modifications.setVisible(false);
			modifications.setManaged(false);
		}
	}

	//-------------------------------------------------------------------
	public void setData(String title, String source, String desc) {
		descTitle.setText(title);
		descSources.setText(source);
		RPGFrameworkJavaFX.parseMarkupAndFillTextFlow(description, desc);
		hdRequires.setVisible(false);
		hdRequires.setManaged(false);
		hdModifies.setVisible(false);
		hdModifies.setManaged(false);
		requirements.setVisible(false);
		requirements.setManaged(false);
		modifications.setVisible(false);
		modifications.setManaged(false);
	}

	//-------------------------------------------------------------------
	public void setResolver(Function<Requirement,String> requirementResolver) {
		this.requirementResolver = requirementResolver;
	}

	//-------------------------------------------------------------------
	public void setModificationResolver(Function<Modification,String> modResolver) {
		this.modificationResolver = modResolver;
	}

	//--------------------------------------------------------------------
	public BooleanProperty showModificationsInDescriptionProperty() { return showModificationsInDescription; }
	public Boolean  isShowModificationsInDescription() { return showModificationsInDescription.get(); }
	public GenericDescriptionVBox setShowModificationsInDescription(Boolean value) { showModificationsInDescription.set(value); return this; }

	//--------------------------------------------------------------------
	public BooleanProperty useScrollPaneProperty() { return useScrollPane; }
	public Boolean  isUseScrollPane() { return useScrollPane.get(); }
	public GenericDescriptionVBox setUseScrollPane(Boolean value) { useScrollPane.set(value); return this; }

}
