package de.rpgframework.jfx;

import java.lang.System.Logger;
import java.lang.System.Logger.Level;

import org.prelle.javafx.SymbolIcon;

import de.rpgframework.genericrpg.NumericalValue;
import de.rpgframework.genericrpg.NumericalValueController;
import javafx.beans.property.ObjectProperty;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.geometry.Pos;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.TextField;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.util.Callback;

public class NumericalValueField<T,V extends NumericalValue<T>> extends HBox {

	private final static Logger logger = System.getLogger("prelle.jfx");

	private V value;
	private ObjectProperty<NumericalValueController<T,V>> control;
	private Button dec;
	private Button inc;
	private TextField tfValue;
	private Label lbValue;
	private Callback<V, Integer> valueCallback;
	private Callback<V, String> converter;
	private int minWidthEm;
	private boolean useLabel;
	private Integer overrideValue;

	private EventHandler<ActionEvent> onAction = null;

	//--------------------------------------------------------------------
	public NumericalValueField() {
		initComponents();
		initLayout();
		initStyle();
		initInteractivity();
		dec.setDisable(true);
		inc.setDisable(true);
	}

	//--------------------------------------------------------------------
	public NumericalValueField(V val, ObjectProperty<NumericalValueController<T,V>> ctrl) {
		if (val==null)
			throw new NullPointerException("val may not be NULL");
		this.value = val;
		this.control = ctrl;
		initComponents();
		initLayout();
		initStyle();
		initInteractivity();
		refresh();
	}

	//--------------------------------------------------------------------
	public NumericalValueField(V val, ObjectProperty<NumericalValueController<T,V>> ctrl, int minWidthEm, boolean useLabel) {
		if (val==null)
			throw new NullPointerException("val may not be NULL");
		this.value = val;
		this.control = ctrl;
		this.minWidthEm   = minWidthEm;
		this.useLabel= useLabel;
		initComponents();
		initLayout();
		initStyle();
		initInteractivity();
		refresh();
	}

	//--------------------------------------------------------------------
	private void initComponents() {
		dec  = new Button(null, new SymbolIcon("remove"));
		inc  = new Button(null, new SymbolIcon("add"));
		tfValue = new TextField();
		tfValue.setPrefColumnCount(2);
		tfValue.setEditable(false);
		tfValue.setFocusTraversable(false);
		lbValue = new Label();
	}

	//--------------------------------------------------------------------
	private void initLayout() {
		getChildren().clear();
		setAlignment(Pos.CENTER);
		if (useLabel)
			getChildren().addAll(dec, lbValue, inc);
		else
			getChildren().addAll(dec, tfValue, inc);
		HBox.setHgrow(lbValue, Priority.SOMETIMES);
		HBox.setHgrow(tfValue, Priority.SOMETIMES);
	}

	//--------------------------------------------------------------------
	private void initStyle() {
		inc.getStyleClass().add("mini-button");
		dec.getStyleClass().add("mini-button");
		if (minWidthEm>0) {
//			setStyle("-fx-background-color: lime;");
			lbValue.setStyle("-fx-padding: 0 0.1em 0 0.1em; -fx-min-width: "+minWidthEm+"em; -fx-alignment:center");
		} else {
//			setStyle("-fx-background-color: pink;");
			lbValue.setStyle("-fx-padding: 0 1em 0 1em; -fx-pref-width: 3em");
		}
	}

	//--------------------------------------------------------------------
	private void initInteractivity() {
		dec.setOnAction(ev -> {
			control.get().decrease(value);
			refresh();
			if (onAction!=null) onAction.handle(ev);
		});
		inc.setOnAction(ev -> {
			control.get().increase(value);
			refresh();
			if (onAction!=null) onAction.handle(ev);
		});

		if (control!=null)
			control.addListener( (ov,o,n) -> refresh());

		tfValue.focusedProperty().addListener( (ov,o,n) -> {
			if (n) {
				ActionEvent ev = new ActionEvent(tfValue, null);
				if (onAction!=null) onAction.handle(ev);
			}
		});
	}

	//--------------------------------------------------------------------
	public void setData(V val, ObjectProperty<NumericalValueController<T,V>> ctrl) {
		this.value = val;
		this.control = ctrl;
		control.addListener( (ov,o,n) -> refresh());
		refresh();
	}

	//--------------------------------------------------------------------
	public void refresh() {
		if (control==null) {
			logger.log(Level.DEBUG, "Mising control for {0} in {1}", this, this.getParent().getParent());
			return;
		}
		if (control.get()==null || value==null)
			return;
		dec.setDisable(!control.get().canBeDecreased(value).get());
		inc.setDisable(!control.get().canBeIncreased(value).get());
		int num = control.get().getValue(value); // value.getDistributed();
		if (overrideValue!=null)
			num = overrideValue;
		if (valueCallback!=null)
			num = valueCallback.call(value);
		String text = String.valueOf(num);
		if (converter!=null) {
			text = converter.call(value);
		}
		tfValue.setText(text);
		lbValue.setText(text);

//		if (text!=null && text.length()>3)
//			lbValue.setStyle("-fx-font-size: smaller");
//		else
//			lbValue.setStyle(null);
	}

	//--------------------------------------------------------------------
	public int getInt() {
		try {
			return Integer.parseInt(tfValue.getText());
		} catch (NumberFormatException e) {
			return 0;
		}
	}

	//--------------------------------------------------------------------
	public void setItem(V val) {
		this.value = val;
		refresh();
	}

	//-------------------------------------------------------------------
	/**
	 * @param valueCallback the valueCallback to set
	 */
	public void setValueCallback(Callback<V, Integer> valueCallback) {
		this.valueCallback = valueCallback;
		initLayout();
	}

	//-------------------------------------------------------------------
	/**
	 * @param converter the converter to set
	 */
	public void setConverter(Callback<V,String> converter) {
		this.converter = converter;
	}

	//-------------------------------------------------------------------
	/**
	 * @param overrideValue the overrideValue to set
	 */
	public void setOverrideValue(Integer overrideValue) {
		this.overrideValue = overrideValue;
	}

	//-------------------------------------------------------------------
	/**
	 * @param onAction the onAction to set
	 */
	public void setOnAction(EventHandler<ActionEvent> onAction) {
		this.onAction = onAction;
	}

	//--------------------------------------------------------------------

}