package de.rpgframework.jfx;

import java.lang.System.Logger;
import java.lang.System.Logger.Level;
import java.util.ArrayList;
import java.util.List;
import java.util.PropertyResourceBundle;

import org.prelle.javafx.CloseType;
import org.prelle.javafx.ManagedDialog;
import org.prelle.javafx.NavigButtonControl;

import de.rpgframework.ResourceI18N;
import de.rpgframework.genericrpg.data.Choice;
import de.rpgframework.genericrpg.data.DataItem;
import de.rpgframework.genericrpg.modification.Modification;
import de.rpgframework.genericrpg.modification.ValueModification;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.scene.Parent;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.VBox;
import javafx.util.Callback;

/**
 * @author prelle
 *
 */
public class UserDistributeDialog extends ManagedDialog {

	public final static Logger logger = System.getLogger(RPGFrameworkJFXConstants.BASE_LOGGER_NAME);

	private static PropertyResourceBundle UI = RPGFrameworkJFXConstants.UI;

	private DataItem decideFor;
	private Choice choice;
	private Parent content;

	private Label label;
	private ListView<Integer> distView;
	private ListView<ValueModification> modView;
	private ListView<ValueModification> resultView;
	private Button btnCombine;

	private NavigButtonControl buttonControl;

	//-------------------------------------------------------------------
	/**
	 */
	public UserDistributeDialog(DataItem decideFor, Choice choice) {
		super(ResourceI18N.get(UI,"wizard.selectMod.distribute"), null, CloseType.OK);
		this.decideFor = decideFor;
		if (choice.getDistribute()==null || choice.getDistribute().length==0)
			throw new IllegalArgumentException("No points to distribute in choice");
		this.choice = choice;

		initComponents(decideFor.getName());
		initLayout();
		initInteractivity();

		buttonControl = new NavigButtonControl();
		buttonControl.setDisabled(CloseType.OK, true);
	}

	//-------------------------------------------------------------------
	private void initComponents(String choiceReason) {
		distView   = new ListView<Integer>();
		modView    = new ListView<ValueModification>();
		resultView = new ListView<ValueModification>();
		distView.getItems().addAll(List.of(choice.getDistribute()));
		
		// Convert choice into modifications
		List<ValueModification> mods = new ArrayList<>();
		for (String key : choice.getChoiceOptions()) {
			mods.add(new ValueModification(choice.getChooseFrom(), key, 0));
		}
		modView.getItems().addAll(mods);

		modView .getStyleClass().add("bordered");
		distView.getStyleClass().add("bordered");
		resultView .getStyleClass().add("bordered");

		modView.setCellFactory(new Callback<ListView<ValueModification>, ListCell<ValueModification>>() {
			public ListCell<ValueModification> call(ListView<ValueModification> list) {
				return new DistributeModificationCell();
			}}
				);
		resultView.setCellFactory(new Callback<ListView<ValueModification>, ListCell<ValueModification>>() {
			public ListCell<ValueModification> call(ListView<ValueModification> list) {
				return new ResultModificationCell();
			}}
				);


		btnCombine = new Button(ResourceI18N.get(UI,"button.combine"));
		btnCombine .getStyleClass().add("bordered");
		canBeCombined();

		/*
		 * Top
		 */
		label = new Label(choiceReason);
		label.getStyleClass().add("wizard-heading");
	}

	//-------------------------------------------------------------------
	private void initLayout() {
		modView .setStyle("-fx-pref-width: 13em; -fx-pref-height: 10em;");
		distView.setStyle("-fx-pref-width: 6em; -fx-pref-height: 10em;");
		resultView.setStyle("-fx-max-width: 28.4em; -fx-pref-height: 10em;");

		HBox bxLTR = new HBox(20);
		bxLTR.getChildren().addAll(modView, btnCombine, distView);
		bxLTR.setAlignment(Pos.CENTER_LEFT);
		VBox content = new VBox(20);
		content.getChildren().addAll(label, bxLTR, resultView);
		BorderPane.setMargin(resultView, new Insets(20,0,0,0));

		resultView.prefWidthProperty().bind(bxLTR.widthProperty());

		setTitle(ResourceI18N.get(UI,"wizard.selectMod.distribute"));
		setContent(content);
	}

	//-------------------------------------------------------------------
	private void initInteractivity() {
		// Button 'combine'
		btnCombine.setOnAction(event -> combine());

		// List selections
		modView.getSelectionModel().selectedItemProperty().addListener( (ov,o,n) -> canBeCombined() );
		distView.getSelectionModel().selectedItemProperty().addListener( (ov,o,n) -> canBeCombined() );
	}

	//-------------------------------------------------------------------
	public NavigButtonControl getButtonControl() {
		return buttonControl;
	}

	//-------------------------------------------------------------------
	public ValueModification[] getChoice() {
		ValueModification[] ret = new ValueModification[choice.getDistribute().length];
		ret = resultView.getItems().toArray(ret);
		return ret;
	}

	//-------------------------------------------------------------------
	private void canBeCombined() {
		btnCombine.setDisable( modView.getSelectionModel().getSelectedItem()==null ||
				distView.getSelectionModel().getSelectedItem()==null);
	}

	//-------------------------------------------------------------------
	private void combine() {
		Modification mod = modView.getSelectionModel().getSelectedItem();
		Integer  value   = distView.getSelectionModel().getSelectedItem();

		logger.log(Level.DEBUG, "Combine "+mod+" with "+value);
		if (mod instanceof ValueModification vMod) {
			vMod.setValue(value);
			modView.getItems().remove(mod);
			if (!distView.getItems().remove(value)) {
				throw new IllegalStateException("Could not remove "+value+" from list "+distView.getItems());
			}
			resultView.getItems().add(vMod);
			logger.log(Level.DEBUG, " mods = "+modView.getItems());
			logger.log(Level.DEBUG, " ints = "+distView.getItems());
			if (!distView.getItems().isEmpty())
				distView.getSelectionModel().select(0);
			if (!modView.getItems().isEmpty())
				modView.getSelectionModel().select(0);
			buttonControl.setDisabled(CloseType.OK, !distView.getItems().isEmpty());
		} else
			logger.log(Level.ERROR, "Unsupported modification type "+mod);
	}
}


//-------------------------------------------------------------------
class DistributeModificationCell extends ListCell<ValueModification> {

	private static Logger logger = UserDistributeDialog.logger;
	@Override
	public void updateItem(ValueModification item, boolean empty) {
		super.updateItem(item, empty);
		if (!empty) {
			setText( ((DataItem)item.getResolvedKey()).getName() );
		} else {
			setText(null);
			setGraphic(null);
		}
	}
}

//-------------------------------------------------------------------
class ResultModificationCell extends ListCell<ValueModification> {

	private static Logger logger = UserDistributeDialog.logger;
	@Override
	public void updateItem(ValueModification item, boolean empty) {
		super.updateItem(item, empty);
		if (empty) {
			setText(null);
			setGraphic(null);
		} else {
				setText(((DataItem)item.getResolvedKey()).getName()+" +"+item.getValue());
		}
	}
}