package org.nustaq.webserver;

import io.netty.handler.codec.http.websocketx.BinaryWebSocketFrame;
import io.netty.handler.codec.http.websocketx.TextWebSocketFrame;
import org.nustaq.netty2go.NettyWSHttpServer;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.*;
import io.netty.util.AttributeKey;
import io.netty.util.CharsetUtil;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.logging.Logger;

import static io.netty.handler.codec.http.HttpHeaders.Names.CONTENT_TYPE;
import static io.netty.handler.codec.http.HttpHeaders.setContentLength;
import static io.netty.handler.codec.http.HttpResponseStatus.NOT_FOUND;
import static io.netty.handler.codec.http.HttpResponseStatus.OK;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

/**
 * Created by ruedi on 25.05.14.
 */
public class WebSocketHttpServer {

    protected final AttributeKey<ClientSession> session = AttributeKey.valueOf("session");
    public static Logger logger = Logger.getLogger(WebSocketHttpServer.class.getName());

    protected File contentRoot;

    public WebSocketHttpServer(File contentRoot) {
        this.contentRoot = contentRoot;
    }

    //////////////////////////////////////////////////////////////////////////
    // callbacks from netty. FIXME: make interface

    public void onOpen(ChannelHandlerContext ctx) {
        ctx.attr(session).set(createNewSession());
        logger.info("onOpen: " + ctx.attr(session).get());
    }

    public void onClose(ChannelHandlerContext ctx) {
        logger.info("close session on:" + ctx.attr(session).get() + " closed.");
    }

    public void onHttpRequest(ChannelHandlerContext ctx, FullHttpRequest req, NettyWSHttpServer.HttpResponseSender sender) {
        logger.info("request on:" + ctx.attr(session));
        serveFile(ctx,req,sender);
    }

    public void onTextMessage(ChannelHandlerContext ctx, String text) {
        logger.info("text on:" + ctx.attr(session) + " '" + text + "'");
        sendWSTextMessage(ctx, text);
    }

    public void onBinaryMessage(ChannelHandlerContext ctx, byte[] buffer) {
        logger.info("binary on:" + ctx.attr(session));
    }

    // end callback from netty
    /////////////////////////////////////////////////////////////////////////////

    // rudimentary session management

    protected ClientSession getSession(ChannelHandlerContext ctx) {
        return ctx.attr(session).get();
    }

    protected ClientSession createNewSession() {
        ClientSession clientSession = new ClientSession(){};
        logger.info("created session "+clientSession);
        return clientSession;
    }

    // utils

    public void sendWSTextMessage( ChannelHandlerContext ctx, String s ) {
        ctx.channel().write(new TextWebSocketFrame(s));
    }

    public void sendWSBinarayMessage( ChannelHandlerContext ctx, byte b[], int off, int len ) {
        ctx.channel().writeAndFlush(new BinaryWebSocketFrame(Unpooled.wrappedBuffer(b, off, len)));
    }

    public void sendWSBinarayMessage( ChannelHandlerContext ctx, byte b[] ) {
        ctx.channel().writeAndFlush(new BinaryWebSocketFrame(Unpooled.wrappedBuffer(b, 0, b.length)));
    }

    public void sendHttpGetResponse(ChannelHandlerContext ctx, FullHttpRequest req, NettyWSHttpServer.HttpResponseSender sender, String response) {
        ByteBuf content = Unpooled.copiedBuffer(response, CharsetUtil.US_ASCII);

        FullHttpResponse res = new DefaultFullHttpResponse(HTTP_1_1, OK, content);
        res.headers().set(CONTENT_TYPE, "text/html; charset=UTF-8");
        setContentLength(res, content.readableBytes());

        sender.sendHttpResponse(ctx,req, res);
    }

    public void serveFile(ChannelHandlerContext ctx, FullHttpRequest req, NettyWSHttpServer.HttpResponseSender sender) {
        File target = new File(contentRoot, File.separator + req.getUri().toString() );
        if ( target.exists() && target.isFile() ) {
            FileChannel inChannel = null;
            RandomAccessFile aFile = null;
            try {
                aFile = new RandomAccessFile(target, "r");
                inChannel = aFile.getChannel();
                long fileSize = inChannel.size();
                ByteBuffer buffer = ByteBuffer.allocate((int) fileSize); // FIXME: reuse ..
                while( inChannel.read(buffer) > 0 ) {}
                buffer.flip();
                ByteBuf content = Unpooled.wrappedBuffer(buffer);
                FullHttpResponse res = new DefaultFullHttpResponse(HTTP_1_1, OK, content);
                res.headers().set(CONTENT_TYPE, "text/html; charset=UTF-8");
                setContentLength(res, content.readableBytes());
                sender.sendHttpResponse(ctx, req, res);
                return;
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                try {
                    if ( inChannel != null )
                        inChannel.close();
                    if ( aFile != null )
                            aFile.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            FullHttpResponse res = new DefaultFullHttpResponse(HTTP_1_1, NOT_FOUND);
            sender.sendHttpResponse(ctx, req, res);
        } else {
            FullHttpResponse res = new DefaultFullHttpResponse(HTTP_1_1, NOT_FOUND);
            sender.sendHttpResponse(ctx, req, res);
        }
    }

    // main

    public static void main(String[] args) throws Exception {
        int port;
        if (args.length > 0) {
            port = Integer.parseInt(args[0]);
        } else {
            port = 8887;
        }
        new NettyWSHttpServer(port, new WebSocketHttpServer(new File(".") )).run();
    }

}
