/*
 * Decompiled with CFR 0.152.
 */
package de.sayayi.lib.pack;

import de.sayayi.lib.pack.PackConfig;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UTFDataFormatException;
import java.util.Arrays;
import java.util.OptionalInt;
import java.util.zip.GZIPInputStream;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Range;

public class PackInputStream
implements Closeable {
    @NotNull
    private InputStream stream;
    private final boolean compressed;
    private final Integer version;
    private int bit = -1;
    private byte b;

    @Contract(mutates="param1,io")
    public PackInputStream(@NotNull InputStream stream) throws IOException {
        this(new PackConfig.Builder().withCompressionSupport().build(), stream);
    }

    @Contract(mutates="param2,io")
    public PackInputStream(@NotNull PackConfig packConfig, @NotNull InputStream stream) throws IOException {
        byte[] header;
        this.stream = stream;
        byte[] magic = packConfig.getMagic();
        int magicLength = magic.length;
        if (!(magicLength == 0 || stream.readNBytes(header = new byte[magicLength], 0, magicLength) == magicLength && Arrays.equals(header, magic))) {
            throw new IOException("pack stream has wrong header magic");
        }
        this.compressed = packConfig.isCompressionSupport() && this.readBoolean();
        int versionBits = packConfig.getVersionBits();
        this.version = versionBits != 0 ? (versionBits <= 8 ? Integer.valueOf(this.readSmallVar() + packConfig.getLowestVersionNumber()) : Integer.valueOf((int)(this.readLarge(versionBits) + (long)packConfig.getLowestVersionNumber()))) : null;
        if (this.compressed) {
            this.forceByteAlignment();
            this.stream = new GZIPInputStream(stream);
        }
    }

    @Contract(pure=true)
    public boolean isCompressed() {
        return this.compressed;
    }

    @Contract(pure=true)
    @NotNull
    public OptionalInt getVersion() {
        return this.version == null ? OptionalInt.empty() : OptionalInt.of(this.version);
    }

    @Contract(mutates="this,io")
    public boolean readBoolean() throws IOException {
        this.assertData();
        return (this.b & 1 << this.bit--) != 0;
    }

    @Contract(mutates="this,io")
    public void skipBoolean() throws IOException {
        this.assertData();
        --this.bit;
    }

    @Contract(mutates="this,io")
    @NotNull
    public <T extends Enum<T>> T readEnum(@NotNull Class<T> enumType, @Range(from=1L, to=16L) int bitWidth) throws IOException {
        if (bitWidth <= 0 || bitWidth > 16) {
            throw new IllegalArgumentException("Invalid bitWidth: " + bitWidth);
        }
        return (T)((Enum[])enumType.getEnumConstants())[bitWidth <= 8 ? this.readSmall(bitWidth) : (int)this.readLarge(bitWidth)];
    }

    @Contract(mutates="this,io")
    @NotNull
    public <T extends Enum<T>> T readEnum(@NotNull Class<T> enumType) throws IOException {
        Enum[] enums;
        int n = (enums = (Enum[])enumType.getEnumConstants()).length;
        int bits = Integer.bitCount(n | n >> 1 | n >> 2 | n >> 4 | n >> 8);
        return (T)enums[bits <= 8 ? this.readSmall(bits) : (int)this.readLarge(bits)];
    }

    @Contract(mutates="this,io")
    public <T extends Enum<T>> void skipEnum(@NotNull Class<T> enumType) throws IOException {
        Enum[] enums = (Enum[])enumType.getEnumConstants();
        int n = enums.length;
        this.skip(Integer.bitCount(n | n >> 1 | n >> 2 | n >> 4 | n >> 8));
    }

    @Contract(mutates="this,io")
    public @Range(from=0L, to=65535L) int readUnsignedShort() throws IOException {
        return (int)this.readLarge(16);
    }

    @Contract(mutates="this,io")
    public void skipUnsignedShort() throws IOException {
        this.skip(16);
    }

    @Contract(mutates="this,io")
    public int readInt() throws IOException {
        return (int)this.readLarge(32);
    }

    @Contract(mutates="this,io")
    public void skipInt() throws IOException {
        this.skip(32);
    }

    @Contract(mutates="this,io")
    public long readLong() throws IOException {
        return this.readLarge(64);
    }

    @Contract(mutates="this,io")
    public void skipLong() throws IOException {
        this.skip(64);
    }

    @Contract(mutates="this,io")
    public String readString() throws IOException {
        int utfBytesRemaining = 0;
        switch (this.readSmall(2)) {
            case 0: {
                return null;
            }
            case 1: {
                utfBytesRemaining = this.readSmall(4);
                if (utfBytesRemaining != 0) break;
                return "";
            }
            case 2: {
                utfBytesRemaining = this.readSmall(8);
                break;
            }
            case 3: {
                utfBytesRemaining = (int)this.readLarge(16);
            }
        }
        this.forceByteAlignment();
        char[] chars = new char[utfBytesRemaining];
        int charIdx = 0;
        while (utfBytesRemaining > 0) {
            byte b2;
            byte b1 = this.read();
            if ((b1 & 0x80) == 0) {
                --utfBytesRemaining;
                chars[charIdx++] = (char)b1;
                continue;
            }
            if ((b1 & 0xE0) == 192) {
                utfBytesRemaining -= 2;
                b2 = this.read();
                if ((b2 & 0xC0) != 128) {
                    throw new UTFDataFormatException();
                }
                chars[charIdx++] = (char)((b1 & 0x1F) << 6 | b2 & 0x3F);
                continue;
            }
            if ((b1 & 0xF0) == 224) {
                utfBytesRemaining -= 3;
                b2 = this.read();
                byte b3 = this.read();
                if ((b2 & 0xC0) != 128 || (b3 & 0xC0) != 128) {
                    throw new UTFDataFormatException();
                }
                chars[charIdx++] = (char)((b1 & 0xF) << 12 | (b2 & 0x3F) << 6 | b3 & 0x3F);
                continue;
            }
            throw new UTFDataFormatException();
        }
        return new String(chars, 0, charIdx);
    }

    @Contract(mutates="this,io")
    public void skipString() throws IOException {
        int utfLength = 0;
        switch (this.readSmall(2)) {
            case 0: {
                return;
            }
            case 1: {
                utfLength = this.readSmall(4);
                if (utfLength != 0) break;
                return;
            }
            case 2: {
                utfLength = this.readSmall(8);
                break;
            }
            case 3: {
                utfLength = (int)this.readLarge(16);
            }
        }
        this.forceByteAlignment();
        while (utfLength-- > 0) {
            this.read();
        }
    }

    @Contract(mutates="this,io")
    public @Range(from=0L, to=255L) int readSmallVar() throws IOException {
        int v4 = this.readSmall(4);
        if ((v4 & 8) == 0) {
            return v4;
        }
        if ((v4 & 4) == 0) {
            return v4 - 4 << 1 | (this.readBoolean() ? 1 : 0);
        }
        return (v4 & 3) << 6 | this.readSmall(6);
    }

    @Contract(mutates="this,io")
    public void skipSmallVar() throws IOException {
        int v4 = this.readSmall(4);
        if ((v4 & 8) != 0) {
            this.skip((v4 & 4) == 0 ? 1 : 6);
        }
    }

    @Contract(mutates="this,io")
    public @Range(from=0L, to=255L) int readSmall(@Range(from=1L, to=8L) int bitWidth) throws IOException {
        this.assertData();
        int bitsRemaining = this.bit + 1 - bitWidth;
        if (bitsRemaining > 0) {
            this.bit = bitsRemaining - 1;
            return this.b >> bitsRemaining & (1 << bitWidth) - 1;
        }
        if (bitsRemaining == 0) {
            this.bit = -1;
            return this.b & (1 << bitWidth) - 1;
        }
        int value = (this.b & (1 << this.bit + 1) - 1) << -bitsRemaining;
        this.bit = -1;
        this.assertData();
        this.bit = 7 + bitsRemaining;
        return value |= this.b >> 8 + bitsRemaining & (1 << -bitsRemaining) - 1;
    }

    @Contract(mutates="this,io")
    public long readLarge(@Range(from=9L, to=64L) int bitWidth) throws IOException {
        int c;
        this.assertData();
        long value = (long)this.b & (1L << this.bit + 1) - 1L;
        bitWidth -= this.bit + 1;
        while (bitWidth >= 8) {
            c = this.stream.read();
            if (c < 0) {
                throw new EOFException();
            }
            value = value << 8 | (long)c;
            bitWidth -= 8;
        }
        this.bit = -1;
        if (bitWidth > 0) {
            this.assertData();
            c = this.b >> 8 - bitWidth & (1 << bitWidth) - 1;
            value = value << bitWidth | (long)c;
            this.bit -= bitWidth;
        }
        return value;
    }

    @Contract(mutates="this,io")
    protected void assertData() throws IOException {
        if (this.bit < 0) {
            this.b = this.read();
            this.bit = 7;
        }
    }

    @Contract(mutates="this")
    protected void forceByteAlignment() {
        if (this.bit >= 0) {
            this.bit = -1;
        }
    }

    @Contract(mutates="this,io")
    public void skip(@Range(from=0L, to=0x7FFFFFFFL) int bitWidth) throws IOException {
        while (bitWidth > 0) {
            this.assertData();
            if (bitWidth <= this.bit + 1) {
                this.bit -= bitWidth;
                break;
            }
            bitWidth -= this.bit + 1;
            this.bit = -1;
        }
    }

    @Override
    public void close() throws IOException {
        this.stream.close();
    }

    @Contract(mutates="io")
    protected byte read() throws IOException {
        int c = this.stream.read();
        if (c < 0) {
            throw new EOFException("unexpected end of pack stream");
        }
        return (byte)c;
    }
}

