/*
 * Copyright (C) 2009-2011 Schlichtherle IT Services
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.schlichtherle.truezip.fs.archive.zip;

import de.schlichtherle.truezip.rof.ReadOnlyFile;
import de.schlichtherle.truezip.socket.InputSocket;
import de.schlichtherle.truezip.zip.CRC32Exception;
import edu.umd.cs.findbugs.annotations.DefaultAnnotation;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

/**
 * A {@link ZipInputShop} which checks the CRC-32 value for all ZIP entries.
 * The additional CRC-32 computation makes this class slower than its super
 * class.
 * <p>
 * If there is a mismatch of the CRC-32 values for a ZIP entry in an input
 * archive, the {@link InputStream#close()} method of the corresponding stream
 * for the archive entry will throw a {@link CRC32Exception}.
 * Other than this, the archive entry will be processed normally.
 * So if just the CRC-32 value for the entry in the archive file has been
 * modified, you can still read its entire contents.
 * 
 * @author Christian Schlichtherle
 * @version $Id$
 * @see CheckedZipDriver
 */
@DefaultAnnotation(NonNull.class)
public class CheckedZipInputShop extends ZipInputShop {
    
    public CheckedZipInputShop(ReadOnlyFile rof, ZipDriver driver)
    throws IOException {
        super(driver, rof);
    }

    /** Overridden to read from a checked input stream. */
    @Override
    public InputSocket<ZipArchiveEntry> getInputSocket(final String name) {
        if (null == name)
            throw new NullPointerException();

        class Input extends InputSocket<ZipArchiveEntry> {
            @Override
            public ZipArchiveEntry getLocalTarget() throws IOException {
                final ZipArchiveEntry entry = getEntry(name);
                if (null == entry)
                    throw new FileNotFoundException(name + " (entry not found)");
                return entry;
            }

            @Override
            public ReadOnlyFile newReadOnlyFile() throws IOException {
                throw new FileNotFoundException(name + " (unsupported operation)"); // TODO: Support this feature for STORED entries.
            }

            @Override
            public InputStream newInputStream() throws IOException {
                return CheckedZipInputShop.this.getInputStream(
                        getLocalTarget().getName(),
                        true,
                        !(getPeerTarget() instanceof ZipArchiveEntry));
            }
        } // class Input

        return new Input();
    }
}
