/*
 * Copyright (C) 2006-2011 Schlichtherle IT Services
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.schlichtherle.truezip.io;

import java.io.IOException;
import java.io.OutputStream;

/**
 * A decorator which synchronizes all access to an {@link OutputStream}
 * via an object provided to its constructor.
 *
 * @see SynchronizedInputStream
 * @author Christian Schlichtherle
 * @version $Id$
 */
public class SynchronizedOutputStream extends DecoratingOutputStream {

    /** The object to synchronize on - never {@code null}. */
    protected final Object lock;

    /**
     * Constructs a new synchronized output stream.
     * This object will synchronize on itself.
     *
     * @param out the output stream to wrap in this decorator.
     */
    public SynchronizedOutputStream(final OutputStream out) {
    	this(out, null);
    }

    /**
     * Constructs a new synchronized output stream.
     *
     * @param out the output stream to wrap in this decorator.
     * @param lock the object to synchronize on.
     *        If {@code null}, then this object is used, not the stream.
     */
    public SynchronizedOutputStream(final OutputStream out, final Object lock) {
        super(out);
        this.lock = null == lock ? this : lock;
    }

    @Override
	public void write(int b) throws IOException {
        synchronized (lock) {
            super.write(b);
        }
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        synchronized (lock) {
            super.write(b, off, len);
        }
    }

    @Override
    public void flush() throws IOException {
        synchronized (lock) {
            super.flush();
        }
    }

    @Override
    public void close() throws IOException {
        synchronized (lock) {
            super.close();
        }
    }
}
