/**
 * <h1>&#64;seeebiii/ses-email-forwarding</h1>
 * <p>
 * This <a href="https://aws.amazon.com/cdk/">AWS CDK</a> construct allows you to setup email forwarding mappings in <a href="https://aws.amazon.com/ses/">AWS SES</a> to receive emails from your domain and forward them to another email address.
 * All of this is possible without hosting your own email server, you just need a domain.
 * <p>
 * For example, if you own a domain <code>example.org</code> and want to receive emails for <code>hello&#64;example.org</code> and <code>privacy&#64;example.org</code>, you can forward emails to <code>whatever&#64;provider.com</code>.
 * This is achieved by using a Lambda function that forwards the emails using <a href="https://github.com/arithmetric/aws-lambda-ses-forwarder">aws-lambda-ses-forwarder</a>.
 * <p>
 * This construct is creating quite a few resources under the hood and can also automatically verify your domain and email addresses in SES.
 * Consider reading the <a href="#architecture">Architecture</a> section below if you want to know more about the details.
 * <p>
 * <h2>Examples</h2>
 * <p>
 * Forward all emails received under <code>hello&#64;example.org</code> to <code>whatever+hello&#64;provider.com</code>:
 * <p>
 * <blockquote><pre>
 * new EmailForwardingRuleSet(this, 'EmailForwardingRuleSet', {
 *   // make the underlying rule set the active one
 *   enableRuleSet: true,
 *   // define how emails are being forwarded
 *   emailForwardingProps: [{
 *     // your domain name you want to use for receiving and sending emails
 *     domainName: 'example.org',
 *     // a prefix that is used for the From email address to forward your mails
 *     fromPrefix: 'noreply',
 *     // a list of mappings between a prefix and target email address
 *     emailMappings: [{
 *       // the prefix matching the receiver address as &lt;prefix&gt;&#64;&lt;domainName&gt;
 *       receivePrefix: 'hello',
 *       // the target email address(es) that you want to forward emails to
 *       targetEmails: ['whatever+hello&#64;provider.com']
 *     }]
 *   }]
 * });
 * </pre></blockquote>
 * <p>
 * Forward all emails to <code>hello&#64;example.org</code> to <code>whatever+hello&#64;provider.com</code> and verify the domain <code>example.org</code> in SES:
 * <p>
 * <blockquote><pre>
 * new EmailForwardingRuleSet(this, 'EmailForwardingRuleSet', {
 *   emailForwardingProps: [{
 *     domainName: 'example.org',
 *     // let the construct automatically verify your domain
 *     verifyDomain: true,
 *     fromPrefix: 'noreply',
 *     emailMappings: [{
 *       receivePrefix: 'hello',
 *       targetEmails: ['whatever+hello&#64;provider.com']
 *     }]
 *   }]
 * });
 * </pre></blockquote>
 * <p>
 * If you don't want to verify your domain in SES or you are in the SES sandbox, you can still send emails to verified email addresses.
 * Use the property <code>verifyTargetEmailAddresses</code> in this case and set it to <code>true</code>.
 * <p>
 * For a full &amp; up-to-date reference of the available options, please look at the source code of  <a href="lib/email-forwarding-rule-set.ts"><code>EmailForwardingRuleSet</code></a> and <a href="lib/email-forwarding-rule.ts"><code>EmailForwardingRule</code></a>.
 * <p>
 * <h4>Note</h4>
 * <p>
 * Since the verification of domains requires to lookup the Route53 domains in your account, you need to define your AWS account and region.
 * You can do it like this in your CDK stack:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Object app = new App();
 * 
 * public class EmailForwardingSetupStack extends Stack {
 *     public EmailForwardingSetupStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public EmailForwardingSetupStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         EmailForwardingRuleSet.Builder.create(this, "EmailForwardingRuleSet").build();
 *     }
 * }
 * 
 * EmailForwardingSetupStack.Builder.create(app, "EmailForwardingSetupStack")
 *         .env(Map.of(
 *                 "account", "&lt;account-id&gt;",
 *                 "region", "&lt;region&gt;"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Use Cases</h2>
 * <p>
 * <ul>
 * <li>Build a landing page on AWS and offer an email address to contact you.</li>
 * <li>Use various aliases to register for different services and forward all mails to the same target email address.</li>
 * </ul>
 * <p>
 * There are probably more - happy to hear them :)
 * <p>
 * <h2>Install</h2>
 * <p>
 * <h3>npm</h3>
 * <p>
 * <blockquote><pre>
 * npm i -D &#64;seeebiii/ses-email-forwarding
 * </pre></blockquote>
 * <p>
 * Take a look at <a href="./package.json">package.json</a> to make sure you're installing the correct version compatible with your current AWS CDK version.
 * See more details on npmjs.com: https://www.npmjs.com/package/&#64;seeebiii/ses-email-forwarding
 * <p>
 * <h3>Maven</h3>
 * <p>
 * <blockquote><pre>
 * &lt;dependency&gt;
 *   &lt;groupId&gt;de.sebastianhesse.cdk-constructs&lt;/groupId&gt;
 *   &lt;artifactId&gt;ses-email-forwarding&lt;/artifactId&gt;
 *   &lt;version&gt;2.0.1&lt;/version&gt;
 * &lt;/dependency&gt;
 * </pre></blockquote>
 * <p>
 * See more details on mvnrepository.com: https://mvnrepository.com/artifact/de.sebastianhesse.cdk-constructs/ses-email-forwarding/
 * <p>
 * <h3>Python</h3>
 * <p>
 * <blockquote><pre>
 * pip install ses-email-forwarding
 * </pre></blockquote>
 * <p>
 * See more details on PyPi: https://pypi.org/project/ses-email-forwarding/
 * <p>
 * <h3>.NET / C#</h3>
 * <p>
 * An artifact is pushed up to NuGet.org: https://www.nuget.org/packages/Ses.Email.Forwarding/
 * <p>
 * <h4>Project Scaffolding &amp; Installation</h4>
 * <p>
 * <blockquote><pre>
 * # Create a new directory
 * mkdir ExampleApplication &amp;&amp; cd ExampleApplication
 * 
 * # Scaffold a C# CDK project
 * cdk init --language csharp
 * 
 * # Add dependencies
 * cd src/ExampleApplication
 * dotnet add package Ses.Email.Forwarding
 * dotnet add package Amazon.CDK.AWS.SNS.Subscriptions
 * 
 * # Remove example stack and global suppressions (silenced by way of using discards)
 * rm ExampleApplicationStack.cs GlobalSuppressions.cs
 * </pre></blockquote>
 * <p>
 * <h4>Example Usage</h4>
 * <p>
 * <blockquote><pre>
 * using Amazon.CDK;
 * using Amazon.CDK.AWS.SNS;
 * using Amazon.CDK.AWS.SNS.Subscriptions;
 * using SebastianHesse.CdkConstructs;
 * using Construct = Constructs.Construct;
 * 
 * namespace ExampleApplication
 * {
 *     public sealed class Program
 *     {
 *         public static void Main()
 *         {
 *             var app = new App();
 * 
 *             _ = new MailboxStack(app, nameof(MailboxStack), new StackProps
 *             {
 *                 Env = new Environment
 *                 {
 *                     // Replace with desired account
 *                     Account = "000000000000",
 * 
 *                     // Replace with desired region
 *                     Region = "us-east-1"
 *                 }
 *             });
 * 
 *             app.Synth();
 *         }
 *     }
 * 
 *     public sealed class MailboxStack : Stack
 *     {
 *         public MailboxStack(Construct scope, string id, IStackProps props = null) : base(scope, id, props)
 *         {
 *             var notificationTopic = new Topic(this, nameof(EmailForwardingProps.NotificationTopic));
 * 
 *             // 'Bounce' and 'Complaint' notification types, in association with the domain being verified, will be sent
 *             // to this email address
 *             notificationTopic.AddSubscription(new EmailSubscription("admin&#64;provider.com"));
 * 
 *             _ = new EmailForwardingRuleSet(this, nameof(EmailForwardingRuleSet), new EmailForwardingRuleSetProps
 *             {
 *                 EmailForwardingProps = new IEmailForwardingProps[]
 *                 {
 *                     new EmailForwardingProps
 *                     {
 *                         // If your domain name has already been verified as a domain identity in SES, this does not
 *                         // need to be toggled on
 *                         VerifyDomain = true,
 * 
 *                         // This is the prefix that will be used in the email address used to forward emails
 *                         FromPrefix = "noreply",
 * 
 *                         // This domain name will be used to send and receive emails
 *                         DomainName = "example.org",
 * 
 *                         // A list of mappings between a prefix and target email addresses
 *                         EmailMappings = new IEmailMapping[]
 *                         {
 *                             new EmailMapping
 *                             {
 *                                 // Emails received by hello&#64;example.org will be forwarded
 *                                 ReceivePrefix = "hello",
 * 
 *                                 // Emails will be forwarded to admin+hello&#64;provider.com
 *                                 TargetEmails = new []
 *                                 {
 *                                     "admin+hello&#64;provider.com"
 *                                 }
 *                             }
 *                         },
 * 
 *                         // This notification topic be published to when events in association with 'Bounce' and
 *                         // 'Complaint' notification types occur
 *                         NotificationTopic = notificationTopic
 *                     }
 *                 }
 *             });
 *         }
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Usage</h2>
 * <p>
 * This package provides two constructs: <a href="lib/email-forwarding-rule-set.ts"><code>EmailForwardingRuleSet</code></a> and <a href="lib/email-forwarding-rule.ts"><code>EmailForwardingRule</code></a>.
 * The <code>EmailForwardingRuleSet</code> is a wrapper around <code>ReceiptRuleSet</code> but adds a bit more magic to e.g. verify a domain or target email address.
 * Similarly, <code>EmailForwardingRule</code> is a wrapper around <code>ReceiptRule</code> but adds two SES rule actions to forward the email addresses appropriately.
 * <p>
 * This means if you want the full flexibility, you can use the <code>EmailForwardingRule</code> construct in your stack.
 * <p>
 * <h3>Sending E-Mails over SMTP</h3>
 * <p>
 * You can also send emails over SES using this construct because it provides the basics for sending emails: a verified SES domain or email address identity.
 * You need to do the following if you're using the <code>EmailForwardingRuleSetConstruct</code>:
 * <p>
 * <ol>
 * <li>Set the <code>verifyDomain</code> or <code>verifyTargetEmailAddresses</code> to <code>true</code>.</li>
 * <li><a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/smtp-credentials.html?icmpid=docs_ses_console">Create SMTP credentials in AWS SES</a> and save them somewhere.</li>
 * <li>Setup your email program or application to use the SMTP settings.</li>
 * </ol>
 * <p>
 * <h2>Architecture</h2>
 * <p>
 * The <code>EmailForwardingRuleSet</code> creates a <code>EmailForwardingRule</code> for each forward mapping.
 * Each rule contains an <code>S3Action</code> to store the incoming emails and a Lambda Function to forward the emails to the target email addresses.
 * The Lambda function is just a thin wrapper around the <a href="https://github.com/arithmetric/aws-lambda-ses-forwarder">aws-lambda-ses-forwarder</a> library.
 * Since this library expects a JSON config with the email mappings, the <code>EmailForwardingRule</code> will create an SSM parameter to store the config.
 * (Note: this is not ideal because an SSM parameter is limited in the size and hence, this might be changed later)
 * The Lambda function receives a reference to this parameter as an environment variable (and a bit more) and forwards everything to the library.
 * <p>
 * In order to verify a domain or email address, the <code>EmailForwardingRuleSet</code> construct is using the package <a href="https://www.npmjs.com/package/&#64;seeebiii/ses-verify-identities">&#64;seeebiii/ses-verify-identities</a>.
 * It provides constructs to verify the SES identities.
 * For domains, it creates appropriate Route53 records like MX, TXT and Cname (for DKIM).
 * For email addresses, it calls the AWS API to initiate email address verification.
 * <p>
 * <h2>TODO</h2>
 * <p>
 * <ul>
 * <li>Encrypt email files on S3 bucket by either using S3 bucket encryption (server side) or enable client encryption using SES actions</li>
 * <li>Write tests</li>
 * <li>Document options/JSDoc in Readme or separate HTML</li>
 * </ul>
 * <p>
 * <h2>Contributing</h2>
 * <p>
 * I'm happy to receive any contributions!
 * Just open an issue or pull request :)
 * <p>
 * These commands should help you while developing:
 * <p>
 * <ul>
 * <li><code>npx projen</code>             synthesize changes in <a href=".projenrc.js">.projenrc.js</a> to the project</li>
 * <li><code>npm run build</code>          compile typescript to js</li>
 * <li><code>npm run watch</code>          watch for changes and compile</li>
 * <li><code>npm run test</code>           perform the jest unit tests</li>
 * <li><code>npm run lint</code>           validate code against best practices</li>
 * </ul>
 * <p>
 * <h2>Thanks</h2>
 * <p>
 * Thanks a lot to <a href="https://github.com/arithmetric">arithmetric</a> for providing the NPM package <a href="https://github.com/arithmetric/aws-lambda-ses-forwarder">aws-lambda-ses-forwarder</a>.
 * This CDK construct is using it in the Lambda function to forward the emails.
 * <p>
 * <h2>Author</h2>
 * <p>
 * <a href="https://www.sebastianhesse.de">Sebastian Hesse</a> - Freelancer for serverless cloud projects on AWS.
 * <p>
 * <h2>License</h2>
 * <p>
 * MIT License
 * <p>
 * Copyright (c) 2020 <a href="https://www.sebastianhesse.de">Sebastian Hesse</a>
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package de.sebastianhesse.cdk.ses.email.forwarding;
