package de.terrestris.shoguncore.service;

import de.terrestris.shoguncore.dao.LayerDao;
import de.terrestris.shoguncore.dao.MapDao;
import de.terrestris.shoguncore.model.PersistentObject;
import de.terrestris.shoguncore.model.layer.Layer;
import de.terrestris.shoguncore.model.module.Map;
import de.terrestris.shoguncore.model.module.Module;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Service class for the {@link Module} model.
 *
 * @author Nils Bühner
 * @see AbstractCrudService
 */
@Service("mapService")
public class MapService<E extends Map, D extends MapDao<E>> extends
    ModuleService<E, D> {

    @Autowired
    @Qualifier("layerService")
    private LayerService<Layer, LayerDao<Layer>> layerService;

    /**
     * Default constructor, which calls the type-constructor
     */
    @SuppressWarnings("unchecked")
    public MapService() {
        this((Class<E>) Map.class);
    }

    /**
     * Constructor that sets the concrete entity class for the service.
     * Subclasses MUST call this constructor.
     */
    protected MapService(Class<E> entityClass) {
        super(entityClass);
    }

    /**
     * We have to use {@link Qualifier} to define the correct dao here.
     * Otherwise, spring can not decide which dao has to be autowired here
     * as there are multiple candidates.
     */
    @Override
    @Autowired
    @Qualifier("mapDao")
    public void setDao(D dao) {
        this.dao = dao;
    }

    /**
     *
     */
    @PreAuthorize("hasRole(@configHolder.getSuperAdminRoleName()) or hasPermission(#layer, 'READ')")
    @Transactional(readOnly = true)
    public Set<E> findMapsWithLayer(Layer layer) {
        return dao.findMapsWithLayer(layer);
    }

    /**
     * TODO secure this method!?
     *
     * @param layerIds
     * @throws Exception
     */
    @SuppressWarnings("unchecked")
    @PreAuthorize("hasRole(@configHolder.getSuperAdminRoleName()) or hasPermission(#mapModuleId, 'de.terrestris.shoguncore.model.module.Map', 'UPDATE')")
    public List<Layer> setLayersForMap(Integer mapModuleId, List<Integer> layerIds) throws Exception {
        E module = this.findById(mapModuleId);
        List<Layer> layers = new ArrayList<Layer>();
        for (Integer id : layerIds) {
            PersistentObject entity = this.layerService.findById(id);
            Layer layer = (Layer) entity;
            if (layer != null) {
                layers.add(layer);
            }
        }
        module.setMapLayers(layers);
        this.saveOrUpdate(module);
        return layers;

    }
}
