/*
 * Decompiled with CFR 0.152.
 */
package de.trustable.ca3s.core.schedule;

import de.trustable.ca3s.core.domain.Authority;
import de.trustable.ca3s.core.domain.CSR;
import de.trustable.ca3s.core.domain.Certificate;
import de.trustable.ca3s.core.domain.User;
import de.trustable.ca3s.core.repository.CSRRepository;
import de.trustable.ca3s.core.repository.CertificateRepository;
import de.trustable.ca3s.core.repository.UserRepository;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

@Component
@Transactional(propagation=Propagation.REQUIRES_NEW)
public class UserRetentionScheduler {
    transient Logger LOG = LoggerFactory.getLogger(UserRetentionScheduler.class);
    private final UserRepository userRepository;
    private final CertificateRepository certificateRepository;
    private final CSRRepository csrRepository;
    private final int defaultRetentionPeriod;
    private final int csrOwnerRetentionPeriod;
    private final int certificateOwnerRetentionPeriod;

    public UserRetentionScheduler(UserRepository userRepository, CertificateRepository certificateRepository, CSRRepository csrRepository, @Value(value="${ca3s.user.retention.default:5}") int defaultRetentionPeriod, @Value(value="${ca3s.user.retention.default:90}") int csrOwnerRetentionPeriod, @Value(value="${ca3s.user.retention.default:3600}") int certificateOwnerRetentionPeriod) {
        this.userRepository = userRepository;
        this.certificateRepository = certificateRepository;
        this.csrRepository = csrRepository;
        this.defaultRetentionPeriod = defaultRetentionPeriod;
        this.csrOwnerRetentionPeriod = csrOwnerRetentionPeriod;
        this.certificateOwnerRetentionPeriod = certificateOwnerRetentionPeriod;
    }

    @Scheduled(cron="0 20 02 * * ?")
    public void retrieveUnrelatedUsers() {
        Instant oldestRelevantCertificateExpiry = Instant.now().minus(this.certificateOwnerRetentionPeriod, ChronoUnit.DAYS);
        Instant oldestRelevantCSR = Instant.now().minus(this.csrOwnerRetentionPeriod, ChronoUnit.DAYS);
        Instant oldestRelevantLogin = Instant.now().minus(this.defaultRetentionPeriod, ChronoUnit.DAYS);
        for (User user : this.userRepository.findAll()) {
            if (this.hasRecentActivity(user, oldestRelevantLogin)) {
                this.LOG.debug("user {} has recent login", (Object)user.getLogin());
                continue;
            }
            if (this.hasAuthority(user, "ROLE_ADMIN")) {
                this.LOG.debug("user {} ignored from retention check. Has role {}", (Object)user.getLogin(), (Object)"ROLE_ADMIN");
                continue;
            }
            if (this.hasAuthority(user, "ROLE_RA")) {
                this.LOG.debug("user {} ignored from retention check. Has role {}", (Object)user.getLogin(), (Object)"ROLE_RA");
                continue;
            }
            if (this.hasAuthority(user, "ROLE_RA_DOMAIN")) {
                this.LOG.debug("user {} ignored from retention check. Has role {}", (Object)user.getLogin(), (Object)"ROLE_RA_DOMAIN");
                continue;
            }
            if (this.hasRelevantCertificate(user, oldestRelevantCertificateExpiry)) {
                this.LOG.debug("user {} has relevant certificate", (Object)user.getLogin());
                continue;
            }
            if (this.hasRelevantCSR(user, oldestRelevantCSR)) {
                this.LOG.debug("user {} has relevant csr", (Object)user.getLogin());
                continue;
            }
            this.LOG.info("user {} has no relevant data left, deleting ...", (Object)user.getLogin());
        }
    }

    private boolean hasAuthority(User user, String authorityName) {
        for (Authority authority : user.getAuthorities()) {
            if (!authorityName.equalsIgnoreCase(authority.getName())) continue;
            return true;
        }
        return false;
    }

    private boolean hasRecentActivity(User user, Instant oldestRelevantLogin) {
        if (user.getCreatedDate() != null && user.getCreatedDate().isAfter(oldestRelevantLogin)) {
            return true;
        }
        if (user.getLastModifiedDate() != null && user.getLastModifiedDate().isAfter(oldestRelevantLogin)) {
            return true;
        }
        return user.getLastUserDetailsUpdate() != null && user.getLastUserDetailsUpdate().isAfter(oldestRelevantLogin);
    }

    private boolean hasRelevantCSR(User user, Instant oldestRelevantCSR) {
        for (CSR csr : this.csrRepository.findByRequestor(user.getLogin())) {
            if (!csr.getRequestedOn().isAfter(oldestRelevantCSR)) continue;
            this.LOG.debug("user {} has relevant csr {} requested on {} ", new Object[]{user.getLogin(), csr.getId(), csr.getRequestedOn()});
            return true;
        }
        return true;
    }

    private boolean hasRelevantCertificate(User user, Instant oldestRelevantCertificateExpiry) {
        for (Certificate certificate : this.certificateRepository.findByRequestor(user.getLogin())) {
            if (!certificate.getValidTo().isAfter(oldestRelevantCertificateExpiry)) continue;
            this.LOG.debug("user {} has relevant certificate {} valid until {} ", new Object[]{user.getLogin(), certificate.getId(), certificate.getValidTo()});
            return true;
        }
        return true;
    }
}

