/*
 * Decompiled with CFR 0.152.
 */
package de.trustable.ca3s.core.service;

import de.trustable.ca3s.core.domain.Authority;
import de.trustable.ca3s.core.domain.ProtectedContent;
import de.trustable.ca3s.core.domain.Tenant;
import de.trustable.ca3s.core.domain.User;
import de.trustable.ca3s.core.domain.enumeration.ContentRelationType;
import de.trustable.ca3s.core.domain.enumeration.ProtectedContentType;
import de.trustable.ca3s.core.repository.AuthorityRepository;
import de.trustable.ca3s.core.repository.TenantRepository;
import de.trustable.ca3s.core.repository.UserRepository;
import de.trustable.ca3s.core.repository.UserSpecifications;
import de.trustable.ca3s.core.security.SecurityUtils;
import de.trustable.ca3s.core.service.EmailAlreadyUsedException;
import de.trustable.ca3s.core.service.InvalidPasswordException;
import de.trustable.ca3s.core.service.UsernameAlreadyUsedException;
import de.trustable.ca3s.core.service.dto.UserDTO;
import de.trustable.ca3s.core.service.util.PasswordUtil;
import de.trustable.ca3s.core.service.util.ProtectedContentUtil;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.persistence.EntityManager;
import javax.persistence.criteria.CriteriaBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.CacheManager;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import tech.jhipster.security.RandomUtil;

@Service
@Transactional
public class UserService {
    private final Logger log = LoggerFactory.getLogger(UserService.class);
    private final UserRepository userRepository;
    private final PasswordEncoder passwordEncoder;
    private final ProtectedContentUtil protectedContentUtil;
    private final AuthorityRepository authorityRepository;
    private final TenantRepository tenantRepository;
    private final CacheManager cacheManager;
    private final PasswordUtil passwordUtil;
    private final EntityManager entityManager;
    private final int activationKeyValidity;

    public UserService(UserRepository userRepository, PasswordEncoder passwordEncoder, ProtectedContentUtil protectedContentUtil, AuthorityRepository authorityRepository, TenantRepository tenantRepository, CacheManager cacheManager, @Value(value="${ca3s.ui.password.check.regexp:^(?=.*\\d)(?=.*[a-z]).{6,100}$}") String passwordCheckRegExp, EntityManager entityManager, @Value(value="${ca3s.ui.password.activation.keyValidity:7}") int activationKeyValidity) {
        this.userRepository = userRepository;
        this.passwordEncoder = passwordEncoder;
        this.protectedContentUtil = protectedContentUtil;
        this.authorityRepository = authorityRepository;
        this.tenantRepository = tenantRepository;
        this.cacheManager = cacheManager;
        this.passwordUtil = new PasswordUtil(passwordCheckRegExp);
        this.entityManager = entityManager;
        this.activationKeyValidity = activationKeyValidity;
    }

    public Optional<User> activateRegistration(String key) {
        this.log.debug("Activating user for activation key {}", (Object)key);
        List protectedContents = this.findActivationKeys(key);
        if (protectedContents.isEmpty()) {
            this.log.info("No User found for activation key: {}", (Object)key);
        } else {
            for (ProtectedContent protectedContent : protectedContents) {
                Optional optUser = this.userRepository.findById((Object)protectedContent.getRelatedId());
                if (!optUser.isPresent()) continue;
                User user = (User)optUser.get();
                user.setActivated(true);
                this.clearUserCaches(user);
                this.log.debug("Activated user: {}", (Object)user);
                return Optional.of(user);
            }
        }
        return Optional.empty();
    }

    public Optional<User> completePasswordReset(String newPassword, String key) {
        this.log.debug("Reset user password for reset key {}", (Object)key);
        return this.userRepository.findOneByResetKey(key).filter(user -> user.getResetDate().isAfter(Instant.now().minusSeconds(86400L))).map(user -> {
            user.setPassword(this.passwordEncoder.encode((CharSequence)newPassword));
            user.setResetKey(null);
            user.setResetDate(null);
            this.clearUserCaches(user);
            return user;
        });
    }

    public Optional<User> requestPasswordReset(String username) {
        return this.userRepository.findOneByLogin(username).filter(User::getActivated).map(user -> {
            user.setResetKey(RandomUtil.generateResetKey());
            user.setResetDate(Instant.now());
            this.clearUserCaches(user);
            return user;
        });
    }

    public User registerUser(UserDTO userDTO, String password, String activationKey) {
        this.userRepository.findOneByLogin(userDTO.getLogin().toLowerCase()).ifPresent(existingUser -> {
            boolean removed = this.removeNonActivatedUser(existingUser);
            if (!removed) {
                throw new UsernameAlreadyUsedException();
            }
        });
        this.userRepository.findOneByEmailIgnoreCase(userDTO.getEmail()).ifPresent(existingUser -> {
            boolean removed = this.removeNonActivatedUser(existingUser);
            if (!removed) {
                throw new EmailAlreadyUsedException();
            }
        });
        User newUser = new User();
        String encryptedPassword = this.passwordEncoder.encode((CharSequence)password);
        newUser.setLogin(userDTO.getLogin().toLowerCase());
        newUser.setPassword(encryptedPassword);
        newUser.setFirstName(userDTO.getFirstName());
        newUser.setLastName(userDTO.getLastName());
        if (userDTO.getEmail() != null) {
            newUser.setEmail(userDTO.getEmail().toLowerCase());
        }
        newUser.setImageUrl(userDTO.getImageUrl());
        newUser.setLangKey(userDTO.getLangKey());
        newUser.setActivated(false);
        HashSet authorities = new HashSet();
        this.authorityRepository.findById((Object)"ROLE_USER").ifPresent(authorities::add);
        newUser.setAuthorities(authorities);
        this.updateTenant(newUser, userDTO);
        newUser = (User)this.userRepository.save((Object)newUser);
        this.clearUserCaches(newUser);
        this.log.debug("Created Information for User: {}", (Object)newUser);
        this.protectedContentUtil.createDerivedProtectedContent(activationKey, ProtectedContentType.DERIVED_SECRET, ContentRelationType.ACTIVATION_KEY, newUser.getId().longValue(), -1, Instant.now().plus((long)this.activationKeyValidity, ChronoUnit.DAYS));
        return newUser;
    }

    private List<ProtectedContent> findActivationKeys(String plainText) {
        return this.protectedContentUtil.findProtectedContentBySecret(plainText, ProtectedContentType.DERIVED_SECRET, ContentRelationType.ACTIVATION_KEY);
    }

    private boolean removeNonActivatedUser(User existingUser) {
        if (existingUser.getActivated()) {
            return false;
        }
        this.userRepository.delete((Object)existingUser);
        this.userRepository.flush();
        this.clearUserCaches(existingUser);
        return true;
    }

    public User createUser(UserDTO userDTO) {
        User user = new User();
        user.setLogin(userDTO.getLogin().toLowerCase());
        user.setFirstName(userDTO.getFirstName());
        user.setLastName(userDTO.getLastName());
        if (userDTO.getEmail() != null) {
            user.setEmail(userDTO.getEmail().toLowerCase());
        }
        user.setImageUrl(userDTO.getImageUrl());
        if (userDTO.getLangKey() == null) {
            user.setLangKey("en");
        } else {
            user.setLangKey(userDTO.getLangKey());
        }
        String encryptedPassword = this.passwordEncoder.encode((CharSequence)RandomUtil.generatePassword());
        user.setPassword(encryptedPassword);
        user.setResetKey(RandomUtil.generateResetKey());
        user.setResetDate(Instant.now());
        user.setActivated(true);
        if (userDTO.getAuthorities() != null) {
            Set authorities = userDTO.getAuthorities().stream().map(arg_0 -> ((AuthorityRepository)this.authorityRepository).findById(arg_0)).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toSet());
            user.setAuthorities(authorities);
        }
        this.updateTenant(user, userDTO);
        this.userRepository.save((Object)user);
        this.clearUserCaches(user);
        this.log.debug("Created Information for User: {}", (Object)user);
        return user;
    }

    public void updateUser(String firstName, String lastName, String email, String langKey, String imageUrl, Long tenantId) {
        SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneByLogin(arg_0)).ifPresent(user -> {
            user.setFirstName(firstName);
            user.setLastName(lastName);
            if (email != null) {
                user.setEmail(email.toLowerCase());
            }
            user.setLangKey(langKey);
            user.setImageUrl(imageUrl);
            this.updateTenant(user, tenantId);
            this.clearUserCaches(user);
            this.log.debug("Changed Information for User: {}", user);
        });
    }

    public Optional<UserDTO> updateUser(UserDTO userDTO) {
        return Optional.of(this.userRepository.findById((Object)userDTO.getId())).filter(Optional::isPresent).map(Optional::get).map(user -> {
            this.clearUserCaches(user);
            user.setLogin(userDTO.getLogin().toLowerCase());
            user.setFirstName(userDTO.getFirstName());
            user.setLastName(userDTO.getLastName());
            if (userDTO.getEmail() != null) {
                user.setEmail(userDTO.getEmail().toLowerCase());
            }
            user.setImageUrl(userDTO.getImageUrl());
            user.setActivated(userDTO.isActivated());
            user.setLangKey(userDTO.getLangKey());
            this.updateTenant(user, userDTO);
            user.setBlockedUntilDate(userDTO.getBlockedUntilDate());
            user.setFailedLogins(userDTO.getFailedLogins());
            Set managedAuthorities = user.getAuthorities();
            managedAuthorities.clear();
            userDTO.getAuthorities().stream().map(arg_0 -> ((AuthorityRepository)this.authorityRepository).findById(arg_0)).filter(Optional::isPresent).map(Optional::get).forEach(managedAuthorities::add);
            this.clearUserCaches(user);
            this.log.debug("Changed Information for User: {}", user);
            return user;
        }).map(UserDTO::new);
    }

    private void updateTenant(User user, UserDTO userDTO) {
        this.updateTenant(user, userDTO.getTenantId());
    }

    private void updateTenant(User user, Long tenantId) {
        Optional optionalTenant;
        Tenant tenant = null;
        if (tenantId != null && tenantId != 0L && (optionalTenant = this.tenantRepository.findById((Object)tenantId)).isPresent()) {
            tenant = (Tenant)optionalTenant.get();
        }
        user.setTenant(tenant);
    }

    public void deleteUser(String login) {
        this.userRepository.findOneByLogin(login).ifPresent(user -> {
            this.userRepository.delete(user);
            this.clearUserCaches(user);
            this.log.debug("Deleted User: {}", user);
        });
    }

    public void changePassword(String currentClearTextPassword, String newPassword) {
        SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneByLogin(arg_0)).ifPresent(user -> {
            String currentEncryptedPassword = user.getPassword();
            if (!this.passwordEncoder.matches((CharSequence)currentClearTextPassword, currentEncryptedPassword)) {
                throw new InvalidPasswordException();
            }
            String encryptedPassword = this.passwordEncoder.encode((CharSequence)newPassword);
            user.setPassword(encryptedPassword);
            this.clearUserCaches(user);
            this.log.debug("Changed password for User: {}", user);
        });
    }

    @Transactional(readOnly=true)
    public Page<UserDTO> getAllManagedUsers(Pageable pageable) {
        return this.userRepository.findAllByLoginNot(pageable, "anonymoususer").map(UserDTO::new);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthoritiesByLogin(String login) {
        return this.userRepository.findOneWithAuthoritiesByLogin(login);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthorities(Long id) {
        return this.userRepository.findOneWithAuthoritiesById(id);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthorities() {
        return SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneWithAuthoritiesByLogin(arg_0));
    }

    @Scheduled(cron="0 0 1 * * ?")
    public void removeNotActivatedUsers() {
        this.userRepository.findAllByActivatedIsFalseAndActivationKeyIsNotNullAndCreatedDateBefore(Instant.now().minus(3L, ChronoUnit.DAYS)).forEach(user -> {
            this.log.debug("Deleting not activated user {}", (Object)user.getLogin());
            this.userRepository.delete(user);
            this.clearUserCaches(user);
        });
    }

    public List<String> getAuthorities() {
        return this.authorityRepository.findAll().stream().map(Authority::getName).collect(Collectors.toList());
    }

    @Transactional(readOnly=true)
    public List<User> getUsersByRole(String role) {
        return this.userRepository.findActiveByRole(role);
    }

    private void clearUserCaches(User user) {
        Objects.requireNonNull(this.cacheManager.getCache("usersByLogin")).evict((Object)user.getLogin());
        if (user.getEmail() != null) {
            Objects.requireNonNull(this.cacheManager.getCache("usersByEmail")).evict((Object)user.getEmail());
        }
    }

    public void checkPassword(String password) {
        this.passwordUtil.checkPassword(password, "user password");
    }

    public Page<UserDTO> findSelection(Map<String, String[]> parameterMap) {
        Page userDTOPage = UserSpecifications.handleQueryParamsUser((EntityManager)this.entityManager, (CriteriaBuilder)this.entityManager.getCriteriaBuilder(), parameterMap);
        for (UserDTO userDTO : userDTOPage.getContent()) {
            this.userRepository.findById((Object)userDTO.getId()).ifPresent(user -> {
                userDTO.setAuthorities(user.getAuthorities().stream().map(authority -> authority.getName()).collect(Collectors.toSet()));
                Tenant tenant = user.getTenant();
                if (tenant != null) {
                    userDTO.setTenantId(tenant.getId());
                    userDTO.setTenantName(tenant.getLongname());
                }
            });
        }
        return userDTOPage;
    }
}

