/*
 * Decompiled with CFR 0.152.
 */
package de.trustable.ca3s.core.web.rest.support;

import de.trustable.ca3s.core.domain.CSR;
import de.trustable.ca3s.core.domain.Certificate;
import de.trustable.ca3s.core.domain.CertificateAttribute;
import de.trustable.ca3s.core.domain.User;
import de.trustable.ca3s.core.exception.CAFailureException;
import de.trustable.ca3s.core.repository.CertificateAttributeRepository;
import de.trustable.ca3s.core.repository.CertificateRepository;
import de.trustable.ca3s.core.repository.UserRepository;
import de.trustable.ca3s.core.schedule.CertExpiryScheduler;
import de.trustable.ca3s.core.service.AuditService;
import de.trustable.ca3s.core.service.NotificationService;
import de.trustable.ca3s.core.service.dto.CRLUpdateInfo;
import de.trustable.ca3s.core.service.dto.NamedValue;
import de.trustable.ca3s.core.service.util.BPMNUtil;
import de.trustable.ca3s.core.service.util.CRLUtil;
import de.trustable.ca3s.core.service.util.CertificateUtil;
import de.trustable.ca3s.core.web.rest.data.AdministrationType;
import de.trustable.ca3s.core.web.rest.data.CertificateAdministrationData;
import de.trustable.util.CryptoUtil;
import java.security.GeneralSecurityException;
import java.time.Instant;
import java.util.Date;
import java.util.HashSet;
import java.util.Optional;
import javax.mail.MessagingException;
import javax.validation.Valid;
import org.bouncycastle.asn1.x509.CRLReason;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
@RequestMapping(value={"/api"})
public class CertificateAdministration {
    private final Logger LOG = LoggerFactory.getLogger(CertificateAdministration.class);
    private final CertificateRepository certificateRepository;
    private final CertificateAttributeRepository certificateAttributeRepository;
    private final BPMNUtil bpmnUtil;
    private final CryptoUtil cryptoUtil;
    private final CertificateUtil certUtil;
    private final CRLUtil crlUtil;
    private final UserRepository userRepository;
    private final CertExpiryScheduler certExpiryScheduler;
    private final NotificationService notificationService;
    private final AuditService auditService;

    public CertificateAdministration(CertificateRepository certificateRepository, CertificateAttributeRepository certificateAttributeRepository, BPMNUtil bpmnUtil, CryptoUtil cryptoUtil, CertificateUtil certUtil, CRLUtil crlUtil, UserRepository userRepository, CertExpiryScheduler certExpiryScheduler, NotificationService notificationService, AuditService auditService) {
        this.certificateRepository = certificateRepository;
        this.certificateAttributeRepository = certificateAttributeRepository;
        this.bpmnUtil = bpmnUtil;
        this.cryptoUtil = cryptoUtil;
        this.certUtil = certUtil;
        this.crlUtil = crlUtil;
        this.userRepository = userRepository;
        this.certExpiryScheduler = certExpiryScheduler;
        this.notificationService = notificationService;
        this.auditService = auditService;
    }

    @PostMapping(value={"/administerCertificate"})
    @Transactional
    public ResponseEntity<Long> administerCertificate(@Valid @RequestBody CertificateAdministrationData adminData) throws MessagingException {
        this.LOG.debug("REST request to revoke / update certificate : {}", (Object)adminData);
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        String raOfficerName = auth.getName();
        Optional optCert = this.certificateRepository.findById((Object)adminData.getCertificateId());
        if (optCert.isPresent()) {
            Certificate cert = (Certificate)optCert.get();
            try {
                if (AdministrationType.REVOKE.equals((Object)adminData.getAdministrationType())) {
                    this.revokeCertificate(cert, adminData, raOfficerName);
                    CSR csr = cert.getCsr();
                    if (csr != null) {
                        Optional optUser = this.userRepository.findOneByLogin(csr.getRequestedBy());
                        if (optUser.isPresent()) {
                            User requestor = (User)optUser.get();
                            if (requestor.getEmail() == null) {
                                this.LOG.debug("Email doesn't exist for user '{}'", (Object)requestor.getLogin());
                            } else {
                                this.notificationService.notifyUserCerificateRevokedAsync(requestor, cert, csr);
                            }
                        } else {
                            this.LOG.info("certificate requestor '{}' unknown!", (Object)csr.getRequestedBy());
                        }
                    }
                    this.updateComment(adminData, cert);
                    this.updateARAttributes(adminData, cert);
                } else if (AdministrationType.UPDATE.equals((Object)adminData.getAdministrationType())) {
                    this.updateComment(adminData, cert);
                    this.updateARAttributes(adminData, cert);
                    this.updateTrustedFlag(adminData, cert);
                } else if (AdministrationType.UPDATE_CRL.equals((Object)adminData.getAdministrationType())) {
                    CRLUpdateInfo crlInfo = this.certUtil.checkAllCRLsForCertificate(cert, CertificateUtil.convertPemToCertificate((String)cert.getContent()), this.crlUtil, new HashSet());
                    if (!crlInfo.isbCRLDownloadSuccess()) {
                        this.LOG.info("Downloading all CRL #{} for certificate {} failed", (Object)crlInfo.getCrlUrlCount(), (Object)cert.getId());
                    }
                } else {
                    this.LOG.info("administration type '{}' unexpected!", (Object)adminData.getAdministrationType());
                    return ResponseEntity.badRequest().build();
                }
                this.certUtil.setCertificateComment(cert, adminData.getComment());
                return new ResponseEntity((Object)adminData.getCertificateId(), HttpStatus.OK);
            }
            catch (RuntimeException | GeneralSecurityException e) {
                this.auditService.saveAuditTrace(this.auditService.createAuditTraceCertificate("AUDIT_CERTIFICATE_ADMINISTRATION_FAILED", e.getMessage(), cert));
                return ResponseEntity.badRequest().build();
            }
        }
        return ResponseEntity.notFound().build();
    }

    private void updateTrustedFlag(CertificateAdministrationData adminData, Certificate cert) {
        if (adminData.getTrusted().booleanValue() && cert.isTrusted()) {
            this.LOG.debug("Certificate id {} already marked as trusted", (Object)cert.getId());
        } else if (adminData.getTrusted().booleanValue() && !cert.isTrusted()) {
            this.LOG.debug("Setting Certificate id {} as trusted", (Object)cert.getId());
            if (cert.isRevoked()) {
                this.LOG.error("Cannot set revoked Certificate id {} as trusted!", (Object)cert.getId());
                return;
            }
            if (!cert.isActive()) {
                this.LOG.error("Cannot set expired Certificate id {} as trusted!", (Object)cert.getId());
                return;
            }
            if (cert.isEndEntity()) {
                this.LOG.error("Cannot set end entity certificate id {} as trusted!", (Object)cert.getId());
                return;
            }
            if (!cert.isSelfsigned()) {
                this.LOG.warn("Trying to set the certificate id {} as trusted, but it is not selfsigned!", (Object)cert.getId());
            }
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceCertificate("CERTIFICATE_SET_TRUSTED", cert));
            cert.setTrusted(Boolean.valueOf(true));
            this.certificateRepository.save((Object)cert);
        } else if (!adminData.getTrusted().booleanValue() && cert.isTrusted()) {
            this.LOG.debug("Revoking 'trusted' status from certificate id {}", (Object)cert.getId());
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceCertificate("CERTIFICATE_UNSET_TRUSTED", cert));
            cert.setTrusted(Boolean.valueOf(false));
            this.certificateRepository.save((Object)cert);
        }
    }

    @PostMapping(value={"/withdrawOwnCertificate"})
    @Transactional
    public ResponseEntity<Long> withdrawOwnCertificate(@Valid @RequestBody CertificateAdministrationData adminData) {
        this.LOG.debug("REST request to withdraw Certificate : {}", (Object)adminData);
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        String userName = auth.getName();
        Optional optCert = this.certificateRepository.findById((Object)adminData.getCertificateId());
        if (optCert.isPresent()) {
            Certificate certificate = (Certificate)optCert.get();
            String requestedBy = certificate.getCsr().getRequestedBy();
            if (userName == null || !userName.equals(requestedBy)) {
                this.LOG.debug("REST request by '{}' to revoke certificate '{}' rejected ", (Object)userName, (Object)adminData.getCertificateId());
                return ResponseEntity.status((HttpStatus)HttpStatus.UNAUTHORIZED).build();
            }
            this.updateComment(adminData, certificate);
            this.updateARAttributes(adminData, certificate);
            try {
                this.revokeCertificate(certificate, adminData, userName);
                return new ResponseEntity((Object)adminData.getCertificateId(), HttpStatus.OK);
            }
            catch (GeneralSecurityException e) {
                return ResponseEntity.badRequest().build();
            }
        }
        return ResponseEntity.notFound().build();
    }

    private void revokeCertificate(Certificate cert, CertificateAdministrationData adminData, String revokingUser) throws GeneralSecurityException {
        if (cert.isRevoked()) {
            this.LOG.warn("failureReason: certificate with id '" + cert.getId() + "' already revoked.");
        }
        this.auditService.saveAuditTrace(this.auditService.createAuditTraceCertificate("CERTIFICATE_REVOKED", cert));
        CRLReason crlReason = this.cryptoUtil.crlReasonFromString(adminData.getRevocationReason());
        String crlReasonStr = this.cryptoUtil.crlReasonAsString(crlReason);
        this.LOG.debug("crlReason : " + crlReasonStr + " from " + adminData.getRevocationReason());
        Date revocationDate = new Date();
        this.bpmnUtil.startCertificateRevocationProcess(cert, crlReason, revocationDate);
        cert.setActive(Boolean.valueOf(false));
        cert.setRevoked(Boolean.valueOf(true));
        cert.setRevokedSince(Instant.now());
        cert.setRevocationReason(crlReasonStr);
        if (adminData.getComment() != null && adminData.getComment().trim().length() > 0) {
            cert.setAdministrationComment(adminData.getComment());
        }
        this.certUtil.setCertAttribute(cert, "REVOKED_BY", revokingUser, false);
        cert.setRevocationExecutionId("39");
        this.certificateRepository.save((Object)cert);
    }

    @PostMapping(value={"/sendExpiringCertificateEmail"})
    @Transactional
    public ResponseEntity<Integer> sendExpiringCertificateEmail() {
        int nExpiringCerts = this.certExpiryScheduler.notifyRAOfficerHolderOnExpiry();
        return new ResponseEntity((Object)nExpiringCerts, HttpStatus.OK);
    }

    @PostMapping(value={"/selfAdministerCertificate"})
    @Transactional(noRollbackFor={CAFailureException.class})
    public ResponseEntity<Long> selfAdministerCertificate(@Valid @RequestBody CertificateAdministrationData adminData) {
        this.LOG.debug("REST request to update certificate : {}", (Object)adminData);
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        String userName = auth.getName();
        Optional optCert = this.certificateRepository.findById((Object)adminData.getCertificateId());
        if (optCert.isPresent()) {
            Certificate certificate = (Certificate)optCert.get();
            String requestedBy = null;
            if (certificate.getCsr() != null) {
                requestedBy = certificate.getCsr().getRequestedBy();
            }
            if (userName == null || !userName.equals(requestedBy)) {
                this.LOG.debug("REST request by '{}' to update certificate '{}' rejected ", (Object)userName, (Object)adminData.getCertificateId());
                return ResponseEntity.status((HttpStatus)HttpStatus.UNAUTHORIZED).build();
            }
            this.updateARAttributes(adminData, certificate);
            this.updateComment(adminData, certificate);
            this.certificateRepository.save((Object)certificate);
            return new ResponseEntity((Object)adminData.getCertificateId(), HttpStatus.OK);
        }
        return ResponseEntity.notFound().build();
    }

    private void updateComment(CertificateAdministrationData adminData, Certificate cert) {
        this.certUtil.setCertificateComment(cert, adminData.getComment());
    }

    private void updateARAttributes(CertificateAdministrationData adminData, Certificate cert) {
        for (CertificateAttribute certAttr : cert.getCertificateAttributes()) {
            if (!certAttr.getName().startsWith("_ARA_")) continue;
            for (NamedValue nv : adminData.getArAttributeArr()) {
                if (!certAttr.getName().equals("_ARA_" + nv.getName()) || certAttr.getValue().equals(nv.getValue())) continue;
                this.auditService.saveAuditTrace(this.auditService.createAuditTraceCertificateAttribute(certAttr.getName(), certAttr.getValue(), nv.getValue(), cert));
                certAttr.setValue(nv.getValue());
                this.LOG.debug("certificate attribute {} updated to {}", (Object)certAttr.getName(), (Object)certAttr.getValue());
            }
        }
        this.certificateAttributeRepository.saveAll((Iterable)cert.getCertificateAttributes());
    }
}

