/*
 * Decompiled with CFR 0.152.
 */
package de.trustable.ca3s.core.web.scepservlet;

import de.trustable.ca3s.core.domain.CSR;
import de.trustable.ca3s.core.domain.Certificate;
import de.trustable.ca3s.core.domain.CsrAttribute;
import de.trustable.ca3s.core.domain.Pipeline;
import de.trustable.ca3s.core.domain.ProtectedContent;
import de.trustable.ca3s.core.domain.ScepOrder;
import de.trustable.ca3s.core.domain.enumeration.ContentRelationType;
import de.trustable.ca3s.core.domain.enumeration.PipelineType;
import de.trustable.ca3s.core.domain.enumeration.ProtectedContentType;
import de.trustable.ca3s.core.domain.enumeration.ScepOrderStatus;
import de.trustable.ca3s.core.exception.CAFailureException;
import de.trustable.ca3s.core.repository.CSRRepository;
import de.trustable.ca3s.core.repository.CertificateRepository;
import de.trustable.ca3s.core.repository.ProtectedContentRepository;
import de.trustable.ca3s.core.repository.ScepOrderRepository;
import de.trustable.ca3s.core.service.AuditService;
import de.trustable.ca3s.core.service.util.CSRUtil;
import de.trustable.ca3s.core.service.util.CertificateProcessingUtil;
import de.trustable.ca3s.core.service.util.CertificateUtil;
import de.trustable.ca3s.core.service.util.CryptoService;
import de.trustable.ca3s.core.service.util.PipelineUtil;
import de.trustable.ca3s.core.service.util.ProtectedContentUtil;
import de.trustable.ca3s.core.service.util.ScepOrderUtil;
import de.trustable.util.CryptoUtil;
import de.trustable.util.Pkcs10RequestHolder;
import java.io.IOException;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.servlet.ServletException;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.pkcs.Attribute;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.jetbrains.annotations.NotNull;
import org.jscep.server.ScepServlet;
import org.jscep.transaction.FailInfo;
import org.jscep.transaction.OperationFailureException;
import org.jscep.transaction.TransactionId;
import org.jscep.transport.response.Capability;
import org.jscep.util.CertificationRequestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class ScepServletImpl
extends ScepServlet {
    private static final long serialVersionUID = 7773233909179939491L;
    private static final Logger LOGGER = LoggerFactory.getLogger(ScepServletImpl.class);
    private static final X500Name pollName = new X500Name("CN=Poll");
    private final CertificateRepository certRepository;
    private final CSRRepository csrRepository;
    private final ScepOrderRepository scepOrderRepository;
    private final CryptoUtil cryptoUtil;
    private final CertificateUtil certUtil;
    private final CertificateProcessingUtil cpUtil;
    private final ScepOrderUtil scepOrderUtil;
    private final ProtectedContentRepository protectedContentRepository;
    private final ProtectedContentUtil protectedContentUtil;
    private final AuditService auditService;
    private final PipelineUtil pipelineUtil;
    public ThreadLocal<Pipeline> threadLocalPipeline = new ThreadLocal();

    public ScepServletImpl(CertificateRepository certRepository, CSRRepository csrRepository, ScepOrderRepository scepOrderRepository, CryptoUtil cryptoUtil, CertificateUtil certUtil, CertificateProcessingUtil cpUtil, ScepOrderUtil scepOrderUtil, ProtectedContentRepository protectedContentRepository, ProtectedContentUtil protectedContentUtil, AuditService auditService, PipelineUtil pipelineUtil) {
        this.certRepository = certRepository;
        this.csrRepository = csrRepository;
        this.scepOrderRepository = scepOrderRepository;
        this.cryptoUtil = cryptoUtil;
        this.certUtil = certUtil;
        this.cpUtil = cpUtil;
        this.scepOrderUtil = scepOrderUtil;
        this.protectedContentRepository = protectedContentRepository;
        this.protectedContentUtil = protectedContentUtil;
        this.auditService = auditService;
        this.pipelineUtil = pipelineUtil;
    }

    public void init() throws ServletException {
    }

    Certificate getCurrentRecepientCert() throws ServletException, OperationFailureException {
        Pipeline pipeline = (Pipeline)this.threadLocalPipeline.get();
        if (pipeline == null) {
            LOGGER.warn("doEnrol: no processing pipeline defined");
            throw new OperationFailureException(FailInfo.badRequest);
        }
        try {
            return this.pipelineUtil.getSCEPRecipientCertificate(pipeline, this.cpUtil);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new ServletException((Throwable)e);
        }
    }

    private Certificate startCertificateCreationProcess(String csrAsPem, TransactionId transId, Pipeline pipeline, ScepOrder scepOrder) throws OperationFailureException, IOException, GeneralSecurityException {
        Pkcs10RequestHolder p10Holder = this.cryptoUtil.parseCertificateRequest(csrAsPem);
        if (!this.pipelineUtil.isPipelineRestrictionsResolved(pipeline, p10Holder, new ArrayList())) {
            throw new OperationFailureException(FailInfo.badRequest);
        }
        String pipelineName = pipeline == null ? "NoPipeline" : pipeline.getName();
        String requestorName = "SCEP client";
        LOGGER.debug("doEnrol: processing request by {} using pipeline {}", (Object)requestorName, (Object)pipelineName);
        CSR csr = this.cpUtil.buildCSR(csrAsPem, requestorName, "SCEP_CERTIFICATE_REQUESTED", "", pipeline);
        if (csr == null) {
            String msg = "creation of certificate by SCEP transaction id '" + String.valueOf(transId) + "' failed ";
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceCsrRejected(csr, msg));
            LOGGER.info(msg);
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            return null;
        }
        CsrAttribute csrAttributeTransId = new CsrAttribute();
        csrAttributeTransId.setName("CA3S:SCEP_TRANS_ID");
        csrAttributeTransId.setValue(transId.toString());
        csr.addCsrAttributes(csrAttributeTransId);
        this.csrRepository.save((Object)csr);
        scepOrder.setCsr(csr);
        Certificate cert = null;
        try {
            cert = this.cpUtil.processCertificateRequest(csr, requestorName, "SCEP_CERTIFICATE_CREATED", pipeline);
        }
        catch (CAFailureException caFailureException) {
            LOGGER.info("certificate creation failed", (Throwable)caFailureException);
        }
        if (cert == null) {
            String msg = "creation of certificate by SCEP transaction id '" + String.valueOf(transId) + "' failed ";
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceCsrRejected(csr, msg));
            LOGGER.info(msg);
            scepOrder.setStatus(ScepOrderStatus.INVALID);
        } else {
            LOGGER.debug("new certificate id '{}' for SCEP transaction id '{}'", (Object)cert.getId(), (Object)transId);
            scepOrder.setCertificate(cert);
            this.certUtil.setCertAttribute(cert, "CA3S:SCEP_TRANS_ID", transId.toString());
            this.certRepository.save((Object)cert);
        }
        return cert;
    }

    protected List<X509Certificate> doEnrol(PKCS10CertificationRequest csr, X509Certificate sender, TransactionId transId) throws OperationFailureException {
        Pipeline pipeline = (Pipeline)this.threadLocalPipeline.get();
        if (pipeline == null) {
            LOGGER.warn("doEnrol: no processing pipeline defined");
            throw new OperationFailureException(FailInfo.badRequest);
        }
        LOGGER.debug("doEnrol(" + csr.toString() + ", " + transId.toString() + ") using pipeline '{}'", (Object)pipeline.getName());
        ScepOrder scepOrder = new ScepOrder();
        scepOrder.setPipeline(pipeline);
        scepOrder.setRealm(pipeline.getUrlPart());
        scepOrder.setTransId(transId.toString());
        scepOrder.setRequestedOn(Instant.now());
        scepOrder.setRequestedBy(sender.getSubjectX500Principal().toString());
        scepOrder.setStatus(ScepOrderStatus.PENDING);
        scepOrder.setPasswordAuthentication(Boolean.valueOf(false));
        scepOrder.setAsyncProcessing(Boolean.valueOf(false));
        try {
            X500Name subject = X500Name.getInstance((Object)csr.getSubject());
            LOGGER.debug(subject.toString());
            if (subject.equals((Object)pollName)) {
                List<X509Certificate> list = Collections.emptyList();
                return list;
            }
            this.scepOrderUtil.setOrderAttribute(scepOrder, "CN", csr.getSubject().toString());
            this.insertSANs(scepOrder, csr);
            this.scepOrderRepository.save((Object)scepOrder);
            Instant currentAuthenticationInstant = Instant.now();
            String password = CertificationRequestUtils.getChallengePassword((PKCS10CertificationRequest)csr);
            if (password != null) {
                this.checkPassword(pipeline, password);
                scepOrder.setPasswordAuthentication(Boolean.valueOf(true));
            } else {
                this.checkPipelineIsRenewalEnabled(pipeline, scepOrder);
                Certificate senderCert = this.certUtil.createCertificate(sender.getEncoded(), null, null, false, "scep sender certificate");
                this.checkSenderCertificate(senderCert, scepOrder);
                currentAuthenticationInstant = this.checkCsrVersusSenderCert(csr, senderCert, scepOrder, pipeline);
            }
            String p10ReqPem = CryptoUtil.pkcs10RequestToPem((PKCS10CertificationRequest)csr);
            Certificate newCertDao = this.startCertificateCreationProcess(p10ReqPem, transId, pipeline, scepOrder);
            if (newCertDao == null) {
                LOGGER.debug("creation of certificate failed");
                scepOrder.setStatus(ScepOrderStatus.INVALID);
                throw new OperationFailureException(FailInfo.badRequest);
            }
            this.certUtil.setCertAttribute(newCertDao, "CA3S:SCEP_TRANS_ID", transId.toString());
            this.certUtil.setCertAttribute(newCertDao, "SCEP_PERIOD_DAYS_RENEWAL", currentAuthenticationInstant.getEpochSecond());
            this.certRepository.save((Object)newCertDao);
            List<X509Certificate> certList = Arrays.asList(this.certUtil.getX509CertificateChain(newCertDao));
            for (X509Certificate x509 : certList) {
                LOGGER.debug("--- chain element: {}", (Object)x509.getSubjectX500Principal().toString());
            }
            scepOrder.setStatus(ScepOrderStatus.READY);
            List<X509Certificate> list = certList;
            return list;
        }
        catch (Exception e) {
            LOGGER.warn("Error in enrollment", (Throwable)e);
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        finally {
            this.scepOrderRepository.save((Object)scepOrder);
        }
    }

    private void checkPipelineIsRenewalEnabled(Pipeline pipeline, ScepOrder scepOrder) throws OperationFailureException {
        if (!this.pipelineUtil.getPipelineAttribute(pipeline, "SCEP_CAPABILITY_RENEWAL", false).booleanValue()) {
            String msg = String.format("renewal not supported by pipeline #%d / '%s'", pipeline.getId(), pipeline.getName());
            LOGGER.warn(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            throw new OperationFailureException(FailInfo.badRequest);
        }
    }

    @NotNull
    private Instant checkCsrVersusSenderCert(PKCS10CertificationRequest csr, Certificate senderCert, ScepOrder scepOrder, Pipeline pipeline) throws OperationFailureException {
        Instant lastAuthInstant;
        if (!this.checkCertificateAuthenticatesCSR(senderCert, csr)) {
            String msg = "SCEP request authentication by certificate failed, csr does not match authenticating certificate!";
            LOGGER.warn(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        int periodDaysRenewal = this.pipelineUtil.getPipelineAttribute(pipeline, "SCEP_PERIOD_DAYS_RENEWAL", 1825);
        Instant currentAuthenticationInstant = lastAuthInstant = Instant.ofEpochSecond(this.certUtil.getCertAttribute(senderCert, "CA3S:SCEP_LAST_AUTHORIZATION_INSTANT_SEC", senderCert.getValidFrom().getEpochSecond()));
        Instant latestReAuthInstant = lastAuthInstant.plusSeconds((long)periodDaysRenewal * 24L * 60L * 60L);
        Instant now = Instant.now();
        if (now.isAfter(latestReAuthInstant)) {
            String msg = "SCEP request authentication by certificate already expired!";
            LOGGER.info(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        LOGGER.debug("SCEP request authentication by certificate valid until {}.", (Object)latestReAuthInstant);
        int percentageValidity = this.pipelineUtil.getPipelineAttribute(pipeline, "SCEP_PERCENTAGE_OF_VALIDITY_BEFORE_RENEWAL", 80);
        long senderValiditySeconds = senderCert.getValidTo().getEpochSecond() - senderCert.getValidFrom().getEpochSecond();
        long minPassedValiditySeconds = senderValiditySeconds * (long)percentageValidity / 100L;
        Instant minRenewalInstant = senderCert.getValidFrom().plusSeconds(minPassedValiditySeconds);
        long remainingValiditySeconds = senderCert.getValidTo().getEpochSecond() - now.getEpochSecond();
        if (now.isBefore(minRenewalInstant)) {
            String msg = String.format("SCEP request authentication by certificate rejected, remaining validity (%d of %d seconds, %d %%) too long!", remainingValiditySeconds, senderValiditySeconds, 100L * remainingValiditySeconds / senderValiditySeconds);
            LOGGER.info(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        LOGGER.debug("SCEP request authentication valid, {}% ( < {} %) of validity period.", (Object)(100L * remainingValiditySeconds / senderValiditySeconds), (Object)percentageValidity);
        return currentAuthenticationInstant;
    }

    private void checkSenderCertificate(Certificate senderCert, ScepOrder scepOrder) throws OperationFailureException, GeneralSecurityException {
        String msg;
        if (!senderCert.isActive()) {
            String msg2 = String.format("certificate %d not active! Revoked %b, expiring on %s", senderCert.getId(), senderCert.isRevoked(), senderCert.getValidTo().toString());
            LOGGER.warn(msg2);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg2));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        boolean isTrusted = false;
        List senderChain = this.certUtil.getCertificateChain(senderCert);
        for (Certificate chainCert : senderChain) {
            if (!chainCert.isActive() || !chainCert.isTrusted()) continue;
            isTrusted = true;
            LOGGER.debug("chain certificate {} is trusted!", (Object)chainCert.getId());
            scepOrder.setAuthenticatedBy(chainCert);
            break;
        }
        if (!isTrusted) {
            msg = "SCEP request authentication by certificate failed, no trusted issuer found!";
            LOGGER.warn(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        if (senderCert.getCsr() == null || senderCert.getCsr().getPipeline() == null || !PipelineType.SCEP.equals((Object)senderCert.getCsr().getPipeline().getType())) {
            msg = "SCEP request authentication by certificate not issued by SCEP!";
            LOGGER.warn(msg);
            this.auditService.saveAuditTrace(this.auditService.createAuditTraceSCEPRequestRejected(scepOrder, msg));
            scepOrder.setStatus(ScepOrderStatus.INVALID);
            throw new OperationFailureException(FailInfo.badRequest);
        }
        LOGGER.debug("SCEP request authentication by certificate issued by SCEP!");
    }

    boolean checkCertificateAuthenticatesCSR(Certificate authenticatingCertificate, PKCS10CertificationRequest csr) {
        Set generalNameSetCSR = CSRUtil.getSANList((Attribute[])csr.getAttributes());
        for (RDN rdn : csr.getSubject().getRDNs()) {
            for (AttributeTypeAndValue atv : rdn.getTypesAndValues()) {
                if (!BCStyle.CN.equals((ASN1Primitive)atv.getType())) continue;
                String cnValue = atv.getValue().toString();
                LOGGER.debug("cn found in CSR: " + cnValue);
                generalNameSetCSR.add(new GeneralName(2, cnValue));
            }
        }
        boolean found = true;
        List vsanList = this.certUtil.getCertAttributes(authenticatingCertificate, "TYPED_VSAN");
        for (GeneralName generalName : generalNameSetCSR) {
            String typedSan = CertificateUtil.getTypedSAN((GeneralName)generalName);
            if (vsanList.contains(typedSan)) {
                LOGGER.debug("typedSan '{}' found in CSR and cert ", (Object)typedSan);
                continue;
            }
            LOGGER.info("typedSan '{}' found in CSR, not in cert ", (Object)typedSan);
            found = false;
        }
        return found;
    }

    void checkPassword(Pipeline pipeline, String password) throws OperationFailureException {
        if (password == null || password.isEmpty()) {
            LOGGER.warn("password not present in SCEP request / is empty!");
            throw new OperationFailureException(FailInfo.badRequest);
        }
        List listPC = this.protectedContentRepository.findByTypeRelationId(ProtectedContentType.PASSWORD, ContentRelationType.SCEP_PW, pipeline.getId());
        for (ProtectedContent pc : listPC) {
            String expectedPassword = this.protectedContentUtil.unprotectString(pc.getContentBase64()).trim();
            if (password.trim().equals(expectedPassword)) {
                LOGGER.debug("Protected Content found matching SCEP password");
                return;
            }
            LOGGER.debug("Protected Content password does not match SCEP password '{}' != '{}'", (Object)this.truncatePassword(expectedPassword), (Object)this.truncatePassword(password));
        }
        LOGGER.warn("no (active) password present in pipeline '" + pipeline.getName() + "' !");
        throw new OperationFailureException(FailInfo.badRequest);
    }

    public String truncatePassword(String password) {
        if (password == null || password.length() < 5) {
            return "******r3t";
        }
        return password.substring(password.length() - 4);
    }

    protected List<X509Certificate> doGetCaCertificate(String identifier) throws OperationFailureException {
        LOGGER.debug("doGetCaCertificate(" + identifier + ")");
        List<X509Certificate> caList = new ArrayList<X509Certificate>();
        try {
            Certificate recepCert = this.getCurrentRecepientCert();
            caList = this.certUtil.getX509CertificateChainAsList(recepCert);
            for (X509Certificate x509 : caList) {
                LOGGER.debug("--- recipient chain element: {}", (Object)x509.getSubjectX500Principal().toString());
            }
        }
        catch (GeneralSecurityException | ServletException e) {
            LOGGER.warn("Failed to retrieve CA certificates", e);
        }
        return caList;
    }

    protected X509CRL doGetCrl(X500Name issuer, BigInteger serial) {
        LOGGER.debug("doGetCrl(" + issuer.toString() + ", " + serial.toString(10) + ")");
        return null;
    }

    protected Set<Capability> doCapabilities(String identifier) {
        LOGGER.debug("doCapabilities(" + identifier + ")");
        HashSet<Capability> capabilitySet = new HashSet<Capability>();
        capabilitySet.add(Capability.SCEP_STANDARD);
        capabilitySet.add(Capability.SHA_256);
        capabilitySet.add(Capability.SHA_512);
        capabilitySet.add(Capability.POST_PKI_OPERATION);
        Pipeline pipeline = (Pipeline)this.threadLocalPipeline.get();
        if (pipeline == null) {
            LOGGER.warn("doCapabilities: no processing pipeline defined");
        } else if (this.pipelineUtil.getPipelineAttribute(pipeline, "SCEP_CAPABILITY_RENEWAL", false).booleanValue()) {
            capabilitySet.add(Capability.RENEWAL);
        }
        return capabilitySet;
    }

    protected List<X509Certificate> doGetCert(X500Name issuer, BigInteger serial) throws OperationFailureException {
        LOGGER.debug("doGetCert(" + issuer.toString() + ", " + serial.toString() + ")");
        List certDaoList = this.certRepository.findByIssuerSerial(issuer.toString(), serial.toString());
        if (certDaoList.isEmpty()) {
            RDN[] rdns;
            LOGGER.debug("no match for doGetCert(" + String.valueOf(issuer) + ", " + String.valueOf(serial) + ")");
            for (RDN rdn : rdns = issuer.getRDNs()) {
                AttributeTypeAndValue[] attTVArr;
                for (AttributeTypeAndValue attTV : attTVArr = rdn.getTypesAndValues()) {
                    LOGGER.debug("AttributeTypeAndValue of issuer :" + attTV.getType().toString() + " = " + attTV.getValue().toString());
                }
            }
            RDN[] rdnsIssuer = issuer.getRDNs(BCStyle.CN);
            if (rdnsIssuer.length > 0) {
                String msg;
                String rdnIssuerString = rdnsIssuer[0].getFirst().getValue().toString();
                String paddedSerial = CertificateUtil.getPaddedSerial((String)serial.toString());
                LOGGER.debug("looking for cert('" + rdnIssuerString + "', '" + paddedSerial + "')");
                certDaoList = this.certRepository.findBySearchTermNamed1("SERIAL_PADDED", paddedSerial);
                if (certDaoList.isEmpty()) {
                    msg = String.format("looking for cert by padded serial '%s' failed, nothing found", paddedSerial);
                    LOGGER.warn(msg);
                    throw new OperationFailureException(FailInfo.badCertId);
                }
                if (certDaoList.size() > 1) {
                    msg = String.format("looking for cert by padded serial '%s' failed, multiple certs found", paddedSerial);
                    LOGGER.warn(msg);
                    throw new OperationFailureException(FailInfo.badCertId);
                }
            }
        }
        if (certDaoList.isEmpty()) {
            throw new OperationFailureException(FailInfo.badCertId);
        }
        ArrayList<X509Certificate> certList = new ArrayList<X509Certificate>();
        for (Certificate certDao : certDaoList) {
            try {
                X509Certificate x509Cert = CryptoUtil.convertPemToCertificate((String)certDao.getContent());
                if (x509Cert.getIssuerX500Principal().getName().equals(issuer.toString()) || x509Cert.getIssuerX500Principal().toString().equals(issuer.toString())) {
                    LOGGER.debug("issuer match for doGetCert(" + String.valueOf(issuer) + ", " + String.valueOf(serial) + ")");
                }
                certList.add(x509Cert);
            }
            catch (GeneralSecurityException e) {
                LOGGER.warn("decoding certificate failed", (Throwable)e);
                throw new OperationFailureException(FailInfo.badRequest);
            }
        }
        return certList;
    }

    protected List<X509Certificate> doGetCertInitial(X500Name issuer, X500Name subject, TransactionId transId) {
        LOGGER.debug("doGetCertInitial(" + issuer.toString() + ", " + subject.toString() + ", " + transId.toString() + ")");
        if (subject.equals((Object)pollName)) {
            return Collections.emptyList();
        }
        return Collections.emptyList();
    }

    protected List<X509Certificate> getNextCaCertificate(String identifier) {
        LOGGER.debug("getNextCaCertificate(" + identifier + ")");
        return Collections.emptyList();
    }

    protected PrivateKey getRecipientKey() {
        try {
            return this.certUtil.getPrivateKey(this.getCurrentRecepientCert());
        }
        catch (ServletException | OperationFailureException e) {
            LOGGER.warn("problem retrieving recipient's private key", e);
            return null;
        }
    }

    protected X509Certificate getRecipient() {
        try {
            X509Certificate recipient = CryptoUtil.convertPemToCertificate((String)this.getCurrentRecepientCert().getContent());
            LOGGER.debug("getRecipient() returns " + recipient.toString());
            return recipient;
        }
        catch (GeneralSecurityException | ServletException | OperationFailureException e) {
            LOGGER.warn("problem retrieving recipient certificate", e);
            return null;
        }
    }

    protected PrivateKey getSignerKey() {
        LOGGER.debug("getSignerKey(), returning getRecipientKey()");
        return this.getRecipientKey();
    }

    protected X509Certificate getSigner() {
        LOGGER.debug("getSigner(), returning getRecipient()");
        return this.getRecipient();
    }

    protected X509Certificate[] getSignerCertificateChain() {
        LOGGER.debug("getSignerCertificateChain()");
        X509Certificate[] signerChainArr = new X509Certificate[]{};
        try {
            Certificate recepCert = this.getCurrentRecepientCert();
            List certList = this.certUtil.getCertificateChain(recepCert);
            int chainLength = certList.size();
            if (chainLength > 1) {
                signerChainArr = new X509Certificate[chainLength - 1];
                int j = 0;
                for (int i = 1; i < chainLength; ++i) {
                    signerChainArr[j++] = CryptoService.convertPemToCertificate((String)((Certificate)certList.get(i)).getContent());
                }
            }
        }
        catch (GeneralSecurityException | ServletException | OperationFailureException e) {
            LOGGER.warn("Failed to retrieve CA certificates", e);
        }
        return signerChainArr;
    }

    private void insertSANs(ScepOrder scepOrder, PKCS10CertificationRequest csr) {
        Set generalNameSet = CSRUtil.getSANList((Attribute[])csr.getAttributes());
        Object allSans = "";
        LOGGER.debug("putting SANs into ScepOrderAttributes");
        for (GeneralName gName : generalNameSet) {
            String sanValue = gName.getName().toString();
            if (0 == gName.getTagNo()) {
                sanValue = "--other value--";
            } else if (7 == gName.getTagNo()) {
                sanValue = CertificateUtil.getTypedSAN((GeneralName)gName);
            }
            if (!((String)allSans).isEmpty()) {
                allSans = (String)allSans + ";";
            }
            allSans = (String)allSans + sanValue;
            this.scepOrderUtil.setOrderAttribute(scepOrder, "SAN", sanValue, true);
            if (2 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "DNS:" + sanValue, true);
                continue;
            }
            if (7 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "IP:" + sanValue, true);
                continue;
            }
            if (5 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "EDI:" + sanValue, true);
                continue;
            }
            if (0 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "other:" + sanValue, true);
                continue;
            }
            if (8 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "regID:" + sanValue, true);
                continue;
            }
            if (1 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "rfc822:" + sanValue, true);
                continue;
            }
            if (6 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "URI:" + sanValue, true);
                continue;
            }
            if (3 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "X400:" + sanValue, true);
                continue;
            }
            if (4 == gName.getTagNo()) {
                this.scepOrderUtil.setOrderAttribute(scepOrder, "TYPED_SAN", "DirName:" + sanValue, true);
                continue;
            }
            LOGGER.info("unexpected name / tag '{}' in SANs", (Object)gName.getTagNo());
        }
    }
}

