// $Id: Whirlpool.java 214 2010-06-03 17:25:08Z tp $

package fr.cryptohash;

/**
 * <p>This class implements the Whirlpool digest algorithm under the
 * {@link Digest} API. This is the current (2003) variant of
 * Whirlpool.</p>
 *
 * <pre>
 * ==========================(LICENSE BEGIN)============================
 *
 * Copyright (c) 2007-2010  Projet RNRT SAPHIR
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ===========================(LICENSE END)=============================
 * </pre>
 *
 * @version   $Revision: 214 $
 * @author    Thomas Pornin &lt;thomas.pornin@cryptolog.com&gt;
 */

public class Whirlpool extends WhirlpoolCore {

	/**
	 * Create the object.
	 */
	public Whirlpool()
	{
		super(T0, T1, T2, T3, T4, T5, T6, T7, RC);
	}

	/** @see Digest */
	public Digest copy()
	{
		return copyState(new Whirlpool());
	}

	private static final long[] T0 = {
		0xD83078C018601818L, 0x2646AF05238C2323L,
		0xB891F97EC63FC6C6L, 0xFBCD6F13E887E8E8L,
		0xCB13A14C87268787L, 0x116D62A9B8DAB8B8L,
		0x0902050801040101L, 0x0D9E6E424F214F4FL,
		0x9B6CEEAD36D83636L, 0xFF510459A6A2A6A6L,
		0x0CB9BDDED26FD2D2L, 0x0EF706FBF5F3F5F5L,
		0x96F280EF79F97979L, 0x30DECE5F6FA16F6FL,
		0x6D3FEFFC917E9191L, 0xF8A407AA52555252L,
		0x47C0FD27609D6060L, 0x35657689BCCABCBCL,
		0x372BCDAC9B569B9BL, 0x8A018C048E028E8EL,
		0xD25B1571A3B6A3A3L, 0x6C183C600C300C0CL,
		0x84F68AFF7BF17B7BL, 0x806AE1B535D43535L,
		0xF53A69E81D741D1DL, 0xB3DD4753E0A7E0E0L,
		0x21B3ACF6D77BD7D7L, 0x9C99ED5EC22FC2C2L,
		0x435C966D2EB82E2EL, 0x29967A624B314B4BL,
		0x5DE121A3FEDFFEFEL, 0xD5AE168257415757L,
		0xBD2A41A815541515L, 0xE8EEB69F77C17777L,
		0x926EEBA537DC3737L, 0x9ED7567BE5B3E5E5L,
		0x1323D98C9F469F9FL, 0x23FD17D3F0E7F0F0L,
		0x20947F6A4A354A4AL, 0x44A9959EDA4FDADAL,
		0xA2B025FA587D5858L, 0xCF8FCA06C903C9C9L,
		0x7C528D5529A42929L, 0x5A1422500A280A0AL,
		0x507F4FE1B1FEB1B1L, 0xC95D1A69A0BAA0A0L,
		0x14D6DA7F6BB16B6BL, 0xD917AB5C852E8585L,
		0x3C677381BDCEBDBDL, 0x8FBA34D25D695D5DL,
		0x9020508010401010L, 0x07F503F3F4F7F4F4L,
		0xDD8BC016CB0BCBCBL, 0xD37CC6ED3EF83E3EL,
		0x2D0A112805140505L, 0x78CEE61F67816767L,
		0x97D55373E4B7E4E4L, 0x024EBB25279C2727L,
		0x7382583241194141L, 0xA70B9D2C8B168B8BL,
		0xF6530151A7A6A7A7L, 0xB2FA94CF7DE97D7DL,
		0x4937FBDC956E9595L, 0x56AD9F8ED847D8D8L,
		0x70EB308BFBCBFBFBL, 0xCDC17123EE9FEEEEL,
		0xBBF891C77CED7C7CL, 0x71CCE31766856666L,
		0x7BA78EA6DD53DDDDL, 0xAF2E4BB8175C1717L,
		0x458E460247014747L, 0x1A21DC849E429E9EL,
		0xD489C51ECA0FCACAL, 0x585A99752DB42D2DL,
		0x2E637991BFC6BFBFL, 0x3F0E1B38071C0707L,
		0xAC472301AD8EADADL, 0xB0B42FEA5A755A5AL,
		0xEF1BB56C83368383L, 0xB666FF8533CC3333L,
		0x5CC6F23F63916363L, 0x12040A1002080202L,
		0x93493839AA92AAAAL, 0xDEE2A8AF71D97171L,
		0xC68DCF0EC807C8C8L, 0xD1327DC819641919L,
		0x3B92707249394949L, 0x5FAF9A86D943D9D9L,
		0x31F91DC3F2EFF2F2L, 0xA8DB484BE3ABE3E3L,
		0xB9B62AE25B715B5BL, 0xBC0D9234881A8888L,
		0x3E29C8A49A529A9AL, 0x0B4CBE2D26982626L,
		0xBF64FA8D32C83232L, 0x597D4AE9B0FAB0B0L,
		0xF2CF6A1BE983E9E9L, 0x771E33780F3C0F0FL,
		0x33B7A6E6D573D5D5L, 0xF41DBA74803A8080L,
		0x27617C99BEC2BEBEL, 0xEB87DE26CD13CDCDL,
		0x8968E4BD34D03434L, 0x3290757A483D4848L,
		0x54E324ABFFDBFFFFL, 0x8DF48FF77AF57A7AL,
		0x643DEAF4907A9090L, 0x9DBE3EC25F615F5FL,
		0x3D40A01D20802020L, 0x0FD0D56768BD6868L,
		0xCA3472D01A681A1AL, 0xB7412C19AE82AEAEL,
		0x7D755EC9B4EAB4B4L, 0xCEA8199A544D5454L,
		0x7F3BE5EC93769393L, 0x2F44AA0D22882222L,
		0x63C8E907648D6464L, 0x2AFF12DBF1E3F1F1L,
		0xCCE6A2BF73D17373L, 0x82245A9012481212L,
		0x7A805D3A401D4040L, 0x4810284008200808L,
		0x959BE856C32BC3C3L, 0xDFC57B33EC97ECECL,
		0x4DAB9096DB4BDBDBL, 0xC05F1F61A1BEA1A1L,
		0x9107831C8D0E8D8DL, 0xC87AC9F53DF43D3DL,
		0x5B33F1CC97669797L, 0x0000000000000000L,
		0xF983D436CF1BCFCFL, 0x6E5687452BAC2B2BL,
		0xE1ECB39776C57676L, 0xE619B06482328282L,
		0x28B1A9FED67FD6D6L, 0xC33677D81B6C1B1BL,
		0x74775BC1B5EEB5B5L, 0xBE432911AF86AFAFL,
		0x1DD4DF776AB56A6AL, 0xEAA00DBA505D5050L,
		0x578A4C1245094545L, 0x38FB18CBF3EBF3F3L,
		0xAD60F09D30C03030L, 0xC4C3742BEF9BEFEFL,
		0xDA7EC3E53FFC3F3FL, 0xC7AA1C9255495555L,
		0xDB591079A2B2A2A2L, 0xE9C96503EA8FEAEAL,
		0x6ACAEC0F65896565L, 0x036968B9BAD2BABAL,
		0x4A5E93652FBC2F2FL, 0x8E9DE74EC027C0C0L,
		0x60A181BEDE5FDEDEL, 0xFC386CE01C701C1CL,
		0x46E72EBBFDD3FDFDL, 0x1F9A64524D294D4DL,
		0x7639E0E492729292L, 0xFAEABC8F75C97575L,
		0x360C1E3006180606L, 0xAE0998248A128A8AL,
		0x4B7940F9B2F2B2B2L, 0x85D15963E6BFE6E6L,
		0x7E1C36700E380E0EL, 0xE73E63F81F7C1F1FL,
		0x55C4F73762956262L, 0x3AB5A3EED477D4D4L,
		0x814D3229A89AA8A8L, 0x5231F4C496629696L,
		0x62EF3A9BF9C3F9F9L, 0xA397F666C533C5C5L,
		0x104AB13525942525L, 0xABB220F259795959L,
		0xD015AE54842A8484L, 0xC5E4A7B772D57272L,
		0xEC72DDD539E43939L, 0x1698615A4C2D4C4CL,
		0x94BC3BCA5E655E5EL, 0x9FF085E778FD7878L,
		0xE570D8DD38E03838L, 0x980586148C0A8C8CL,
		0x17BFB2C6D163D1D1L, 0xE4570B41A5AEA5A5L,
		0xA1D94D43E2AFE2E2L, 0x4EC2F82F61996161L,
		0x427B45F1B3F6B3B3L, 0x3442A51521842121L,
		0x0825D6949C4A9C9CL, 0xEE3C66F01E781E1EL,
		0x6186522243114343L, 0xB193FC76C73BC7C7L,
		0x4FE52BB3FCD7FCFCL, 0x2408142004100404L,
		0xE3A208B251595151L, 0x252FC7BC995E9999L,
		0x22DAC44F6DA96D6DL, 0x651A39680D340D0DL,
		0x79E93583FACFFAFAL, 0x69A384B6DF5BDFDFL,
		0xA9FC9BD77EE57E7EL, 0x1948B43D24902424L,
		0xFE76D7C53BEC3B3BL, 0x9A4B3D31AB96ABABL,
		0xF081D13ECE1FCECEL, 0x9922558811441111L,
		0x8303890C8F068F8FL, 0x049C6B4A4E254E4EL,
		0x667351D1B7E6B7B7L, 0xE0CB600BEB8BEBEBL,
		0xC178CCFD3CF03C3CL, 0xFD1FBF7C813E8181L,
		0x4035FED4946A9494L, 0x1CF30CEBF7FBF7F7L,
		0x186F67A1B9DEB9B9L, 0x8B265F98134C1313L,
		0x51589C7D2CB02C2CL, 0x05BBB8D6D36BD3D3L,
		0x8CD35C6BE7BBE7E7L, 0x39DCCB576EA56E6EL,
		0xAA95F36EC437C4C4L, 0x1B060F18030C0303L,
		0xDCAC138A56455656L, 0x5E88491A440D4444L,
		0xA0FE9EDF7FE17F7FL, 0x884F3721A99EA9A9L,
		0x6754824D2AA82A2AL, 0x0A6B6DB1BBD6BBBBL,
		0x879FE246C123C1C1L, 0xF1A602A253515353L,
		0x72A58BAEDC57DCDCL, 0x531627580B2C0B0BL,
		0x0127D39C9D4E9D9DL, 0x2BD8C1476CAD6C6CL,
		0xA462F59531C43131L, 0xF3E8B98774CD7474L,
		0x15F109E3F6FFF6F6L, 0x4C8C430A46054646L,
		0xA5452609AC8AACACL, 0xB50F973C891E8989L,
		0xB42844A014501414L, 0xBADF425BE1A3E1E1L,
		0xA62C4EB016581616L, 0xF774D2CD3AE83A3AL,
		0x06D2D06F69B96969L, 0x41122D4809240909L,
		0xD7E0ADA770DD7070L, 0x6F7154D9B6E2B6B6L,
		0x1EBDB7CED067D0D0L, 0xD6C77E3BED93EDEDL,
		0xE285DB2ECC17CCCCL, 0x6884572A42154242L,
		0x2C2DC2B4985A9898L, 0xED550E49A4AAA4A4L,
		0x7550885D28A02828L, 0x86B831DA5C6D5C5CL,
		0x6BED3F93F8C7F8F8L, 0xC211A44486228686L
	};

	private static final long[] T1 = {
		0x3078C018601818D8L, 0x46AF05238C232326L,
		0x91F97EC63FC6C6B8L, 0xCD6F13E887E8E8FBL,
		0x13A14C87268787CBL, 0x6D62A9B8DAB8B811L,
		0x0205080104010109L, 0x9E6E424F214F4F0DL,
		0x6CEEAD36D836369BL, 0x510459A6A2A6A6FFL,
		0xB9BDDED26FD2D20CL, 0xF706FBF5F3F5F50EL,
		0xF280EF79F9797996L, 0xDECE5F6FA16F6F30L,
		0x3FEFFC917E91916DL, 0xA407AA52555252F8L,
		0xC0FD27609D606047L, 0x657689BCCABCBC35L,
		0x2BCDAC9B569B9B37L, 0x018C048E028E8E8AL,
		0x5B1571A3B6A3A3D2L, 0x183C600C300C0C6CL,
		0xF68AFF7BF17B7B84L, 0x6AE1B535D4353580L,
		0x3A69E81D741D1DF5L, 0xDD4753E0A7E0E0B3L,
		0xB3ACF6D77BD7D721L, 0x99ED5EC22FC2C29CL,
		0x5C966D2EB82E2E43L, 0x967A624B314B4B29L,
		0xE121A3FEDFFEFE5DL, 0xAE168257415757D5L,
		0x2A41A815541515BDL, 0xEEB69F77C17777E8L,
		0x6EEBA537DC373792L, 0xD7567BE5B3E5E59EL,
		0x23D98C9F469F9F13L, 0xFD17D3F0E7F0F023L,
		0x947F6A4A354A4A20L, 0xA9959EDA4FDADA44L,
		0xB025FA587D5858A2L, 0x8FCA06C903C9C9CFL,
		0x528D5529A429297CL, 0x1422500A280A0A5AL,
		0x7F4FE1B1FEB1B150L, 0x5D1A69A0BAA0A0C9L,
		0xD6DA7F6BB16B6B14L, 0x17AB5C852E8585D9L,
		0x677381BDCEBDBD3CL, 0xBA34D25D695D5D8FL,
		0x2050801040101090L, 0xF503F3F4F7F4F407L,
		0x8BC016CB0BCBCBDDL, 0x7CC6ED3EF83E3ED3L,
		0x0A1128051405052DL, 0xCEE61F6781676778L,
		0xD55373E4B7E4E497L, 0x4EBB25279C272702L,
		0x8258324119414173L, 0x0B9D2C8B168B8BA7L,
		0x530151A7A6A7A7F6L, 0xFA94CF7DE97D7DB2L,
		0x37FBDC956E959549L, 0xAD9F8ED847D8D856L,
		0xEB308BFBCBFBFB70L, 0xC17123EE9FEEEECDL,
		0xF891C77CED7C7CBBL, 0xCCE3176685666671L,
		0xA78EA6DD53DDDD7BL, 0x2E4BB8175C1717AFL,
		0x8E46024701474745L, 0x21DC849E429E9E1AL,
		0x89C51ECA0FCACAD4L, 0x5A99752DB42D2D58L,
		0x637991BFC6BFBF2EL, 0x0E1B38071C07073FL,
		0x472301AD8EADADACL, 0xB42FEA5A755A5AB0L,
		0x1BB56C83368383EFL, 0x66FF8533CC3333B6L,
		0xC6F23F639163635CL, 0x040A100208020212L,
		0x493839AA92AAAA93L, 0xE2A8AF71D97171DEL,
		0x8DCF0EC807C8C8C6L, 0x327DC819641919D1L,
		0x927072493949493BL, 0xAF9A86D943D9D95FL,
		0xF91DC3F2EFF2F231L, 0xDB484BE3ABE3E3A8L,
		0xB62AE25B715B5BB9L, 0x0D9234881A8888BCL,
		0x29C8A49A529A9A3EL, 0x4CBE2D269826260BL,
		0x64FA8D32C83232BFL, 0x7D4AE9B0FAB0B059L,
		0xCF6A1BE983E9E9F2L, 0x1E33780F3C0F0F77L,
		0xB7A6E6D573D5D533L, 0x1DBA74803A8080F4L,
		0x617C99BEC2BEBE27L, 0x87DE26CD13CDCDEBL,
		0x68E4BD34D0343489L, 0x90757A483D484832L,
		0xE324ABFFDBFFFF54L, 0xF48FF77AF57A7A8DL,
		0x3DEAF4907A909064L, 0xBE3EC25F615F5F9DL,
		0x40A01D208020203DL, 0xD0D56768BD68680FL,
		0x3472D01A681A1ACAL, 0x412C19AE82AEAEB7L,
		0x755EC9B4EAB4B47DL, 0xA8199A544D5454CEL,
		0x3BE5EC937693937FL, 0x44AA0D228822222FL,
		0xC8E907648D646463L, 0xFF12DBF1E3F1F12AL,
		0xE6A2BF73D17373CCL, 0x245A901248121282L,
		0x805D3A401D40407AL, 0x1028400820080848L,
		0x9BE856C32BC3C395L, 0xC57B33EC97ECECDFL,
		0xAB9096DB4BDBDB4DL, 0x5F1F61A1BEA1A1C0L,
		0x07831C8D0E8D8D91L, 0x7AC9F53DF43D3DC8L,
		0x33F1CC976697975BL, 0x0000000000000000L,
		0x83D436CF1BCFCFF9L, 0x5687452BAC2B2B6EL,
		0xECB39776C57676E1L, 0x19B06482328282E6L,
		0xB1A9FED67FD6D628L, 0x3677D81B6C1B1BC3L,
		0x775BC1B5EEB5B574L, 0x432911AF86AFAFBEL,
		0xD4DF776AB56A6A1DL, 0xA00DBA505D5050EAL,
		0x8A4C124509454557L, 0xFB18CBF3EBF3F338L,
		0x60F09D30C03030ADL, 0xC3742BEF9BEFEFC4L,
		0x7EC3E53FFC3F3FDAL, 0xAA1C9255495555C7L,
		0x591079A2B2A2A2DBL, 0xC96503EA8FEAEAE9L,
		0xCAEC0F658965656AL, 0x6968B9BAD2BABA03L,
		0x5E93652FBC2F2F4AL, 0x9DE74EC027C0C08EL,
		0xA181BEDE5FDEDE60L, 0x386CE01C701C1CFCL,
		0xE72EBBFDD3FDFD46L, 0x9A64524D294D4D1FL,
		0x39E0E49272929276L, 0xEABC8F75C97575FAL,
		0x0C1E300618060636L, 0x0998248A128A8AAEL,
		0x7940F9B2F2B2B24BL, 0xD15963E6BFE6E685L,
		0x1C36700E380E0E7EL, 0x3E63F81F7C1F1FE7L,
		0xC4F7376295626255L, 0xB5A3EED477D4D43AL,
		0x4D3229A89AA8A881L, 0x31F4C49662969652L,
		0xEF3A9BF9C3F9F962L, 0x97F666C533C5C5A3L,
		0x4AB1352594252510L, 0xB220F259795959ABL,
		0x15AE54842A8484D0L, 0xE4A7B772D57272C5L,
		0x72DDD539E43939ECL, 0x98615A4C2D4C4C16L,
		0xBC3BCA5E655E5E94L, 0xF085E778FD78789FL,
		0x70D8DD38E03838E5L, 0x0586148C0A8C8C98L,
		0xBFB2C6D163D1D117L, 0x570B41A5AEA5A5E4L,
		0xD94D43E2AFE2E2A1L, 0xC2F82F619961614EL,
		0x7B45F1B3F6B3B342L, 0x42A5152184212134L,
		0x25D6949C4A9C9C08L, 0x3C66F01E781E1EEEL,
		0x8652224311434361L, 0x93FC76C73BC7C7B1L,
		0xE52BB3FCD7FCFC4FL, 0x0814200410040424L,
		0xA208B251595151E3L, 0x2FC7BC995E999925L,
		0xDAC44F6DA96D6D22L, 0x1A39680D340D0D65L,
		0xE93583FACFFAFA79L, 0xA384B6DF5BDFDF69L,
		0xFC9BD77EE57E7EA9L, 0x48B43D2490242419L,
		0x76D7C53BEC3B3BFEL, 0x4B3D31AB96ABAB9AL,
		0x81D13ECE1FCECEF0L, 0x2255881144111199L,
		0x03890C8F068F8F83L, 0x9C6B4A4E254E4E04L,
		0x7351D1B7E6B7B766L, 0xCB600BEB8BEBEBE0L,
		0x78CCFD3CF03C3CC1L, 0x1FBF7C813E8181FDL,
		0x35FED4946A949440L, 0xF30CEBF7FBF7F71CL,
		0x6F67A1B9DEB9B918L, 0x265F98134C13138BL,
		0x589C7D2CB02C2C51L, 0xBBB8D6D36BD3D305L,
		0xD35C6BE7BBE7E78CL, 0xDCCB576EA56E6E39L,
		0x95F36EC437C4C4AAL, 0x060F18030C03031BL,
		0xAC138A56455656DCL, 0x88491A440D44445EL,
		0xFE9EDF7FE17F7FA0L, 0x4F3721A99EA9A988L,
		0x54824D2AA82A2A67L, 0x6B6DB1BBD6BBBB0AL,
		0x9FE246C123C1C187L, 0xA602A253515353F1L,
		0xA58BAEDC57DCDC72L, 0x1627580B2C0B0B53L,
		0x27D39C9D4E9D9D01L, 0xD8C1476CAD6C6C2BL,
		0x62F59531C43131A4L, 0xE8B98774CD7474F3L,
		0xF109E3F6FFF6F615L, 0x8C430A460546464CL,
		0x452609AC8AACACA5L, 0x0F973C891E8989B5L,
		0x2844A014501414B4L, 0xDF425BE1A3E1E1BAL,
		0x2C4EB016581616A6L, 0x74D2CD3AE83A3AF7L,
		0xD2D06F69B9696906L, 0x122D480924090941L,
		0xE0ADA770DD7070D7L, 0x7154D9B6E2B6B66FL,
		0xBDB7CED067D0D01EL, 0xC77E3BED93EDEDD6L,
		0x85DB2ECC17CCCCE2L, 0x84572A4215424268L,
		0x2DC2B4985A98982CL, 0x550E49A4AAA4A4EDL,
		0x50885D28A0282875L, 0xB831DA5C6D5C5C86L,
		0xED3F93F8C7F8F86BL, 0x11A44486228686C2L
	};

	private static final long[] T2 = {
		0x78C018601818D830L, 0xAF05238C23232646L,
		0xF97EC63FC6C6B891L, 0x6F13E887E8E8FBCDL,
		0xA14C87268787CB13L, 0x62A9B8DAB8B8116DL,
		0x0508010401010902L, 0x6E424F214F4F0D9EL,
		0xEEAD36D836369B6CL, 0x0459A6A2A6A6FF51L,
		0xBDDED26FD2D20CB9L, 0x06FBF5F3F5F50EF7L,
		0x80EF79F9797996F2L, 0xCE5F6FA16F6F30DEL,
		0xEFFC917E91916D3FL, 0x07AA52555252F8A4L,
		0xFD27609D606047C0L, 0x7689BCCABCBC3565L,
		0xCDAC9B569B9B372BL, 0x8C048E028E8E8A01L,
		0x1571A3B6A3A3D25BL, 0x3C600C300C0C6C18L,
		0x8AFF7BF17B7B84F6L, 0xE1B535D43535806AL,
		0x69E81D741D1DF53AL, 0x4753E0A7E0E0B3DDL,
		0xACF6D77BD7D721B3L, 0xED5EC22FC2C29C99L,
		0x966D2EB82E2E435CL, 0x7A624B314B4B2996L,
		0x21A3FEDFFEFE5DE1L, 0x168257415757D5AEL,
		0x41A815541515BD2AL, 0xB69F77C17777E8EEL,
		0xEBA537DC3737926EL, 0x567BE5B3E5E59ED7L,
		0xD98C9F469F9F1323L, 0x17D3F0E7F0F023FDL,
		0x7F6A4A354A4A2094L, 0x959EDA4FDADA44A9L,
		0x25FA587D5858A2B0L, 0xCA06C903C9C9CF8FL,
		0x8D5529A429297C52L, 0x22500A280A0A5A14L,
		0x4FE1B1FEB1B1507FL, 0x1A69A0BAA0A0C95DL,
		0xDA7F6BB16B6B14D6L, 0xAB5C852E8585D917L,
		0x7381BDCEBDBD3C67L, 0x34D25D695D5D8FBAL,
		0x5080104010109020L, 0x03F3F4F7F4F407F5L,
		0xC016CB0BCBCBDD8BL, 0xC6ED3EF83E3ED37CL,
		0x1128051405052D0AL, 0xE61F6781676778CEL,
		0x5373E4B7E4E497D5L, 0xBB25279C2727024EL,
		0x5832411941417382L, 0x9D2C8B168B8BA70BL,
		0x0151A7A6A7A7F653L, 0x94CF7DE97D7DB2FAL,
		0xFBDC956E95954937L, 0x9F8ED847D8D856ADL,
		0x308BFBCBFBFB70EBL, 0x7123EE9FEEEECDC1L,
		0x91C77CED7C7CBBF8L, 0xE3176685666671CCL,
		0x8EA6DD53DDDD7BA7L, 0x4BB8175C1717AF2EL,
		0x460247014747458EL, 0xDC849E429E9E1A21L,
		0xC51ECA0FCACAD489L, 0x99752DB42D2D585AL,
		0x7991BFC6BFBF2E63L, 0x1B38071C07073F0EL,
		0x2301AD8EADADAC47L, 0x2FEA5A755A5AB0B4L,
		0xB56C83368383EF1BL, 0xFF8533CC3333B666L,
		0xF23F639163635CC6L, 0x0A10020802021204L,
		0x3839AA92AAAA9349L, 0xA8AF71D97171DEE2L,
		0xCF0EC807C8C8C68DL, 0x7DC819641919D132L,
		0x7072493949493B92L, 0x9A86D943D9D95FAFL,
		0x1DC3F2EFF2F231F9L, 0x484BE3ABE3E3A8DBL,
		0x2AE25B715B5BB9B6L, 0x9234881A8888BC0DL,
		0xC8A49A529A9A3E29L, 0xBE2D269826260B4CL,
		0xFA8D32C83232BF64L, 0x4AE9B0FAB0B0597DL,
		0x6A1BE983E9E9F2CFL, 0x33780F3C0F0F771EL,
		0xA6E6D573D5D533B7L, 0xBA74803A8080F41DL,
		0x7C99BEC2BEBE2761L, 0xDE26CD13CDCDEB87L,
		0xE4BD34D034348968L, 0x757A483D48483290L,
		0x24ABFFDBFFFF54E3L, 0x8FF77AF57A7A8DF4L,
		0xEAF4907A9090643DL, 0x3EC25F615F5F9DBEL,
		0xA01D208020203D40L, 0xD56768BD68680FD0L,
		0x72D01A681A1ACA34L, 0x2C19AE82AEAEB741L,
		0x5EC9B4EAB4B47D75L, 0x199A544D5454CEA8L,
		0xE5EC937693937F3BL, 0xAA0D228822222F44L,
		0xE907648D646463C8L, 0x12DBF1E3F1F12AFFL,
		0xA2BF73D17373CCE6L, 0x5A90124812128224L,
		0x5D3A401D40407A80L, 0x2840082008084810L,
		0xE856C32BC3C3959BL, 0x7B33EC97ECECDFC5L,
		0x9096DB4BDBDB4DABL, 0x1F61A1BEA1A1C05FL,
		0x831C8D0E8D8D9107L, 0xC9F53DF43D3DC87AL,
		0xF1CC976697975B33L, 0x0000000000000000L,
		0xD436CF1BCFCFF983L, 0x87452BAC2B2B6E56L,
		0xB39776C57676E1ECL, 0xB06482328282E619L,
		0xA9FED67FD6D628B1L, 0x77D81B6C1B1BC336L,
		0x5BC1B5EEB5B57477L, 0x2911AF86AFAFBE43L,
		0xDF776AB56A6A1DD4L, 0x0DBA505D5050EAA0L,
		0x4C1245094545578AL, 0x18CBF3EBF3F338FBL,
		0xF09D30C03030AD60L, 0x742BEF9BEFEFC4C3L,
		0xC3E53FFC3F3FDA7EL, 0x1C9255495555C7AAL,
		0x1079A2B2A2A2DB59L, 0x6503EA8FEAEAE9C9L,
		0xEC0F658965656ACAL, 0x68B9BAD2BABA0369L,
		0x93652FBC2F2F4A5EL, 0xE74EC027C0C08E9DL,
		0x81BEDE5FDEDE60A1L, 0x6CE01C701C1CFC38L,
		0x2EBBFDD3FDFD46E7L, 0x64524D294D4D1F9AL,
		0xE0E4927292927639L, 0xBC8F75C97575FAEAL,
		0x1E3006180606360CL, 0x98248A128A8AAE09L,
		0x40F9B2F2B2B24B79L, 0x5963E6BFE6E685D1L,
		0x36700E380E0E7E1CL, 0x63F81F7C1F1FE73EL,
		0xF7376295626255C4L, 0xA3EED477D4D43AB5L,
		0x3229A89AA8A8814DL, 0xF4C4966296965231L,
		0x3A9BF9C3F9F962EFL, 0xF666C533C5C5A397L,
		0xB13525942525104AL, 0x20F259795959ABB2L,
		0xAE54842A8484D015L, 0xA7B772D57272C5E4L,
		0xDDD539E43939EC72L, 0x615A4C2D4C4C1698L,
		0x3BCA5E655E5E94BCL, 0x85E778FD78789FF0L,
		0xD8DD38E03838E570L, 0x86148C0A8C8C9805L,
		0xB2C6D163D1D117BFL, 0x0B41A5AEA5A5E457L,
		0x4D43E2AFE2E2A1D9L, 0xF82F619961614EC2L,
		0x45F1B3F6B3B3427BL, 0xA515218421213442L,
		0xD6949C4A9C9C0825L, 0x66F01E781E1EEE3CL,
		0x5222431143436186L, 0xFC76C73BC7C7B193L,
		0x2BB3FCD7FCFC4FE5L, 0x1420041004042408L,
		0x08B251595151E3A2L, 0xC7BC995E9999252FL,
		0xC44F6DA96D6D22DAL, 0x39680D340D0D651AL,
		0x3583FACFFAFA79E9L, 0x84B6DF5BDFDF69A3L,
		0x9BD77EE57E7EA9FCL, 0xB43D249024241948L,
		0xD7C53BEC3B3BFE76L, 0x3D31AB96ABAB9A4BL,
		0xD13ECE1FCECEF081L, 0x5588114411119922L,
		0x890C8F068F8F8303L, 0x6B4A4E254E4E049CL,
		0x51D1B7E6B7B76673L, 0x600BEB8BEBEBE0CBL,
		0xCCFD3CF03C3CC178L, 0xBF7C813E8181FD1FL,
		0xFED4946A94944035L, 0x0CEBF7FBF7F71CF3L,
		0x67A1B9DEB9B9186FL, 0x5F98134C13138B26L,
		0x9C7D2CB02C2C5158L, 0xB8D6D36BD3D305BBL,
		0x5C6BE7BBE7E78CD3L, 0xCB576EA56E6E39DCL,
		0xF36EC437C4C4AA95L, 0x0F18030C03031B06L,
		0x138A56455656DCACL, 0x491A440D44445E88L,
		0x9EDF7FE17F7FA0FEL, 0x3721A99EA9A9884FL,
		0x824D2AA82A2A6754L, 0x6DB1BBD6BBBB0A6BL,
		0xE246C123C1C1879FL, 0x02A253515353F1A6L,
		0x8BAEDC57DCDC72A5L, 0x27580B2C0B0B5316L,
		0xD39C9D4E9D9D0127L, 0xC1476CAD6C6C2BD8L,
		0xF59531C43131A462L, 0xB98774CD7474F3E8L,
		0x09E3F6FFF6F615F1L, 0x430A460546464C8CL,
		0x2609AC8AACACA545L, 0x973C891E8989B50FL,
		0x44A014501414B428L, 0x425BE1A3E1E1BADFL,
		0x4EB016581616A62CL, 0xD2CD3AE83A3AF774L,
		0xD06F69B9696906D2L, 0x2D48092409094112L,
		0xADA770DD7070D7E0L, 0x54D9B6E2B6B66F71L,
		0xB7CED067D0D01EBDL, 0x7E3BED93EDEDD6C7L,
		0xDB2ECC17CCCCE285L, 0x572A421542426884L,
		0xC2B4985A98982C2DL, 0x0E49A4AAA4A4ED55L,
		0x885D28A028287550L, 0x31DA5C6D5C5C86B8L,
		0x3F93F8C7F8F86BEDL, 0xA44486228686C211L
	};

	private static final long[] T3 = {
		0xC018601818D83078L, 0x05238C23232646AFL,
		0x7EC63FC6C6B891F9L, 0x13E887E8E8FBCD6FL,
		0x4C87268787CB13A1L, 0xA9B8DAB8B8116D62L,
		0x0801040101090205L, 0x424F214F4F0D9E6EL,
		0xAD36D836369B6CEEL, 0x59A6A2A6A6FF5104L,
		0xDED26FD2D20CB9BDL, 0xFBF5F3F5F50EF706L,
		0xEF79F9797996F280L, 0x5F6FA16F6F30DECEL,
		0xFC917E91916D3FEFL, 0xAA52555252F8A407L,
		0x27609D606047C0FDL, 0x89BCCABCBC356576L,
		0xAC9B569B9B372BCDL, 0x048E028E8E8A018CL,
		0x71A3B6A3A3D25B15L, 0x600C300C0C6C183CL,
		0xFF7BF17B7B84F68AL, 0xB535D43535806AE1L,
		0xE81D741D1DF53A69L, 0x53E0A7E0E0B3DD47L,
		0xF6D77BD7D721B3ACL, 0x5EC22FC2C29C99EDL,
		0x6D2EB82E2E435C96L, 0x624B314B4B29967AL,
		0xA3FEDFFEFE5DE121L, 0x8257415757D5AE16L,
		0xA815541515BD2A41L, 0x9F77C17777E8EEB6L,
		0xA537DC3737926EEBL, 0x7BE5B3E5E59ED756L,
		0x8C9F469F9F1323D9L, 0xD3F0E7F0F023FD17L,
		0x6A4A354A4A20947FL, 0x9EDA4FDADA44A995L,
		0xFA587D5858A2B025L, 0x06C903C9C9CF8FCAL,
		0x5529A429297C528DL, 0x500A280A0A5A1422L,
		0xE1B1FEB1B1507F4FL, 0x69A0BAA0A0C95D1AL,
		0x7F6BB16B6B14D6DAL, 0x5C852E8585D917ABL,
		0x81BDCEBDBD3C6773L, 0xD25D695D5D8FBA34L,
		0x8010401010902050L, 0xF3F4F7F4F407F503L,
		0x16CB0BCBCBDD8BC0L, 0xED3EF83E3ED37CC6L,
		0x28051405052D0A11L, 0x1F6781676778CEE6L,
		0x73E4B7E4E497D553L, 0x25279C2727024EBBL,
		0x3241194141738258L, 0x2C8B168B8BA70B9DL,
		0x51A7A6A7A7F65301L, 0xCF7DE97D7DB2FA94L,
		0xDC956E95954937FBL, 0x8ED847D8D856AD9FL,
		0x8BFBCBFBFB70EB30L, 0x23EE9FEEEECDC171L,
		0xC77CED7C7CBBF891L, 0x176685666671CCE3L,
		0xA6DD53DDDD7BA78EL, 0xB8175C1717AF2E4BL,
		0x0247014747458E46L, 0x849E429E9E1A21DCL,
		0x1ECA0FCACAD489C5L, 0x752DB42D2D585A99L,
		0x91BFC6BFBF2E6379L, 0x38071C07073F0E1BL,
		0x01AD8EADADAC4723L, 0xEA5A755A5AB0B42FL,
		0x6C83368383EF1BB5L, 0x8533CC3333B666FFL,
		0x3F639163635CC6F2L, 0x100208020212040AL,
		0x39AA92AAAA934938L, 0xAF71D97171DEE2A8L,
		0x0EC807C8C8C68DCFL, 0xC819641919D1327DL,
		0x72493949493B9270L, 0x86D943D9D95FAF9AL,
		0xC3F2EFF2F231F91DL, 0x4BE3ABE3E3A8DB48L,
		0xE25B715B5BB9B62AL, 0x34881A8888BC0D92L,
		0xA49A529A9A3E29C8L, 0x2D269826260B4CBEL,
		0x8D32C83232BF64FAL, 0xE9B0FAB0B0597D4AL,
		0x1BE983E9E9F2CF6AL, 0x780F3C0F0F771E33L,
		0xE6D573D5D533B7A6L, 0x74803A8080F41DBAL,
		0x99BEC2BEBE27617CL, 0x26CD13CDCDEB87DEL,
		0xBD34D034348968E4L, 0x7A483D4848329075L,
		0xABFFDBFFFF54E324L, 0xF77AF57A7A8DF48FL,
		0xF4907A9090643DEAL, 0xC25F615F5F9DBE3EL,
		0x1D208020203D40A0L, 0x6768BD68680FD0D5L,
		0xD01A681A1ACA3472L, 0x19AE82AEAEB7412CL,
		0xC9B4EAB4B47D755EL, 0x9A544D5454CEA819L,
		0xEC937693937F3BE5L, 0x0D228822222F44AAL,
		0x07648D646463C8E9L, 0xDBF1E3F1F12AFF12L,
		0xBF73D17373CCE6A2L, 0x901248121282245AL,
		0x3A401D40407A805DL, 0x4008200808481028L,
		0x56C32BC3C3959BE8L, 0x33EC97ECECDFC57BL,
		0x96DB4BDBDB4DAB90L, 0x61A1BEA1A1C05F1FL,
		0x1C8D0E8D8D910783L, 0xF53DF43D3DC87AC9L,
		0xCC976697975B33F1L, 0x0000000000000000L,
		0x36CF1BCFCFF983D4L, 0x452BAC2B2B6E5687L,
		0x9776C57676E1ECB3L, 0x6482328282E619B0L,
		0xFED67FD6D628B1A9L, 0xD81B6C1B1BC33677L,
		0xC1B5EEB5B574775BL, 0x11AF86AFAFBE4329L,
		0x776AB56A6A1DD4DFL, 0xBA505D5050EAA00DL,
		0x1245094545578A4CL, 0xCBF3EBF3F338FB18L,
		0x9D30C03030AD60F0L, 0x2BEF9BEFEFC4C374L,
		0xE53FFC3F3FDA7EC3L, 0x9255495555C7AA1CL,
		0x79A2B2A2A2DB5910L, 0x03EA8FEAEAE9C965L,
		0x0F658965656ACAECL, 0xB9BAD2BABA036968L,
		0x652FBC2F2F4A5E93L, 0x4EC027C0C08E9DE7L,
		0xBEDE5FDEDE60A181L, 0xE01C701C1CFC386CL,
		0xBBFDD3FDFD46E72EL, 0x524D294D4D1F9A64L,
		0xE4927292927639E0L, 0x8F75C97575FAEABCL,
		0x3006180606360C1EL, 0x248A128A8AAE0998L,
		0xF9B2F2B2B24B7940L, 0x63E6BFE6E685D159L,
		0x700E380E0E7E1C36L, 0xF81F7C1F1FE73E63L,
		0x376295626255C4F7L, 0xEED477D4D43AB5A3L,
		0x29A89AA8A8814D32L, 0xC4966296965231F4L,
		0x9BF9C3F9F962EF3AL, 0x66C533C5C5A397F6L,
		0x3525942525104AB1L, 0xF259795959ABB220L,
		0x54842A8484D015AEL, 0xB772D57272C5E4A7L,
		0xD539E43939EC72DDL, 0x5A4C2D4C4C169861L,
		0xCA5E655E5E94BC3BL, 0xE778FD78789FF085L,
		0xDD38E03838E570D8L, 0x148C0A8C8C980586L,
		0xC6D163D1D117BFB2L, 0x41A5AEA5A5E4570BL,
		0x43E2AFE2E2A1D94DL, 0x2F619961614EC2F8L,
		0xF1B3F6B3B3427B45L, 0x15218421213442A5L,
		0x949C4A9C9C0825D6L, 0xF01E781E1EEE3C66L,
		0x2243114343618652L, 0x76C73BC7C7B193FCL,
		0xB3FCD7FCFC4FE52BL, 0x2004100404240814L,
		0xB251595151E3A208L, 0xBC995E9999252FC7L,
		0x4F6DA96D6D22DAC4L, 0x680D340D0D651A39L,
		0x83FACFFAFA79E935L, 0xB6DF5BDFDF69A384L,
		0xD77EE57E7EA9FC9BL, 0x3D249024241948B4L,
		0xC53BEC3B3BFE76D7L, 0x31AB96ABAB9A4B3DL,
		0x3ECE1FCECEF081D1L, 0x8811441111992255L,
		0x0C8F068F8F830389L, 0x4A4E254E4E049C6BL,
		0xD1B7E6B7B7667351L, 0x0BEB8BEBEBE0CB60L,
		0xFD3CF03C3CC178CCL, 0x7C813E8181FD1FBFL,
		0xD4946A94944035FEL, 0xEBF7FBF7F71CF30CL,
		0xA1B9DEB9B9186F67L, 0x98134C13138B265FL,
		0x7D2CB02C2C51589CL, 0xD6D36BD3D305BBB8L,
		0x6BE7BBE7E78CD35CL, 0x576EA56E6E39DCCBL,
		0x6EC437C4C4AA95F3L, 0x18030C03031B060FL,
		0x8A56455656DCAC13L, 0x1A440D44445E8849L,
		0xDF7FE17F7FA0FE9EL, 0x21A99EA9A9884F37L,
		0x4D2AA82A2A675482L, 0xB1BBD6BBBB0A6B6DL,
		0x46C123C1C1879FE2L, 0xA253515353F1A602L,
		0xAEDC57DCDC72A58BL, 0x580B2C0B0B531627L,
		0x9C9D4E9D9D0127D3L, 0x476CAD6C6C2BD8C1L,
		0x9531C43131A462F5L, 0x8774CD7474F3E8B9L,
		0xE3F6FFF6F615F109L, 0x0A460546464C8C43L,
		0x09AC8AACACA54526L, 0x3C891E8989B50F97L,
		0xA014501414B42844L, 0x5BE1A3E1E1BADF42L,
		0xB016581616A62C4EL, 0xCD3AE83A3AF774D2L,
		0x6F69B9696906D2D0L, 0x480924090941122DL,
		0xA770DD7070D7E0ADL, 0xD9B6E2B6B66F7154L,
		0xCED067D0D01EBDB7L, 0x3BED93EDEDD6C77EL,
		0x2ECC17CCCCE285DBL, 0x2A42154242688457L,
		0xB4985A98982C2DC2L, 0x49A4AAA4A4ED550EL,
		0x5D28A02828755088L, 0xDA5C6D5C5C86B831L,
		0x93F8C7F8F86BED3FL, 0x4486228686C211A4L
	};

	private static final long[] T4 = {
		0x18601818D83078C0L, 0x238C23232646AF05L,
		0xC63FC6C6B891F97EL, 0xE887E8E8FBCD6F13L,
		0x87268787CB13A14CL, 0xB8DAB8B8116D62A9L,
		0x0104010109020508L, 0x4F214F4F0D9E6E42L,
		0x36D836369B6CEEADL, 0xA6A2A6A6FF510459L,
		0xD26FD2D20CB9BDDEL, 0xF5F3F5F50EF706FBL,
		0x79F9797996F280EFL, 0x6FA16F6F30DECE5FL,
		0x917E91916D3FEFFCL, 0x52555252F8A407AAL,
		0x609D606047C0FD27L, 0xBCCABCBC35657689L,
		0x9B569B9B372BCDACL, 0x8E028E8E8A018C04L,
		0xA3B6A3A3D25B1571L, 0x0C300C0C6C183C60L,
		0x7BF17B7B84F68AFFL, 0x35D43535806AE1B5L,
		0x1D741D1DF53A69E8L, 0xE0A7E0E0B3DD4753L,
		0xD77BD7D721B3ACF6L, 0xC22FC2C29C99ED5EL,
		0x2EB82E2E435C966DL, 0x4B314B4B29967A62L,
		0xFEDFFEFE5DE121A3L, 0x57415757D5AE1682L,
		0x15541515BD2A41A8L, 0x77C17777E8EEB69FL,
		0x37DC3737926EEBA5L, 0xE5B3E5E59ED7567BL,
		0x9F469F9F1323D98CL, 0xF0E7F0F023FD17D3L,
		0x4A354A4A20947F6AL, 0xDA4FDADA44A9959EL,
		0x587D5858A2B025FAL, 0xC903C9C9CF8FCA06L,
		0x29A429297C528D55L, 0x0A280A0A5A142250L,
		0xB1FEB1B1507F4FE1L, 0xA0BAA0A0C95D1A69L,
		0x6BB16B6B14D6DA7FL, 0x852E8585D917AB5CL,
		0xBDCEBDBD3C677381L, 0x5D695D5D8FBA34D2L,
		0x1040101090205080L, 0xF4F7F4F407F503F3L,
		0xCB0BCBCBDD8BC016L, 0x3EF83E3ED37CC6EDL,
		0x051405052D0A1128L, 0x6781676778CEE61FL,
		0xE4B7E4E497D55373L, 0x279C2727024EBB25L,
		0x4119414173825832L, 0x8B168B8BA70B9D2CL,
		0xA7A6A7A7F6530151L, 0x7DE97D7DB2FA94CFL,
		0x956E95954937FBDCL, 0xD847D8D856AD9F8EL,
		0xFBCBFBFB70EB308BL, 0xEE9FEEEECDC17123L,
		0x7CED7C7CBBF891C7L, 0x6685666671CCE317L,
		0xDD53DDDD7BA78EA6L, 0x175C1717AF2E4BB8L,
		0x47014747458E4602L, 0x9E429E9E1A21DC84L,
		0xCA0FCACAD489C51EL, 0x2DB42D2D585A9975L,
		0xBFC6BFBF2E637991L, 0x071C07073F0E1B38L,
		0xAD8EADADAC472301L, 0x5A755A5AB0B42FEAL,
		0x83368383EF1BB56CL, 0x33CC3333B666FF85L,
		0x639163635CC6F23FL, 0x0208020212040A10L,
		0xAA92AAAA93493839L, 0x71D97171DEE2A8AFL,
		0xC807C8C8C68DCF0EL, 0x19641919D1327DC8L,
		0x493949493B927072L, 0xD943D9D95FAF9A86L,
		0xF2EFF2F231F91DC3L, 0xE3ABE3E3A8DB484BL,
		0x5B715B5BB9B62AE2L, 0x881A8888BC0D9234L,
		0x9A529A9A3E29C8A4L, 0x269826260B4CBE2DL,
		0x32C83232BF64FA8DL, 0xB0FAB0B0597D4AE9L,
		0xE983E9E9F2CF6A1BL, 0x0F3C0F0F771E3378L,
		0xD573D5D533B7A6E6L, 0x803A8080F41DBA74L,
		0xBEC2BEBE27617C99L, 0xCD13CDCDEB87DE26L,
		0x34D034348968E4BDL, 0x483D48483290757AL,
		0xFFDBFFFF54E324ABL, 0x7AF57A7A8DF48FF7L,
		0x907A9090643DEAF4L, 0x5F615F5F9DBE3EC2L,
		0x208020203D40A01DL, 0x68BD68680FD0D567L,
		0x1A681A1ACA3472D0L, 0xAE82AEAEB7412C19L,
		0xB4EAB4B47D755EC9L, 0x544D5454CEA8199AL,
		0x937693937F3BE5ECL, 0x228822222F44AA0DL,
		0x648D646463C8E907L, 0xF1E3F1F12AFF12DBL,
		0x73D17373CCE6A2BFL, 0x1248121282245A90L,
		0x401D40407A805D3AL, 0x0820080848102840L,
		0xC32BC3C3959BE856L, 0xEC97ECECDFC57B33L,
		0xDB4BDBDB4DAB9096L, 0xA1BEA1A1C05F1F61L,
		0x8D0E8D8D9107831CL, 0x3DF43D3DC87AC9F5L,
		0x976697975B33F1CCL, 0x0000000000000000L,
		0xCF1BCFCFF983D436L, 0x2BAC2B2B6E568745L,
		0x76C57676E1ECB397L, 0x82328282E619B064L,
		0xD67FD6D628B1A9FEL, 0x1B6C1B1BC33677D8L,
		0xB5EEB5B574775BC1L, 0xAF86AFAFBE432911L,
		0x6AB56A6A1DD4DF77L, 0x505D5050EAA00DBAL,
		0x45094545578A4C12L, 0xF3EBF3F338FB18CBL,
		0x30C03030AD60F09DL, 0xEF9BEFEFC4C3742BL,
		0x3FFC3F3FDA7EC3E5L, 0x55495555C7AA1C92L,
		0xA2B2A2A2DB591079L, 0xEA8FEAEAE9C96503L,
		0x658965656ACAEC0FL, 0xBAD2BABA036968B9L,
		0x2FBC2F2F4A5E9365L, 0xC027C0C08E9DE74EL,
		0xDE5FDEDE60A181BEL, 0x1C701C1CFC386CE0L,
		0xFDD3FDFD46E72EBBL, 0x4D294D4D1F9A6452L,
		0x927292927639E0E4L, 0x75C97575FAEABC8FL,
		0x06180606360C1E30L, 0x8A128A8AAE099824L,
		0xB2F2B2B24B7940F9L, 0xE6BFE6E685D15963L,
		0x0E380E0E7E1C3670L, 0x1F7C1F1FE73E63F8L,
		0x6295626255C4F737L, 0xD477D4D43AB5A3EEL,
		0xA89AA8A8814D3229L, 0x966296965231F4C4L,
		0xF9C3F9F962EF3A9BL, 0xC533C5C5A397F666L,
		0x25942525104AB135L, 0x59795959ABB220F2L,
		0x842A8484D015AE54L, 0x72D57272C5E4A7B7L,
		0x39E43939EC72DDD5L, 0x4C2D4C4C1698615AL,
		0x5E655E5E94BC3BCAL, 0x78FD78789FF085E7L,
		0x38E03838E570D8DDL, 0x8C0A8C8C98058614L,
		0xD163D1D117BFB2C6L, 0xA5AEA5A5E4570B41L,
		0xE2AFE2E2A1D94D43L, 0x619961614EC2F82FL,
		0xB3F6B3B3427B45F1L, 0x218421213442A515L,
		0x9C4A9C9C0825D694L, 0x1E781E1EEE3C66F0L,
		0x4311434361865222L, 0xC73BC7C7B193FC76L,
		0xFCD7FCFC4FE52BB3L, 0x0410040424081420L,
		0x51595151E3A208B2L, 0x995E9999252FC7BCL,
		0x6DA96D6D22DAC44FL, 0x0D340D0D651A3968L,
		0xFACFFAFA79E93583L, 0xDF5BDFDF69A384B6L,
		0x7EE57E7EA9FC9BD7L, 0x249024241948B43DL,
		0x3BEC3B3BFE76D7C5L, 0xAB96ABAB9A4B3D31L,
		0xCE1FCECEF081D13EL, 0x1144111199225588L,
		0x8F068F8F8303890CL, 0x4E254E4E049C6B4AL,
		0xB7E6B7B7667351D1L, 0xEB8BEBEBE0CB600BL,
		0x3CF03C3CC178CCFDL, 0x813E8181FD1FBF7CL,
		0x946A94944035FED4L, 0xF7FBF7F71CF30CEBL,
		0xB9DEB9B9186F67A1L, 0x134C13138B265F98L,
		0x2CB02C2C51589C7DL, 0xD36BD3D305BBB8D6L,
		0xE7BBE7E78CD35C6BL, 0x6EA56E6E39DCCB57L,
		0xC437C4C4AA95F36EL, 0x030C03031B060F18L,
		0x56455656DCAC138AL, 0x440D44445E88491AL,
		0x7FE17F7FA0FE9EDFL, 0xA99EA9A9884F3721L,
		0x2AA82A2A6754824DL, 0xBBD6BBBB0A6B6DB1L,
		0xC123C1C1879FE246L, 0x53515353F1A602A2L,
		0xDC57DCDC72A58BAEL, 0x0B2C0B0B53162758L,
		0x9D4E9D9D0127D39CL, 0x6CAD6C6C2BD8C147L,
		0x31C43131A462F595L, 0x74CD7474F3E8B987L,
		0xF6FFF6F615F109E3L, 0x460546464C8C430AL,
		0xAC8AACACA5452609L, 0x891E8989B50F973CL,
		0x14501414B42844A0L, 0xE1A3E1E1BADF425BL,
		0x16581616A62C4EB0L, 0x3AE83A3AF774D2CDL,
		0x69B9696906D2D06FL, 0x0924090941122D48L,
		0x70DD7070D7E0ADA7L, 0xB6E2B6B66F7154D9L,
		0xD067D0D01EBDB7CEL, 0xED93EDEDD6C77E3BL,
		0xCC17CCCCE285DB2EL, 0x421542426884572AL,
		0x985A98982C2DC2B4L, 0xA4AAA4A4ED550E49L,
		0x28A028287550885DL, 0x5C6D5C5C86B831DAL,
		0xF8C7F8F86BED3F93L, 0x86228686C211A444L
	};

	private static final long[] T5 = {
		0x601818D83078C018L, 0x8C23232646AF0523L,
		0x3FC6C6B891F97EC6L, 0x87E8E8FBCD6F13E8L,
		0x268787CB13A14C87L, 0xDAB8B8116D62A9B8L,
		0x0401010902050801L, 0x214F4F0D9E6E424FL,
		0xD836369B6CEEAD36L, 0xA2A6A6FF510459A6L,
		0x6FD2D20CB9BDDED2L, 0xF3F5F50EF706FBF5L,
		0xF9797996F280EF79L, 0xA16F6F30DECE5F6FL,
		0x7E91916D3FEFFC91L, 0x555252F8A407AA52L,
		0x9D606047C0FD2760L, 0xCABCBC35657689BCL,
		0x569B9B372BCDAC9BL, 0x028E8E8A018C048EL,
		0xB6A3A3D25B1571A3L, 0x300C0C6C183C600CL,
		0xF17B7B84F68AFF7BL, 0xD43535806AE1B535L,
		0x741D1DF53A69E81DL, 0xA7E0E0B3DD4753E0L,
		0x7BD7D721B3ACF6D7L, 0x2FC2C29C99ED5EC2L,
		0xB82E2E435C966D2EL, 0x314B4B29967A624BL,
		0xDFFEFE5DE121A3FEL, 0x415757D5AE168257L,
		0x541515BD2A41A815L, 0xC17777E8EEB69F77L,
		0xDC3737926EEBA537L, 0xB3E5E59ED7567BE5L,
		0x469F9F1323D98C9FL, 0xE7F0F023FD17D3F0L,
		0x354A4A20947F6A4AL, 0x4FDADA44A9959EDAL,
		0x7D5858A2B025FA58L, 0x03C9C9CF8FCA06C9L,
		0xA429297C528D5529L, 0x280A0A5A1422500AL,
		0xFEB1B1507F4FE1B1L, 0xBAA0A0C95D1A69A0L,
		0xB16B6B14D6DA7F6BL, 0x2E8585D917AB5C85L,
		0xCEBDBD3C677381BDL, 0x695D5D8FBA34D25DL,
		0x4010109020508010L, 0xF7F4F407F503F3F4L,
		0x0BCBCBDD8BC016CBL, 0xF83E3ED37CC6ED3EL,
		0x1405052D0A112805L, 0x81676778CEE61F67L,
		0xB7E4E497D55373E4L, 0x9C2727024EBB2527L,
		0x1941417382583241L, 0x168B8BA70B9D2C8BL,
		0xA6A7A7F6530151A7L, 0xE97D7DB2FA94CF7DL,
		0x6E95954937FBDC95L, 0x47D8D856AD9F8ED8L,
		0xCBFBFB70EB308BFBL, 0x9FEEEECDC17123EEL,
		0xED7C7CBBF891C77CL, 0x85666671CCE31766L,
		0x53DDDD7BA78EA6DDL, 0x5C1717AF2E4BB817L,
		0x014747458E460247L, 0x429E9E1A21DC849EL,
		0x0FCACAD489C51ECAL, 0xB42D2D585A99752DL,
		0xC6BFBF2E637991BFL, 0x1C07073F0E1B3807L,
		0x8EADADAC472301ADL, 0x755A5AB0B42FEA5AL,
		0x368383EF1BB56C83L, 0xCC3333B666FF8533L,
		0x9163635CC6F23F63L, 0x08020212040A1002L,
		0x92AAAA93493839AAL, 0xD97171DEE2A8AF71L,
		0x07C8C8C68DCF0EC8L, 0x641919D1327DC819L,
		0x3949493B92707249L, 0x43D9D95FAF9A86D9L,
		0xEFF2F231F91DC3F2L, 0xABE3E3A8DB484BE3L,
		0x715B5BB9B62AE25BL, 0x1A8888BC0D923488L,
		0x529A9A3E29C8A49AL, 0x9826260B4CBE2D26L,
		0xC83232BF64FA8D32L, 0xFAB0B0597D4AE9B0L,
		0x83E9E9F2CF6A1BE9L, 0x3C0F0F771E33780FL,
		0x73D5D533B7A6E6D5L, 0x3A8080F41DBA7480L,
		0xC2BEBE27617C99BEL, 0x13CDCDEB87DE26CDL,
		0xD034348968E4BD34L, 0x3D48483290757A48L,
		0xDBFFFF54E324ABFFL, 0xF57A7A8DF48FF77AL,
		0x7A9090643DEAF490L, 0x615F5F9DBE3EC25FL,
		0x8020203D40A01D20L, 0xBD68680FD0D56768L,
		0x681A1ACA3472D01AL, 0x82AEAEB7412C19AEL,
		0xEAB4B47D755EC9B4L, 0x4D5454CEA8199A54L,
		0x7693937F3BE5EC93L, 0x8822222F44AA0D22L,
		0x8D646463C8E90764L, 0xE3F1F12AFF12DBF1L,
		0xD17373CCE6A2BF73L, 0x48121282245A9012L,
		0x1D40407A805D3A40L, 0x2008084810284008L,
		0x2BC3C3959BE856C3L, 0x97ECECDFC57B33ECL,
		0x4BDBDB4DAB9096DBL, 0xBEA1A1C05F1F61A1L,
		0x0E8D8D9107831C8DL, 0xF43D3DC87AC9F53DL,
		0x6697975B33F1CC97L, 0x0000000000000000L,
		0x1BCFCFF983D436CFL, 0xAC2B2B6E5687452BL,
		0xC57676E1ECB39776L, 0x328282E619B06482L,
		0x7FD6D628B1A9FED6L, 0x6C1B1BC33677D81BL,
		0xEEB5B574775BC1B5L, 0x86AFAFBE432911AFL,
		0xB56A6A1DD4DF776AL, 0x5D5050EAA00DBA50L,
		0x094545578A4C1245L, 0xEBF3F338FB18CBF3L,
		0xC03030AD60F09D30L, 0x9BEFEFC4C3742BEFL,
		0xFC3F3FDA7EC3E53FL, 0x495555C7AA1C9255L,
		0xB2A2A2DB591079A2L, 0x8FEAEAE9C96503EAL,
		0x8965656ACAEC0F65L, 0xD2BABA036968B9BAL,
		0xBC2F2F4A5E93652FL, 0x27C0C08E9DE74EC0L,
		0x5FDEDE60A181BEDEL, 0x701C1CFC386CE01CL,
		0xD3FDFD46E72EBBFDL, 0x294D4D1F9A64524DL,
		0x7292927639E0E492L, 0xC97575FAEABC8F75L,
		0x180606360C1E3006L, 0x128A8AAE0998248AL,
		0xF2B2B24B7940F9B2L, 0xBFE6E685D15963E6L,
		0x380E0E7E1C36700EL, 0x7C1F1FE73E63F81FL,
		0x95626255C4F73762L, 0x77D4D43AB5A3EED4L,
		0x9AA8A8814D3229A8L, 0x6296965231F4C496L,
		0xC3F9F962EF3A9BF9L, 0x33C5C5A397F666C5L,
		0x942525104AB13525L, 0x795959ABB220F259L,
		0x2A8484D015AE5484L, 0xD57272C5E4A7B772L,
		0xE43939EC72DDD539L, 0x2D4C4C1698615A4CL,
		0x655E5E94BC3BCA5EL, 0xFD78789FF085E778L,
		0xE03838E570D8DD38L, 0x0A8C8C980586148CL,
		0x63D1D117BFB2C6D1L, 0xAEA5A5E4570B41A5L,
		0xAFE2E2A1D94D43E2L, 0x9961614EC2F82F61L,
		0xF6B3B3427B45F1B3L, 0x8421213442A51521L,
		0x4A9C9C0825D6949CL, 0x781E1EEE3C66F01EL,
		0x1143436186522243L, 0x3BC7C7B193FC76C7L,
		0xD7FCFC4FE52BB3FCL, 0x1004042408142004L,
		0x595151E3A208B251L, 0x5E9999252FC7BC99L,
		0xA96D6D22DAC44F6DL, 0x340D0D651A39680DL,
		0xCFFAFA79E93583FAL, 0x5BDFDF69A384B6DFL,
		0xE57E7EA9FC9BD77EL, 0x9024241948B43D24L,
		0xEC3B3BFE76D7C53BL, 0x96ABAB9A4B3D31ABL,
		0x1FCECEF081D13ECEL, 0x4411119922558811L,
		0x068F8F8303890C8FL, 0x254E4E049C6B4A4EL,
		0xE6B7B7667351D1B7L, 0x8BEBEBE0CB600BEBL,
		0xF03C3CC178CCFD3CL, 0x3E8181FD1FBF7C81L,
		0x6A94944035FED494L, 0xFBF7F71CF30CEBF7L,
		0xDEB9B9186F67A1B9L, 0x4C13138B265F9813L,
		0xB02C2C51589C7D2CL, 0x6BD3D305BBB8D6D3L,
		0xBBE7E78CD35C6BE7L, 0xA56E6E39DCCB576EL,
		0x37C4C4AA95F36EC4L, 0x0C03031B060F1803L,
		0x455656DCAC138A56L, 0x0D44445E88491A44L,
		0xE17F7FA0FE9EDF7FL, 0x9EA9A9884F3721A9L,
		0xA82A2A6754824D2AL, 0xD6BBBB0A6B6DB1BBL,
		0x23C1C1879FE246C1L, 0x515353F1A602A253L,
		0x57DCDC72A58BAEDCL, 0x2C0B0B531627580BL,
		0x4E9D9D0127D39C9DL, 0xAD6C6C2BD8C1476CL,
		0xC43131A462F59531L, 0xCD7474F3E8B98774L,
		0xFFF6F615F109E3F6L, 0x0546464C8C430A46L,
		0x8AACACA5452609ACL, 0x1E8989B50F973C89L,
		0x501414B42844A014L, 0xA3E1E1BADF425BE1L,
		0x581616A62C4EB016L, 0xE83A3AF774D2CD3AL,
		0xB9696906D2D06F69L, 0x24090941122D4809L,
		0xDD7070D7E0ADA770L, 0xE2B6B66F7154D9B6L,
		0x67D0D01EBDB7CED0L, 0x93EDEDD6C77E3BEDL,
		0x17CCCCE285DB2ECCL, 0x1542426884572A42L,
		0x5A98982C2DC2B498L, 0xAAA4A4ED550E49A4L,
		0xA028287550885D28L, 0x6D5C5C86B831DA5CL,
		0xC7F8F86BED3F93F8L, 0x228686C211A44486L
	};

	private static final long[] T6 = {
		0x1818D83078C01860L, 0x23232646AF05238CL,
		0xC6C6B891F97EC63FL, 0xE8E8FBCD6F13E887L,
		0x8787CB13A14C8726L, 0xB8B8116D62A9B8DAL,
		0x0101090205080104L, 0x4F4F0D9E6E424F21L,
		0x36369B6CEEAD36D8L, 0xA6A6FF510459A6A2L,
		0xD2D20CB9BDDED26FL, 0xF5F50EF706FBF5F3L,
		0x797996F280EF79F9L, 0x6F6F30DECE5F6FA1L,
		0x91916D3FEFFC917EL, 0x5252F8A407AA5255L,
		0x606047C0FD27609DL, 0xBCBC35657689BCCAL,
		0x9B9B372BCDAC9B56L, 0x8E8E8A018C048E02L,
		0xA3A3D25B1571A3B6L, 0x0C0C6C183C600C30L,
		0x7B7B84F68AFF7BF1L, 0x3535806AE1B535D4L,
		0x1D1DF53A69E81D74L, 0xE0E0B3DD4753E0A7L,
		0xD7D721B3ACF6D77BL, 0xC2C29C99ED5EC22FL,
		0x2E2E435C966D2EB8L, 0x4B4B29967A624B31L,
		0xFEFE5DE121A3FEDFL, 0x5757D5AE16825741L,
		0x1515BD2A41A81554L, 0x7777E8EEB69F77C1L,
		0x3737926EEBA537DCL, 0xE5E59ED7567BE5B3L,
		0x9F9F1323D98C9F46L, 0xF0F023FD17D3F0E7L,
		0x4A4A20947F6A4A35L, 0xDADA44A9959EDA4FL,
		0x5858A2B025FA587DL, 0xC9C9CF8FCA06C903L,
		0x29297C528D5529A4L, 0x0A0A5A1422500A28L,
		0xB1B1507F4FE1B1FEL, 0xA0A0C95D1A69A0BAL,
		0x6B6B14D6DA7F6BB1L, 0x8585D917AB5C852EL,
		0xBDBD3C677381BDCEL, 0x5D5D8FBA34D25D69L,
		0x1010902050801040L, 0xF4F407F503F3F4F7L,
		0xCBCBDD8BC016CB0BL, 0x3E3ED37CC6ED3EF8L,
		0x05052D0A11280514L, 0x676778CEE61F6781L,
		0xE4E497D55373E4B7L, 0x2727024EBB25279CL,
		0x4141738258324119L, 0x8B8BA70B9D2C8B16L,
		0xA7A7F6530151A7A6L, 0x7D7DB2FA94CF7DE9L,
		0x95954937FBDC956EL, 0xD8D856AD9F8ED847L,
		0xFBFB70EB308BFBCBL, 0xEEEECDC17123EE9FL,
		0x7C7CBBF891C77CEDL, 0x666671CCE3176685L,
		0xDDDD7BA78EA6DD53L, 0x1717AF2E4BB8175CL,
		0x4747458E46024701L, 0x9E9E1A21DC849E42L,
		0xCACAD489C51ECA0FL, 0x2D2D585A99752DB4L,
		0xBFBF2E637991BFC6L, 0x07073F0E1B38071CL,
		0xADADAC472301AD8EL, 0x5A5AB0B42FEA5A75L,
		0x8383EF1BB56C8336L, 0x3333B666FF8533CCL,
		0x63635CC6F23F6391L, 0x020212040A100208L,
		0xAAAA93493839AA92L, 0x7171DEE2A8AF71D9L,
		0xC8C8C68DCF0EC807L, 0x1919D1327DC81964L,
		0x49493B9270724939L, 0xD9D95FAF9A86D943L,
		0xF2F231F91DC3F2EFL, 0xE3E3A8DB484BE3ABL,
		0x5B5BB9B62AE25B71L, 0x8888BC0D9234881AL,
		0x9A9A3E29C8A49A52L, 0x26260B4CBE2D2698L,
		0x3232BF64FA8D32C8L, 0xB0B0597D4AE9B0FAL,
		0xE9E9F2CF6A1BE983L, 0x0F0F771E33780F3CL,
		0xD5D533B7A6E6D573L, 0x8080F41DBA74803AL,
		0xBEBE27617C99BEC2L, 0xCDCDEB87DE26CD13L,
		0x34348968E4BD34D0L, 0x48483290757A483DL,
		0xFFFF54E324ABFFDBL, 0x7A7A8DF48FF77AF5L,
		0x9090643DEAF4907AL, 0x5F5F9DBE3EC25F61L,
		0x20203D40A01D2080L, 0x68680FD0D56768BDL,
		0x1A1ACA3472D01A68L, 0xAEAEB7412C19AE82L,
		0xB4B47D755EC9B4EAL, 0x5454CEA8199A544DL,
		0x93937F3BE5EC9376L, 0x22222F44AA0D2288L,
		0x646463C8E907648DL, 0xF1F12AFF12DBF1E3L,
		0x7373CCE6A2BF73D1L, 0x121282245A901248L,
		0x40407A805D3A401DL, 0x0808481028400820L,
		0xC3C3959BE856C32BL, 0xECECDFC57B33EC97L,
		0xDBDB4DAB9096DB4BL, 0xA1A1C05F1F61A1BEL,
		0x8D8D9107831C8D0EL, 0x3D3DC87AC9F53DF4L,
		0x97975B33F1CC9766L, 0x0000000000000000L,
		0xCFCFF983D436CF1BL, 0x2B2B6E5687452BACL,
		0x7676E1ECB39776C5L, 0x8282E619B0648232L,
		0xD6D628B1A9FED67FL, 0x1B1BC33677D81B6CL,
		0xB5B574775BC1B5EEL, 0xAFAFBE432911AF86L,
		0x6A6A1DD4DF776AB5L, 0x5050EAA00DBA505DL,
		0x4545578A4C124509L, 0xF3F338FB18CBF3EBL,
		0x3030AD60F09D30C0L, 0xEFEFC4C3742BEF9BL,
		0x3F3FDA7EC3E53FFCL, 0x5555C7AA1C925549L,
		0xA2A2DB591079A2B2L, 0xEAEAE9C96503EA8FL,
		0x65656ACAEC0F6589L, 0xBABA036968B9BAD2L,
		0x2F2F4A5E93652FBCL, 0xC0C08E9DE74EC027L,
		0xDEDE60A181BEDE5FL, 0x1C1CFC386CE01C70L,
		0xFDFD46E72EBBFDD3L, 0x4D4D1F9A64524D29L,
		0x92927639E0E49272L, 0x7575FAEABC8F75C9L,
		0x0606360C1E300618L, 0x8A8AAE0998248A12L,
		0xB2B24B7940F9B2F2L, 0xE6E685D15963E6BFL,
		0x0E0E7E1C36700E38L, 0x1F1FE73E63F81F7CL,
		0x626255C4F7376295L, 0xD4D43AB5A3EED477L,
		0xA8A8814D3229A89AL, 0x96965231F4C49662L,
		0xF9F962EF3A9BF9C3L, 0xC5C5A397F666C533L,
		0x2525104AB1352594L, 0x5959ABB220F25979L,
		0x8484D015AE54842AL, 0x7272C5E4A7B772D5L,
		0x3939EC72DDD539E4L, 0x4C4C1698615A4C2DL,
		0x5E5E94BC3BCA5E65L, 0x78789FF085E778FDL,
		0x3838E570D8DD38E0L, 0x8C8C980586148C0AL,
		0xD1D117BFB2C6D163L, 0xA5A5E4570B41A5AEL,
		0xE2E2A1D94D43E2AFL, 0x61614EC2F82F6199L,
		0xB3B3427B45F1B3F6L, 0x21213442A5152184L,
		0x9C9C0825D6949C4AL, 0x1E1EEE3C66F01E78L,
		0x4343618652224311L, 0xC7C7B193FC76C73BL,
		0xFCFC4FE52BB3FCD7L, 0x0404240814200410L,
		0x5151E3A208B25159L, 0x9999252FC7BC995EL,
		0x6D6D22DAC44F6DA9L, 0x0D0D651A39680D34L,
		0xFAFA79E93583FACFL, 0xDFDF69A384B6DF5BL,
		0x7E7EA9FC9BD77EE5L, 0x24241948B43D2490L,
		0x3B3BFE76D7C53BECL, 0xABAB9A4B3D31AB96L,
		0xCECEF081D13ECE1FL, 0x1111992255881144L,
		0x8F8F8303890C8F06L, 0x4E4E049C6B4A4E25L,
		0xB7B7667351D1B7E6L, 0xEBEBE0CB600BEB8BL,
		0x3C3CC178CCFD3CF0L, 0x8181FD1FBF7C813EL,
		0x94944035FED4946AL, 0xF7F71CF30CEBF7FBL,
		0xB9B9186F67A1B9DEL, 0x13138B265F98134CL,
		0x2C2C51589C7D2CB0L, 0xD3D305BBB8D6D36BL,
		0xE7E78CD35C6BE7BBL, 0x6E6E39DCCB576EA5L,
		0xC4C4AA95F36EC437L, 0x03031B060F18030CL,
		0x5656DCAC138A5645L, 0x44445E88491A440DL,
		0x7F7FA0FE9EDF7FE1L, 0xA9A9884F3721A99EL,
		0x2A2A6754824D2AA8L, 0xBBBB0A6B6DB1BBD6L,
		0xC1C1879FE246C123L, 0x5353F1A602A25351L,
		0xDCDC72A58BAEDC57L, 0x0B0B531627580B2CL,
		0x9D9D0127D39C9D4EL, 0x6C6C2BD8C1476CADL,
		0x3131A462F59531C4L, 0x7474F3E8B98774CDL,
		0xF6F615F109E3F6FFL, 0x46464C8C430A4605L,
		0xACACA5452609AC8AL, 0x8989B50F973C891EL,
		0x1414B42844A01450L, 0xE1E1BADF425BE1A3L,
		0x1616A62C4EB01658L, 0x3A3AF774D2CD3AE8L,
		0x696906D2D06F69B9L, 0x090941122D480924L,
		0x7070D7E0ADA770DDL, 0xB6B66F7154D9B6E2L,
		0xD0D01EBDB7CED067L, 0xEDEDD6C77E3BED93L,
		0xCCCCE285DB2ECC17L, 0x42426884572A4215L,
		0x98982C2DC2B4985AL, 0xA4A4ED550E49A4AAL,
		0x28287550885D28A0L, 0x5C5C86B831DA5C6DL,
		0xF8F86BED3F93F8C7L, 0x8686C211A4448622L
	};

	private static final long[] T7 = {
		0x18D83078C0186018L, 0x232646AF05238C23L,
		0xC6B891F97EC63FC6L, 0xE8FBCD6F13E887E8L,
		0x87CB13A14C872687L, 0xB8116D62A9B8DAB8L,
		0x0109020508010401L, 0x4F0D9E6E424F214FL,
		0x369B6CEEAD36D836L, 0xA6FF510459A6A2A6L,
		0xD20CB9BDDED26FD2L, 0xF50EF706FBF5F3F5L,
		0x7996F280EF79F979L, 0x6F30DECE5F6FA16FL,
		0x916D3FEFFC917E91L, 0x52F8A407AA525552L,
		0x6047C0FD27609D60L, 0xBC35657689BCCABCL,
		0x9B372BCDAC9B569BL, 0x8E8A018C048E028EL,
		0xA3D25B1571A3B6A3L, 0x0C6C183C600C300CL,
		0x7B84F68AFF7BF17BL, 0x35806AE1B535D435L,
		0x1DF53A69E81D741DL, 0xE0B3DD4753E0A7E0L,
		0xD721B3ACF6D77BD7L, 0xC29C99ED5EC22FC2L,
		0x2E435C966D2EB82EL, 0x4B29967A624B314BL,
		0xFE5DE121A3FEDFFEL, 0x57D5AE1682574157L,
		0x15BD2A41A8155415L, 0x77E8EEB69F77C177L,
		0x37926EEBA537DC37L, 0xE59ED7567BE5B3E5L,
		0x9F1323D98C9F469FL, 0xF023FD17D3F0E7F0L,
		0x4A20947F6A4A354AL, 0xDA44A9959EDA4FDAL,
		0x58A2B025FA587D58L, 0xC9CF8FCA06C903C9L,
		0x297C528D5529A429L, 0x0A5A1422500A280AL,
		0xB1507F4FE1B1FEB1L, 0xA0C95D1A69A0BAA0L,
		0x6B14D6DA7F6BB16BL, 0x85D917AB5C852E85L,
		0xBD3C677381BDCEBDL, 0x5D8FBA34D25D695DL,
		0x1090205080104010L, 0xF407F503F3F4F7F4L,
		0xCBDD8BC016CB0BCBL, 0x3ED37CC6ED3EF83EL,
		0x052D0A1128051405L, 0x6778CEE61F678167L,
		0xE497D55373E4B7E4L, 0x27024EBB25279C27L,
		0x4173825832411941L, 0x8BA70B9D2C8B168BL,
		0xA7F6530151A7A6A7L, 0x7DB2FA94CF7DE97DL,
		0x954937FBDC956E95L, 0xD856AD9F8ED847D8L,
		0xFB70EB308BFBCBFBL, 0xEECDC17123EE9FEEL,
		0x7CBBF891C77CED7CL, 0x6671CCE317668566L,
		0xDD7BA78EA6DD53DDL, 0x17AF2E4BB8175C17L,
		0x47458E4602470147L, 0x9E1A21DC849E429EL,
		0xCAD489C51ECA0FCAL, 0x2D585A99752DB42DL,
		0xBF2E637991BFC6BFL, 0x073F0E1B38071C07L,
		0xADAC472301AD8EADL, 0x5AB0B42FEA5A755AL,
		0x83EF1BB56C833683L, 0x33B666FF8533CC33L,
		0x635CC6F23F639163L, 0x0212040A10020802L,
		0xAA93493839AA92AAL, 0x71DEE2A8AF71D971L,
		0xC8C68DCF0EC807C8L, 0x19D1327DC8196419L,
		0x493B927072493949L, 0xD95FAF9A86D943D9L,
		0xF231F91DC3F2EFF2L, 0xE3A8DB484BE3ABE3L,
		0x5BB9B62AE25B715BL, 0x88BC0D9234881A88L,
		0x9A3E29C8A49A529AL, 0x260B4CBE2D269826L,
		0x32BF64FA8D32C832L, 0xB0597D4AE9B0FAB0L,
		0xE9F2CF6A1BE983E9L, 0x0F771E33780F3C0FL,
		0xD533B7A6E6D573D5L, 0x80F41DBA74803A80L,
		0xBE27617C99BEC2BEL, 0xCDEB87DE26CD13CDL,
		0x348968E4BD34D034L, 0x483290757A483D48L,
		0xFF54E324ABFFDBFFL, 0x7A8DF48FF77AF57AL,
		0x90643DEAF4907A90L, 0x5F9DBE3EC25F615FL,
		0x203D40A01D208020L, 0x680FD0D56768BD68L,
		0x1ACA3472D01A681AL, 0xAEB7412C19AE82AEL,
		0xB47D755EC9B4EAB4L, 0x54CEA8199A544D54L,
		0x937F3BE5EC937693L, 0x222F44AA0D228822L,
		0x6463C8E907648D64L, 0xF12AFF12DBF1E3F1L,
		0x73CCE6A2BF73D173L, 0x1282245A90124812L,
		0x407A805D3A401D40L, 0x0848102840082008L,
		0xC3959BE856C32BC3L, 0xECDFC57B33EC97ECL,
		0xDB4DAB9096DB4BDBL, 0xA1C05F1F61A1BEA1L,
		0x8D9107831C8D0E8DL, 0x3DC87AC9F53DF43DL,
		0x975B33F1CC976697L, 0x0000000000000000L,
		0xCFF983D436CF1BCFL, 0x2B6E5687452BAC2BL,
		0x76E1ECB39776C576L, 0x82E619B064823282L,
		0xD628B1A9FED67FD6L, 0x1BC33677D81B6C1BL,
		0xB574775BC1B5EEB5L, 0xAFBE432911AF86AFL,
		0x6A1DD4DF776AB56AL, 0x50EAA00DBA505D50L,
		0x45578A4C12450945L, 0xF338FB18CBF3EBF3L,
		0x30AD60F09D30C030L, 0xEFC4C3742BEF9BEFL,
		0x3FDA7EC3E53FFC3FL, 0x55C7AA1C92554955L,
		0xA2DB591079A2B2A2L, 0xEAE9C96503EA8FEAL,
		0x656ACAEC0F658965L, 0xBA036968B9BAD2BAL,
		0x2F4A5E93652FBC2FL, 0xC08E9DE74EC027C0L,
		0xDE60A181BEDE5FDEL, 0x1CFC386CE01C701CL,
		0xFD46E72EBBFDD3FDL, 0x4D1F9A64524D294DL,
		0x927639E0E4927292L, 0x75FAEABC8F75C975L,
		0x06360C1E30061806L, 0x8AAE0998248A128AL,
		0xB24B7940F9B2F2B2L, 0xE685D15963E6BFE6L,
		0x0E7E1C36700E380EL, 0x1FE73E63F81F7C1FL,
		0x6255C4F737629562L, 0xD43AB5A3EED477D4L,
		0xA8814D3229A89AA8L, 0x965231F4C4966296L,
		0xF962EF3A9BF9C3F9L, 0xC5A397F666C533C5L,
		0x25104AB135259425L, 0x59ABB220F2597959L,
		0x84D015AE54842A84L, 0x72C5E4A7B772D572L,
		0x39EC72DDD539E439L, 0x4C1698615A4C2D4CL,
		0x5E94BC3BCA5E655EL, 0x789FF085E778FD78L,
		0x38E570D8DD38E038L, 0x8C980586148C0A8CL,
		0xD117BFB2C6D163D1L, 0xA5E4570B41A5AEA5L,
		0xE2A1D94D43E2AFE2L, 0x614EC2F82F619961L,
		0xB3427B45F1B3F6B3L, 0x213442A515218421L,
		0x9C0825D6949C4A9CL, 0x1EEE3C66F01E781EL,
		0x4361865222431143L, 0xC7B193FC76C73BC7L,
		0xFC4FE52BB3FCD7FCL, 0x0424081420041004L,
		0x51E3A208B2515951L, 0x99252FC7BC995E99L,
		0x6D22DAC44F6DA96DL, 0x0D651A39680D340DL,
		0xFA79E93583FACFFAL, 0xDF69A384B6DF5BDFL,
		0x7EA9FC9BD77EE57EL, 0x241948B43D249024L,
		0x3BFE76D7C53BEC3BL, 0xAB9A4B3D31AB96ABL,
		0xCEF081D13ECE1FCEL, 0x1199225588114411L,
		0x8F8303890C8F068FL, 0x4E049C6B4A4E254EL,
		0xB7667351D1B7E6B7L, 0xEBE0CB600BEB8BEBL,
		0x3CC178CCFD3CF03CL, 0x81FD1FBF7C813E81L,
		0x944035FED4946A94L, 0xF71CF30CEBF7FBF7L,
		0xB9186F67A1B9DEB9L, 0x138B265F98134C13L,
		0x2C51589C7D2CB02CL, 0xD305BBB8D6D36BD3L,
		0xE78CD35C6BE7BBE7L, 0x6E39DCCB576EA56EL,
		0xC4AA95F36EC437C4L, 0x031B060F18030C03L,
		0x56DCAC138A564556L, 0x445E88491A440D44L,
		0x7FA0FE9EDF7FE17FL, 0xA9884F3721A99EA9L,
		0x2A6754824D2AA82AL, 0xBB0A6B6DB1BBD6BBL,
		0xC1879FE246C123C1L, 0x53F1A602A2535153L,
		0xDC72A58BAEDC57DCL, 0x0B531627580B2C0BL,
		0x9D0127D39C9D4E9DL, 0x6C2BD8C1476CAD6CL,
		0x31A462F59531C431L, 0x74F3E8B98774CD74L,
		0xF615F109E3F6FFF6L, 0x464C8C430A460546L,
		0xACA5452609AC8AACL, 0x89B50F973C891E89L,
		0x14B42844A0145014L, 0xE1BADF425BE1A3E1L,
		0x16A62C4EB0165816L, 0x3AF774D2CD3AE83AL,
		0x6906D2D06F69B969L, 0x0941122D48092409L,
		0x70D7E0ADA770DD70L, 0xB66F7154D9B6E2B6L,
		0xD01EBDB7CED067D0L, 0xEDD6C77E3BED93EDL,
		0xCCE285DB2ECC17CCL, 0x426884572A421542L,
		0x982C2DC2B4985A98L, 0xA4ED550E49A4AAA4L,
		0x287550885D28A028L, 0x5C86B831DA5C6D5CL,
		0xF86BED3F93F8C7F8L, 0x86C211A444862286L
	};

	/*
	 * Round constants.
	 */
	private static final long[] RC = {
		0x4F01B887E8C62318L,
		0x52916F79F5D2A636L,
		0x357B0CA38E9BBC60L,
		0x57FE4B2EC2D7E01DL,
		0xDA4AF09FE5377715L,
		0x856BA0B10A29C958L,
		0x67053ECBF4105DBDL,
		0xD8957DA78B4127E4L,
		0x9E4717DD667CEEFBL,
		0x33835AAD07BF2DCAL
	};

	/** @see Digest */
	public String toString()
	{
		return "Whirlpool";
	}
}
