package de.wenzlaff.mathe;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;

/**
 * Berechne den Erwartungswert die Varinaz und die Standardabweichung bei diesen
 * Werten 10 Euro mit 0,25 und -5 Euro mit 0,75 mit kleiner GUI.
 * 
 * @author Thomas Wenzlaff
 */
public class Standardabweichung extends JFrame {

	private static final long serialVersionUID = 1L;

	private JTextField[] valueFields = new JTextField[2];
	private JTextField[] probabilityFields = new JTextField[2];
	private double[] values = { 10, -5 };
	private double[] probabilities = { 0.25, 0.75 };
	private JPanel dataPanel;

	public Standardabweichung() {
		setTitle("Wahrscheinlichkeitsverteilung");
		setSize(800, 600);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setLayout(new BorderLayout());

		// Eingabebereich
		JPanel inputPanel = createInputPanel();
		add(inputPanel, BorderLayout.NORTH);

		// Datenvisualisierung
		dataPanel = new DataPanel();
		add(dataPanel, BorderLayout.CENTER);
	}

	private JPanel createInputPanel() {
		JPanel inputPanel = new JPanel();
		inputPanel.setLayout(new BoxLayout(inputPanel, BoxLayout.Y_AXIS));

		// Eingabefelder für erste Wert-Wahrscheinlichkeit Kombination
		inputPanel.add(createLabeledField("Wert 1:", valueFields[0] = new JTextField(String.valueOf(values[0]))));
		inputPanel.add(createLabeledField("Wahrscheinlichkeit 1:", probabilityFields[0] = new JTextField(String.valueOf(probabilities[0]))));

		// Eingabefelder für zweite Wert-Wahrscheinlichkeit Kombination
		inputPanel.add(createLabeledField("Wert 2:", valueFields[1] = new JTextField(String.valueOf(values[1]))));
		inputPanel.add(createLabeledField("Wahrscheinlichkeit 2:", probabilityFields[1] = new JTextField(String.valueOf(probabilities[1]))));

		// Berechnungsbutton
		JButton calculateButton = new JButton("Berechnen");
		calculateButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				updateValues();
			}
		});
		inputPanel.add(calculateButton);

		return inputPanel;
	}

	private JPanel createLabeledField(String labelText, JTextField textField) {
		JPanel panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		panel.add(new JLabel(labelText));
		panel.add(textField);
		textField.setColumns(10);
		return panel;
	}

	private void updateValues() {
		try {
			// Werte aus Eingabefeldern auslesen
			values[0] = Double.parseDouble(valueFields[0].getText());
			values[1] = Double.parseDouble(valueFields[1].getText());
			probabilities[0] = Double.parseDouble(probabilityFields[0].getText());
			probabilities[1] = Double.parseDouble(probabilityFields[1].getText());

			// Neuzeichnen des Panels
			dataPanel.repaint();
		} catch (NumberFormatException ex) {
			JOptionPane.showMessageDialog(this, "Bitte gültige Zahlen eingeben!");
		}
	}

	private class DataPanel extends JPanel {
		@Override
		protected void paintComponent(Graphics g) {
			super.paintComponent(g);
			Graphics2D g2d = (Graphics2D) g;

			// Hintergrund
			g2d.setColor(Color.WHITE);
			g2d.fillRect(0, 0, getWidth(), getHeight());

			// Balkendiagramm
			int startX = 100;
			int startY = 300;
			int barWidth = 100;
			int maxHeight = 200;

			for (int i = 0; i < values.length; i++) {
				double scaledHeight = Math.abs(values[i]) * (maxHeight / 10.0);

				g2d.setColor(values[i] >= 0 ? Color.GREEN : Color.RED);
				g2d.fillRect(startX + i * 150, startY - (int) scaledHeight, barWidth, (int) scaledHeight);

				g2d.setColor(Color.BLACK);
				g2d.drawString(String.format("Wert %d", i + 1), startX + i * 150, startY + 30);
				g2d.drawString(String.format("%.2f€ (%.2f%%)", values[i], probabilities[i] * 100), startX + i * 150, startY + 50);
			}

			// Statistische Berechnungen
			double expectation = calculateExpectation();
			double variance = calculateVariance(expectation);
			double stdDeviation = Math.sqrt(variance);

			g2d.drawString(String.format("Erwartungswert: %.2f€", expectation), 50, 50);
			g2d.drawString(String.format("Varianz: %.2f", variance), 50, 70);
			g2d.drawString(String.format("Standardabweichung: %.2f€", stdDeviation), 50, 90);
		}

		private double calculateExpectation() {
			double sum = 0;
			for (int i = 0; i < values.length; i++) {
				sum += values[i] * probabilities[i];
			}
			return sum;
		}

		private double calculateVariance(double expectation) {
			double variance = 0;
			for (int i = 0; i < values.length; i++) {
				variance += Math.pow(values[i] - expectation, 2) * probabilities[i];
			}
			return variance;
		}
	}

	public static void main(String[] args) {
		SwingUtilities.invokeLater(() -> {
			new Standardabweichung().setVisible(true);
		});
	}
}
