package de.wenzlaff.mathe;

import java.math.BigInteger;

/**
 * GoldbachsVermutung.
 * 
 * Starke Goldbachsche Vermutung: Jede gerade Zahl größer als 2 lässt sich als
 * Summe von zwei Primzahlen darstellen. Dies ist die berühmtere und unbewiesene
 * Form der Vermutung.
 * 
 * Beispiele:
 * 
 * 4 = 2 + 2
 * 
 * 6 = 3 + 3
 * 
 * 8 = 3 + 5
 * 
 * 10 = 5 + 5 oder 3 + 7
 * 
 * @author Thomas Wenzlaff
 */
public class GoldbachsVermutung {

	public static void main(String[] args) {
		if (args.length != 2) {
			System.out.println("Bitte zwei Parameter eingeben: eine obere Grenze und eine Startzahl.");
			return;
		}

		try {
			BigInteger obergrenze = new BigInteger(args[0]);
			BigInteger zwei = BigInteger.valueOf(2);
			BigInteger startzahl = new BigInteger(args[1]);

			if (obergrenze.compareTo(startzahl) >= 0 && startzahl.compareTo(BigInteger.valueOf(2)) >= 0 && startzahl.mod(zwei).equals(BigInteger.ZERO)) {
				for (BigInteger zahl = startzahl; zahl.compareTo(obergrenze) <= 0; zahl = zahl.add(zwei)) {
					boolean ergebnis = pruefeGoldbach(zahl);
					if (!ergebnis) {
						String text = "Die Zahl " + zahl + " lässt sich NICHT als Summe zweier Primzahlen darstellen.";
						System.err.println(text);
						throw new IllegalArgumentException(text);
					}
				}
			} else {
				System.out.println("Bitte eine gültige obere Grenze und eine gerade Startzahl größer als 2 eingeben.");
			}
		} catch (NumberFormatException e) {
			System.out.println("Bitte zwei Parameter eingeben: eine obere Grenze und eine Startzahl.");
		}
	}

	// Überprüft, ob eine Zahl als Summe zweier Primzahlen darstellbar ist
	public static boolean pruefeGoldbach(BigInteger zahl) {
		BigInteger zwei = BigInteger.valueOf(2);
		for (BigInteger i = zwei; i.compareTo(zahl.divide(zwei)) <= 0; i = i.add(BigInteger.ONE)) {
			if (istPrimzahl(i) && istPrimzahl(zahl.subtract(i))) {
				System.out.println(zahl + " = " + i + " + " + zahl.subtract(i));
				return true;
			}
		}
		return false;
	}

	// Überprüft, ob eine Zahl eine Primzahl ist
	public static boolean istPrimzahl(BigInteger zahl) {
		if (zahl.compareTo(BigInteger.ONE) <= 0)
			return false;
		BigInteger sqrtZahl = zahl.sqrt();
		for (BigInteger i = BigInteger.valueOf(2); i.compareTo(sqrtZahl) <= 0; i = i.add(BigInteger.ONE)) {
			if (zahl.mod(i).equals(BigInteger.ZERO)) {
				return false;
			}
		}
		return true;
	}
}
