/*
 * Decompiled with CFR 0.152.
 */
package de.westnordost.countryboundaries;

import de.westnordost.countryboundaries.CountryBoundariesCell;
import de.westnordost.countryboundaries.CountryBoundariesDeserializer;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class CountryBoundaries {
    final CountryBoundariesCell[] raster;
    final int rasterWidth;
    final Map<String, Double> geometrySizes;

    CountryBoundaries(CountryBoundariesCell[] raster, int rasterWidth, Map<String, Double> geometrySizes) {
        this.raster = raster;
        this.rasterWidth = rasterWidth;
        this.geometrySizes = geometrySizes;
    }

    public static CountryBoundaries load(InputStream is) throws IOException {
        return new CountryBoundariesDeserializer().read(new DataInputStream(is));
    }

    public boolean isInAny(double longitude, double latitude, Collection<String> ids) {
        this.validatePosition(longitude, latitude);
        longitude = CountryBoundaries.normalize(longitude, -180.0, 360.0);
        int cellX = this.longitudeToCellX(longitude);
        int cellY = this.latitudeToCellY(latitude);
        int localX = this.longitudeToLocalX(cellX, longitude);
        int localY = this.latitudeToLocalY(cellY, latitude);
        return this.getCell(cellX, cellY).isInAny(localX, localY, ids);
    }

    public boolean isIn(double longitude, double latitude, String id) {
        return this.isInAny(longitude, latitude, Collections.singleton(id));
    }

    public List<String> getIds(double longitude, double latitude) {
        this.validatePosition(longitude, latitude);
        longitude = CountryBoundaries.normalize(longitude, -180.0, 360.0);
        int cellX = this.longitudeToCellX(longitude);
        int cellY = this.latitudeToCellY(latitude);
        int localX = this.longitudeToLocalX(cellX, longitude);
        int localY = this.latitudeToLocalY(cellY, latitude);
        List<String> result = this.getCell(cellX, cellY).getIds(localX, localY);
        Collections.sort(result, this::compareSize);
        return result;
    }

    public Set<String> getContainingIds(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude) {
        HashSet<String> ids = new HashSet<String>();
        boolean[] firstCell = new boolean[]{true};
        this.forCellsIn(minLongitude, minLatitude, maxLongitude, maxLatitude, cell -> {
            if (firstCell[0]) {
                ids.addAll(cell.containingIds);
                firstCell[0] = false;
            } else {
                ids.retainAll(cell.containingIds);
                if (ids.isEmpty()) {
                    return false;
                }
            }
            return true;
        });
        return ids;
    }

    public Set<String> getIntersectingIds(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude) {
        HashSet<String> ids = new HashSet<String>();
        this.forCellsIn(minLongitude, minLatitude, maxLongitude, maxLatitude, cell -> {
            ids.addAll(cell.getAllIds());
            return true;
        });
        return ids;
    }

    private void forCellsIn(double minLongitude, double minLatitude, double maxLongitude, double maxLatitude, CellRunnable runnable) {
        if (!Double.isFinite(minLongitude)) {
            throw new IllegalArgumentException("minLongitude must be finite");
        }
        if (!Double.isFinite(minLatitude)) {
            throw new IllegalArgumentException("minLatitude must be finite");
        }
        if (!Double.isFinite(maxLongitude)) {
            throw new IllegalArgumentException("maxLongitude must be finite");
        }
        if (!Double.isFinite(maxLatitude)) {
            throw new IllegalArgumentException("maxLatitude must be finite");
        }
        if (minLatitude < -90.0 || minLatitude > 90.0) {
            throw new IllegalArgumentException("minLatitude is out of bounds");
        }
        if (maxLatitude < -90.0 || maxLatitude > 90.0) {
            throw new IllegalArgumentException("maxLatitude is out of bounds");
        }
        if (minLatitude > maxLatitude) {
            throw new IllegalArgumentException("maxLatitude is smaller than minLatitude");
        }
        minLongitude = CountryBoundaries.normalize(minLongitude, -180.0, 360.0);
        maxLongitude = CountryBoundaries.normalize(maxLongitude, -180.0, 360.0);
        int minX = this.longitudeToCellX(minLongitude);
        int maxY = this.latitudeToCellY(minLatitude);
        int maxX = this.longitudeToCellX(maxLongitude);
        int minY = this.latitudeToCellY(maxLatitude);
        int stepsX = minX > maxX ? this.rasterWidth - minX + maxX : maxX - minX;
        for (int xStep = 0; xStep <= stepsX; ++xStep) {
            int x = (minX + xStep) % this.rasterWidth;
            for (int y = minY; y <= maxY; ++y) {
                if (runnable.run(this.getCell(x, y))) continue;
                return;
            }
        }
    }

    private CountryBoundariesCell getCell(int x, int y) {
        return this.raster[y * this.rasterWidth + x];
    }

    private void validatePosition(double longitude, double latitude) {
        if (!Double.isFinite(longitude)) {
            throw new IllegalArgumentException("longitude must be finite");
        }
        if (!Double.isFinite(latitude)) {
            throw new IllegalArgumentException("latitude must be finite");
        }
        if (latitude < -90.0 || latitude > 90.0) {
            throw new IllegalArgumentException("latitude is out of bounds");
        }
    }

    private int longitudeToCellX(double longitude) {
        return (int)Math.min((double)(this.rasterWidth - 1), Math.floor((double)this.rasterWidth * (180.0 + longitude) / 360.0));
    }

    private int latitudeToCellY(double latitude) {
        int rasterHeight = this.raster.length / this.rasterWidth;
        return (int)Math.max(0.0, Math.ceil((double)rasterHeight * (90.0 - latitude) / 180.0) - 1.0);
    }

    private int longitudeToLocalX(int cellX, double longitude) {
        double cellLongitude = -180.0 + 360.0 * (double)cellX / (double)this.rasterWidth;
        return (int)((longitude - cellLongitude) * 360.0 * 65535.0 / (double)this.rasterWidth);
    }

    private int latitudeToLocalY(int cellY, double latitude) {
        int rasterHeight = this.raster.length / this.rasterWidth;
        double cellLatitude = 90.0 - 180.0 * (double)(cellY + 1) / (double)rasterHeight;
        return (int)((latitude - cellLatitude) * 180.0 * 65535.0 / (double)rasterHeight);
    }

    private static double normalize(double value, double startAt, double base) {
        if ((value %= base) < startAt) {
            value += base;
        } else if (value >= startAt + base) {
            value -= base;
        }
        return value;
    }

    private double getSize(String id) {
        Double size = this.geometrySizes.get(id);
        return size != null ? size : 0.0;
    }

    private int compareSize(String isoCode1, String isoCode2) {
        return (int)(this.getSize(isoCode1) - this.getSize(isoCode2));
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        CountryBoundaries that = (CountryBoundaries)o;
        return this.rasterWidth == that.rasterWidth && Arrays.equals(this.raster, that.raster) && this.geometrySizes.equals(that.geometrySizes);
    }

    public int hashCode() {
        return this.rasterWidth + 31 * (Arrays.hashCode(this.raster) + 31 * this.geometrySizes.hashCode());
    }

    private static interface CellRunnable {
        public boolean run(CountryBoundariesCell var1);
    }
}

