@file:OptIn(ExperimentalContracts::class)

package dev.kikugie.commons

import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

/**
 * Purely cosmetic function for creating cursed one-liners.
 * Disregards the receiver and returns the [next] value.
 * @sample dev.kikugie.commons_samples.ControlFlow.then
 */
@Suppress("NOTHING_TO_INLINE")
public inline infix fun <T> Any?.then(next: T): T = next

/**
 * Applies the [action] to the receiver if the [check] succeeds.
 * In either case returns the receiver itself.
 */
public inline fun <T> T.applyIf(check: Boolean, action: T.() -> Unit): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (check) apply(action) else this
}

/**
 * Applies the [action] to the receiver if the [check] succeeds.
 * In either case returns the receiver itself.
 */
public inline fun <T> T.applyIf(check: (T) -> Boolean, action: T.() -> Unit): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (check(this)) apply(action) else this
}

/**
 * Applies the [action] to the receiver if the [value] is not null.
 * In either case returns the receiver itself.
 */
public inline fun <T, A> T.applyIfNotNull(value: A?, action: T.(A) -> Unit): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (value != null) apply { action(value) } else this
}

/**
 * Transforms the receiver with the provided [action] if the [check] succeeds.
 * Otherwise, returns the unmodified receiver.
 */
public inline fun <T> T.runIf(check: Boolean, action: T.() -> T): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (check) action() else this
}


/**
 * Transforms the receiver with the provided [action] if the [check] succeeds.
 * Otherwise, returns the unmodified receiver.
 */
public inline fun <T> T.runIf(check: (T) -> Boolean, action: T.() -> T): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (check(this)) action() else this
}

/**
 * Transforms the receiver with the provided [action] if the [value] is not null.
 * Otherwise, returns the unmodified receiver.
 */
public inline fun <T, A> T.runIfNotNull(value: A?, action: T.(A) -> T): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (value != null) action(value) else this
}

public inline fun <reified T> Any.takeAs(): T =
    this as T

public inline fun <reified T> Any.takeAsOrNull(): T? =
    this as? T