@file:Suppress("NOTHING_TO_INLINE", "UNCHECKED_CAST")

package dev.kikugie.commons.collections

import dev.kikugie.commons.applyIf


// region ifNotEmpty-s
/**Applies the [action] to this array if it's not empty.*/
public inline fun <T> Array<out T>.ifNotEmpty(action: Array<out T>.() -> Unit): Array<out T> =
    applyIf(isNotEmpty(), action)

/**Applies the [action] to this collection if it's not empty.*/
public inline fun <T, C : Collection<T>> C.ifNotEmpty(action: C.() -> Unit): C =
    applyIf(isNotEmpty(), action)

/**Applies the [action] to this map if it's not empty.*/
public inline fun <K, V, M : Map<K, V>> M.ifNotEmpty(action: M.() -> Unit): M =
    applyIf(isNotEmpty(), action)
// endregion

// region getOrDefault-s
/**An alternative to the [Array.getOrElse] function for precomputed values.*/
public inline fun <T> Array<out T>.getOrDefault(index: Int, default: T): T =
    getOrElse(index) { default }

/**An alternative to the [List.getOrElse] function for precomputed values.*/
public inline fun <T> List<T>.getOrDefault(index: Int, default: T): T =
    getOrElse(index) { default }

/**An alternative to the [Map.getOrElse] function for precomputed values.*/
public inline fun <K, V> Map<K, V>.getOrDefault(key: K, default: V): V =
    getOrElse(key) { default }
// endregion

// region getOrThrow-s
/**An alternative to the [Array.get] function that throws an exception with the provided [message].*/
public inline fun <T> Array<out T>.getOrThrow(index: Int, message: Array<out T>.(Int) -> String): T =
    if (index in indices) get(index) else throw IndexOutOfBoundsException(message(index))

/**An alternative to the [List.get] function that throws an exception with the provided [message].*/
public inline fun <T> List<T>.getOrThrow(index: Int, message: List<T>.(Int) -> String): T =
    if (index in indices) get(index) else throw IndexOutOfBoundsException(message(index))

/**An alternative to the [Map.get] function that throws an exception with the provided [message].*/
public inline fun <K, V> Map<K, V>.getOrThrow(key: K, message: Map<K, V>.(K) -> String): V =
    if (key in this) get(key) as V else throw IllegalArgumentException(message(key))

// endregion

// region getOrResult-s
/**An alternative to the [Array.get] function that wraps it in a [Result].*/
public inline fun <T> Array<out T>.getResult(index: Int): Result<T> =
    runCatching { get(index) }

/**An alternative to the [List.get] function that wraps it in a [Result].*/
public inline fun <T> List<T>.getResult(index: Int): Result<T> =
    runCatching { get(index) }

/**An alternative to the [Map.get] function that wraps it in a [Result].*/
public inline fun <K, V> Map<K, V>.getResult(key: K): Result<V> =
    runCatching { requireNotNull(get(key)) }
// endregion

// region getOrResult-s
/**An alternative to the [Array.get] function that wraps it in a [Result] with the provided [message]*/
public inline fun <T> Array<out T>.getResult(index: Int, message: Array<out T>.(Int) -> String): Result<T> =
    runCatching { getOrThrow(index, message) }

/**An alternative to the [List.get] function that wraps it in a [Result] with the provided [message]*/
public inline fun <T> List<T>.getResult(index: Int, message: List<T>.(Int) -> String): Result<T> =
    runCatching { getOrThrow(index, message) }

/**An alternative to the [Map.get] function that wraps it in a [Result] with the provided [message]*/
public inline fun <K, V> Map<K, V>.getResult(key: K, message: Map<K, V>.(K) -> String): Result<V> =
    runCatching { getOrThrow(key, message) }
// endregion

// region findIsInstance-s
/**Finds the first [T] element of this array that satisfies the [check], returning `null` if nothing was found.*/
public inline fun <reified T : Any> Array<*>.findIsInstance(check: (T) -> Boolean = { true }): T? {
    for (it in this) if (it is T && check(it)) return it
    return null
}

/**Finds the first [T] element of this iterator that satisfies the [check], returning `null` if nothing was found.*/
public inline fun <reified T : Any> Iterator<*>.findIsInstance(check: (T) -> Boolean = { true }): T? {
    for (it in this) if (it is T && check(it)) return it
    return null
}

/**Finds the first [T] element of this iterable that satisfies the [check], returning `null` if nothing was found.*/
public inline fun <reified T : Any> Iterable<*>.findIsInstance(check: (T) -> Boolean = { true }): T? {
    for (it in this) if (it is T && check(it)) return it
    return null
}

/**Finds the first [T] element of this sequence that satisfies the [check], returning `null` if nothing was found.*/
public inline fun <reified T : Any> Sequence<*>.findIsInstance(check: (T) -> Boolean = { true }): T? {
    for (it in this) if (it is T && check(it)) return it
    return null
}
// endregion

// region firstIsInstance-s
/**Finds the first [T] element of this array that satisfies the [check], throwing an error if nothing was found.*/
public inline fun <reified T : Any> Array<*>.firstIsInstance(check: (T) -> Boolean = { true }): T {
    for (it in this) if (it is T && check(it)) return it
    throw NoSuchElementException("No element found")
}

/**Finds the first [T] element of this iterator that satisfies the [check], throwing an error if nothing was found.*/
public inline fun <reified T : Any> Iterator<*>.firstIsInstance(check: (T) -> Boolean = { true }): T {
    for (it in this) if (it is T && check(it)) return it
    throw NoSuchElementException("No element found")
}

/**Finds the first [T] element of this iterable that satisfies the [check], throwing an error if nothing was found.*/
public inline fun <reified T : Any> Iterable<*>.firstIsInstance(check: (T) -> Boolean = { true }): T {
    for (it in this) if (it is T && check(it)) return it
    throw NoSuchElementException("No element found")
}

/**Finds the first [T] element of this sequence that satisfies the [check], throwing an error if nothing was found.*/
public inline fun <reified T : Any> Sequence<*>.firstIsInstance(check: (T) -> Boolean = { true }): T {
    for (it in this) if (it is T && check(it)) return it
    throw NoSuchElementException("No element found")
}
// endregion
