package dev.kikugie.commons.ranges

import kotlin.math.max
import kotlin.math.min

public inline fun bind(pos: Int, range: IntRange, message: () -> String) {
    if (pos !in range) throw IndexOutOfBoundsException(message())
}

/**
 * Creates a range starting at [this], spanning the provided [length]. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.extend
 */
public infix fun Int.extend(length: Int): IntRange = this..(this + length)

/**
 * Creates a range starting at [this], spanning the provided [length]. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.extend
 */
public infix fun Long.extend(length: Long): LongRange = this..(this + length)

/**
 * Shifts [this] range [n] steps to the right. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.shiftRight
 */
public infix fun IntRange.shr(n: Int): IntRange = (first + n)..(last + n)

/**
 * Shifts [this] range [n] steps to the right. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.shiftRight
 */
public infix fun LongRange.shr(n: Long): LongRange = (first + n)..(last + n)

/**
 * Shifts [this] range [n] steps to the left. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.shiftLeft
 */
public infix fun IntRange.shl(n: Int): IntRange = (first - n)..(last - n)

/**
 * Shifts [this] range [n] steps to the left. The result is unchecked.
 * @sample dev.kikugie.commons_samples.Ranges.shiftLeft
 */
public infix fun LongRange.shl(n: Long): LongRange = (first - n)..(last - n)

/**
 * Creates an intersection between [this] and the [other] range,
 * returning [IntRange.EMPTY] if there's no overlap.
 * @sample dev.kikugie.commons_samples.Ranges.crossRange
 */
public infix fun IntRange.cross(other: IntRange): IntRange =
    if (this overlaps other) max(first, other.first)..min(last, other.last)
    else IntRange.EMPTY

/**
 * Creates an intersection between [this] and the [other] range,
 * returning [LongRange.EMPTY] if there's no overlap.
 * @sample dev.kikugie.commons_samples.Ranges.crossRange
 */
public infix fun LongRange.cross(other: LongRange): LongRange =
    if (this overlaps other) max(first, other.first)..min(last, other.last)
    else LongRange.EMPTY

/**
 * Creates a union of [this] and the [other] range **if they overlap**,
 * otherwise returns [IntRange.EMPTY].
 * @sample dev.kikugie.commons_samples.Ranges.mergeRange
 */
public infix fun IntRange.merge(other: IntRange): IntRange =
    if (this overlaps other) extend(other)
    else IntRange.EMPTY

/**
 * Creates a union of [this] and the [other] range **if they overlap**,
 * otherwise returns [LongRange.EMPTY].
 * @sample dev.kikugie.commons_samples.Ranges.mergeRange
 */
public infix fun LongRange.merge(other: LongRange): LongRange =
    if (this overlaps other) extend(other)
    else LongRange.EMPTY

/**
 * Creates a union of [this] and the [other] range,
 * **covering all values between them**.
 * @sample dev.kikugie.commons_samples.Ranges.extendRange
 */
public infix fun IntRange.extend(other: IntRange): IntRange =
    min(first, other.first)..max(last, other.last)

/**
 * Creates a union of [this] and the [other] range,
 * **covering all values between them**.
 * @sample dev.kikugie.commons_samples.Ranges.extendRange
 */
public infix fun LongRange.extend(other: LongRange): LongRange =
    min(first, other.first)..max(last, other.last)

/**
 * Checks if [this] and the [other] range have an overlapping segment.
 * @sample dev.kikugie.commons_samples.Ranges.overlaps
 */
public infix fun <T : Comparable<T>> ClosedRange<T>.overlaps(other: ClosedRange<T>): Boolean =
    other.start in this || other.endInclusive in this

/**
 * Checks if both endpoints of the [other] range are inside of [this] one.
 * @sample dev.kikugie.commons_samples.Ranges.contains
 */
public operator fun <T : Comparable<T>> ClosedRange<T>.contains(other: ClosedRange<T>): Boolean =
    other.start in this && other.endInclusive in this
