@file:Suppress("UNCHECKED_CAST", "NOTHING_TO_INLINE", "MoveLambdaOutsideParentheses")
@file:OptIn(ExperimentalContracts::class)

package dev.kikugie.commons.result

import dev.kikugie.commons.ExperimentalCommonsAPI
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

/**
 * Represents one of the possible values.
 */
@ExperimentalCommonsAPI
public sealed interface Either<L, R> {
    public companion object {
        public inline fun <L, R> left(value: L): Either<L, R> = Left(value)
        public inline fun <L, R> right(value: R): Either<L, R> = Right(value)
    }

    @ConsistentCopyVisibility
    public data class Left<L, R> @PublishedApi internal constructor(val value: L) : Either<L, R> {
        override val isLeft: Boolean get() = true
    }

    @ConsistentCopyVisibility
    public data class Right<L, R> @PublishedApi internal constructor(val value: R) : Either<L, R> {
        override val isRight: Boolean get() = true
    }

    public val isRight: Boolean get() = false
    public val isLeft: Boolean get() = false
}

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.swap(): Either<R, L> =
    fold({ Either.right(it) }, { Either.left(it) })

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.ifLeft(action: (L) -> Unit): Either<L, R> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    fold(action, {})
    return this
}

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.leftOrNull(): L? =
    fold({ it }, { null })

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.leftOrThrow(): L =
    leftOrElse { throw NoSuchElementException("No value present") }

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.leftOrDefault(default: L): L =
    leftOrElse { default }

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.leftOrElse(action: (R) -> L): L {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return fold({ it }, action)
}

@ExperimentalCommonsAPI
public inline fun <L, R, O> Either<L, R>.mapLeft(action: (L) -> O): Either<O, R> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return fold({ Either.left(action(it)) }, { it as Either<O, R> })
}

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.ifRight(action: (R) -> Unit): Either<L, R> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    fold({}, action)
    return this
}

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.rightOrNull(): R? =
    fold({ null }, { it })

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.rightOrThrow(): R =
    rightOrElse { throw NoSuchElementException("No value present") }

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.rightOrDefault(default: R): R =
    rightOrElse { default }

@ExperimentalCommonsAPI
public inline fun <L, R> Either<L, R>.rightOrElse(action: (L) -> R): R {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return fold(action, { it })
}

@ExperimentalCommonsAPI
public inline fun <L, R, O> Either<L, R>.mapRight(action: (R) -> O): Either<L, O> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return fold({ it as Either<L, O> }, { Either.right(action(it)) })
}

@ExperimentalCommonsAPI
public inline fun <L, R, O> Either<L, R>.fold(ifLeft: (L) -> O, ifRight: (R) -> O): O {
    contract {
        callsInPlace(ifLeft, InvocationKind.AT_MOST_ONCE)
        callsInPlace(ifRight, InvocationKind.AT_MOST_ONCE)
    }
    return when (this) {
        is Either.Left<*, *> -> ifLeft(value as L)
        is Either.Right<*, *> -> ifRight(value as R)
    }
}