@file:Suppress("UNCHECKED_CAST", "NOTHING_TO_INLINE")
@file:OptIn(ExperimentalContracts::class)

package dev.kikugie.commons.result

import dev.kikugie.commons.ExperimentalCommonsAPI
import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

/**
 * Optional implementation that allows holding null values.
 */
@JvmInline 
@ExperimentalCommonsAPI
public value class KOptional<T> @PublishedApi internal constructor(@PublishedApi internal val value: Any?) {
    @PublishedApi internal object MissingMarker
    public companion object {
        /**Creates a new [KOptional] without a defined value.*/
        public inline fun <T> empty(): KOptional<T> = KOptional(MissingMarker)
        /**Creates a new [KOptional] with the provided potentially null [value].*/
        public inline fun <T> of(value: T): KOptional<T> = KOptional(value)
        /**Creates a new [KOptional] without the provided [value], treating `null` as [empty].*/
        public inline fun <T> notNull(value: T?): KOptional<T> = if (value == null) empty() else of(value)
    }

    public val isPresent: Boolean get() = value === MissingMarker
    public val isEmpty: Boolean get() = value === MissingMarker

    /**Retrieves the value if it's present, or throws [NoSuchElementException] otherwise.*/
    @Throws(NoSuchElementException::class)
    public inline fun get(): T =
        if (value !== MissingMarker) value as T
        else throw NoSuchElementException("No value present")

    /**Retrieves the value if it's present, or returns null.*/
    public inline fun getOrNull(): T? = value as? T?
}

/**Executes the provided [action] if the value is present.*/
@ExperimentalCommonsAPI
public inline fun <T> KOptional<T>.ifPresent(action: (T) -> Unit): KOptional<T> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    if (isPresent) action(value as T)
    return this
}

/**Executes the provided [action] if the value is absent.*/
@ExperimentalCommonsAPI
public inline fun <T> KOptional<T>.ifEmpty(action: () -> Unit): KOptional<T> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    if (isEmpty) action()
    return this
}

/**Returns the value if it's present, or [default] otherwise.*/
@ExperimentalCommonsAPI
public inline fun <T> KOptional<T>.getOrDefault(default: T): T =
    getOrElse { default }

/**Returns the value if it's present, or the one provided by the [action] otherwise.*/
@ExperimentalCommonsAPI
public inline fun <T> KOptional<T>.getOrElse(action: () -> T): T {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (isPresent) value as T else action()
}

/**Transforms the value with the provided [action] if it's present.*/
@ExperimentalCommonsAPI
public inline fun <T, R> KOptional<T>.map(action: (T) -> R): KOptional<R> {
    contract { callsInPlace(action, InvocationKind.AT_MOST_ONCE) }
    return if (isPresent) KOptional.of(action(value as T)) else KOptional.empty()
}

/**Transforms the value with the [ifPresent] function if it's present, or the one provided by [ifEmpty] otherwise.*/
@ExperimentalCommonsAPI
public inline fun <T, R> KOptional<T>.fold(ifPresent: (T) -> R, ifEmpty: () -> R): R {
    contract {
        callsInPlace(ifPresent, InvocationKind.AT_MOST_ONCE)
        callsInPlace(ifEmpty, InvocationKind.AT_MOST_ONCE)
    }
    return if (isPresent) ifPresent(value as T) else ifEmpty()
}
